% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fdt.R
\name{fdt}
\alias{fdt}
\title{Format a datetime with a formatting string}
\usage{
fdt(input, format = NULL, use_tz = NULL, locale = NULL)
}
\arguments{
\item{input}{A vector of date, time, or datetime values. Several
representations are acceptable here including strings, \code{Date} objects, or
\code{POSIXct} objects. Refer to the \emph{Valid Input Values} section for more
information.}

\item{format}{The formatting string to apply to all \code{input} values. If not
provided, the inputs will be formatted to ISO 8601 datetime strings. The
\emph{Date/Time Format Syntax} section has detailed information on how to create
a formatting string.}

\item{use_tz}{A tzid (e.g., \code{"America/New_York"}) time-zone designation for
precise formatting of related outputs. This overrides any time zone
information available in \code{character}-based input values and is applied to
all vector components.}

\item{locale}{The output locale to use for formatting the input value
according to the specified locale's rules. Example locale names include
\code{"en"} for English (United States) and \code{"es-EC"} for Spanish (Ecuador). If
a locale isn't provided the \code{"en"} locale will be used. The
\link{fdt_locales_vec} vector contains the valid locales and \link{fdt_locales_lst}
list provides an easy way to obtain a valid locale.}
}
\value{
A character vector of formatted dates, times, or datetimes.
}
\description{
With \code{fdt()}, we can format datetime values with the greatest of ease, and,
with great power. There is a lot of leniency in what types of input
date/time/datetime values can be passed in. The formatting string allows for
a huge array of possibilities when formatting. Not only that, we can set a
\code{locale} value and get the formatted values localized in the language/region
of choice. There's plenty of ways to represent time zone information, and
this goes along with the option to enrich the input values with a precise
time zone identifier (like \code{"America/Los_Angeles"}). The choices are ample
here, with the goal being a comprehensiveness and ease-of-use in date/time
formatting.
}
\section{Valid Input Values}{


The \code{input} argument of the \code{fdt()} function allows for some flexibility on
what can be passed in. This section describes the kinds of inputs that are
understandable by \code{fdt()}. A vector of strings is allowed, as are vectors of
\code{Date} or \code{POSIXct} values.

If using strings, a good option is to use those that adhere to the ISO
8601:2004 standard. For a datetime this can be of the form
\verb{YYYY-MM-DDThh:mm:ss.s<TZD>}. With this, \code{YYYY-MM-DD} corresponds to the
date, the literal \code{"T"} is optional, \code{hh:mm:ss} is the time (where seconds,
\code{ss}, is optional as are \code{.s} for fractional seconds), and \verb{<TZD>} refers to
an optional time-zone designation (more on time zones in the next paragraph).
You can provide just the date part, and this assumes midnight as an implicit
time. It's also possible to provide just the time part, and this internally
assembles a datetime that uses the current date. When formatting standalone
dates or times, you'll probably just format the explicit parts but \code{fdt()}
won't error if you format the complementary parts.

The time zone designation on string-based datetimes is completely optional.
If not provided then \code{"UTC"} is assumed. If you do want to supply time zone
information, it can be given as an offset value with the following
constructions:
\itemize{
\item \verb{<time>Z}
\item \verb{<time>(+/-)hh:mm}
\item \verb{<time>(+/-)hhmm}
\item \verb{<time>(+/-)hh}
}

The first, \verb{<time>Z}, is zone designator for the zero UTC offset; it's
equivalent to \code{"+00:00"}. The next two are formats for providing the time
offsets from UTC with hours and minutes fields. Examples are \code{"-05:00"} (New
York, standard time), \code{"+0200"} (Cairo), and \code{"+05:30"} (Mumbai). Note that
the colon is optional but leading zeros to maintain two-digit widths are
essential. The final format, \verb{<time>(+/-)hh}, omits the minutes field and as
so many offsets have \code{"00"} minutes values, this can be convenient.

We can also supply an Olson/IANA-style time zone identifier (tzid) in
parentheses within the string, or, as a value supplied to \code{use_tz} (should a
tzid apply to all date/time/datetime values in the \code{input} vector). By
extension, this would use the form: \verb{YYYY-MM-DDThh:mm:ss.s<TZD>(<tzid>)}.
Both a \verb{<TZD>} (UTC offset value) and a \verb{<tzid>} shouldn't really be used
together but if that occurs the \verb{<tzid>} overrides the UTC offset. Here are
some examples:
\itemize{
\item \code{"2018-07-04T22:05 (America/Vancouver)"} (preferable)
\item \code{"2018-07-04T22:05-0800(America/Vancouver)"} (redundant, but still okay)
}

A tzid contains much more information about the time zone than a UTC offset
value since it is tied to some geographical location and the timing of
Standard Time (STD) and Daylight Saving Time (DST) is known. In essence we
can derive UTC offset values from a tzid and also a host of other identifiers
(time zone names, their abbreviations, etc.). Here are some examples of valid
tzid values that can be used:
\itemize{
\item \code{"America/Jamaica"} (the official time in Jamaica, or, \code{"Jamaica Time"})
\item \code{"Australia/Perth"} (\code{"+08:00"} year round in Western Australia)
\item \code{"Europe/Dublin"} (IST/GMT time: \code{"+01:00"}/\code{"+00:00"})
}

The tz database (a compilation of information about the world's time zones)
consists of canonical zone names (those that are primary and preferred) and
alternative names (less preferred in modern usage, and was either discarded
or more commonly replaced by a canonical zone name). The \code{fdt()} function can
handle both types and what occurs is that non-canonical tzid values are
internally mapped onto canonical zone names. Here's a few examples:
\itemize{
\item \code{"Africa/Luanda"} (in Angola) maps to \code{"Africa/Lagos"}
\item \code{"America/Indianapolis"} maps to \code{"America/Indiana/Indianapolis"}
\item \code{"Asia/Calcutta"} maps to \code{"Asia/Kolkata"}
\item \code{"Pacific/Midway"} maps to \code{"Pacific/Pago_Pago"}
\item \code{"Egypt"} maps to \code{"Africa/Cairo"}
}

For the most part, the Olson-format tzid follows the form \code{"{region}/{city}"}
where the region is usually a continent, the city is considered an 'exemplar
city', and the exemplar city itself belongs in a country.
}

\section{Date/Time Format Syntax}{


A formatting pattern as used in \strong{bigD} consists of a string of characters,
where certain strings are replaced with date and time data that are derived
from the parsed input.

The characters used in patterns are tabulated below to show which specific
strings produce which outputs (e.g., \code{"y"} for the year). A common pattern is
characters that are used consecutively to produce variations on a date, time,
or timezone output. Say that the year in the input is 2015. If using \code{"yy"}
you'll get \code{"15"} but with \code{"yyyy"} the output becomes \code{"1999"}. There's a
whole lot of this, so the following subsections try to illustrate as best as
possible what each string will produce. All of the examples will use this
string-based datetime input unless otherwise indicated:

\code{"2018-07-04T22:05:09.2358(America/Vancouver)"}
\subsection{The Era Designator (big G)}{\tabular{ll}{
   Formatting String \tab Output \cr
   \code{"G"}, \code{"GG"}, or \code{"GGG"} \tab \code{"AD"} \cr
   \code{"GGGG"} \tab \code{"Anno Domini"} \cr
   \code{"GGGGG"} \tab \code{"A"} \cr
}

}

\subsection{Year (little y)}{\tabular{ll}{
   Formatting String \tab Output \cr
   \code{"y"} \tab \code{"2018"} \cr
   \code{"yy"} \tab \code{"18"} \cr
   \code{"yyy"} \tab \code{"2018"} \cr
   \code{"yyyy"} \tab \code{"2018"} \cr
   \code{"yyyyy"} \tab \code{"02018"} \cr
   \code{"yyyyyy"} \tab \code{"002018"} \cr
   \code{"yyyyyyy"} \tab \code{"0002018"} \cr
   \code{"yyyyyyyy"} \tab \code{"00002018"} \cr
   \code{"yyyyyyyyy"} \tab \code{"000002018"} \cr
}

}

\subsection{Year in the Week in Year Calendar (big Y)}{

This is the year in 'Week of Year' based calendars in which the year
transition occurs on a week boundary. This may differ from calendar year 'y'
near a year transition. This numeric year designation is used in conjunction
with pattern character 'w' in the ISO year-week calendar as defined by ISO
8601.\tabular{ll}{
   Formatting String \tab Output \cr
   \code{"Y"} \tab \code{"2018"} \cr
   \code{"YY"} \tab \code{"18"} \cr
   \code{"YYY"} \tab \code{"2018"} \cr
   \code{"YYYY"} \tab \code{"2018"} \cr
   \code{"YYYYY"} \tab \code{"02018"} \cr
   \code{"YYYYYY"} \tab \code{"002018"} \cr
   \code{"YYYYYYY"} \tab \code{"0002018"} \cr
   \code{"YYYYYYYY"} \tab \code{"00002018"} \cr
   \code{"YYYYYYYYY"} \tab \code{"000002018"} \cr
}

}

\subsection{Quarter of the Year: formatting ver. (big Q)}{\tabular{ll}{
   Formatting String \tab Output \cr
   \code{"Q"} \tab \code{"3"} \cr
   \code{"QQ"} \tab \code{"03"} \cr
   \code{"QQQ"} \tab \code{"Q3"} \cr
   \code{"QQQQ"} \tab \code{"3rd quarter"} \cr
   \code{"QQQQQ"} \tab \code{"3"} \cr
}

}

\subsection{Quarter of the Year: standalone ver. (little q)}{\tabular{ll}{
   Formatting String \tab Output \cr
   \code{"q"} \tab \code{"3"} \cr
   \code{"qq"} \tab \code{"03"} \cr
   \code{"qqq"} \tab \code{"Q3"} \cr
   \code{"qqqq"} \tab \code{"3rd quarter"} \cr
   \code{"qqqqq"} \tab \code{"3"} \cr
}

}

\subsection{Month: formatting ver. (big M)}{\tabular{ll}{
   Formatting String \tab Output \cr
   \code{"M"} \tab \code{"7"} \cr
   \code{"MM"} \tab \code{"07"} \cr
   \code{"MMM"} \tab \code{"Jul"} \cr
   \code{"MMMM"} \tab \code{"July"} \cr
   \code{"MMMMM"} \tab \code{"J"} \cr
}

}

\subsection{Month: standalone ver. (big L)}{\tabular{ll}{
   Formatting String \tab Output \cr
   \code{"L"} \tab \code{"7"} \cr
   \code{"LL"} \tab \code{"07"} \cr
   \code{"LLL"} \tab \code{"Jul"} \cr
   \code{"LLLL"} \tab \code{"July"} \cr
   \code{"LLLLL"} \tab \code{"J"} \cr
}

}

\subsection{Week of Year (little w)}{\tabular{ll}{
   Formatting String \tab Output \cr
   \code{"w"} \tab \code{"27"} (minimal digits) \cr
   \code{"ww"} \tab \code{"27"} (two digits, zero padded) \cr
}

}

\subsection{Week of Month (big W)}{\tabular{ll}{
   Formatting String \tab Output \cr
   \code{"W"} \tab \code{"1"} \cr
}

}

\subsection{Day of Month (little d)}{\tabular{ll}{
   Formatting String \tab Output \cr
   \code{"d"} \tab \code{"4"} (minimal digits) \cr
   \code{"dd"} \tab \code{"04"} (two digits, zero padded) \cr
}

}

\subsection{Day of Year (big D)}{\tabular{ll}{
   Formatting String \tab Output \cr
   \code{"D"} \tab \code{"185"} \cr
   \code{"DD"} \tab \code{"185"} (zero padded to min-width of 2) \cr
   \code{"DDD"} \tab \code{"185"} (zero padded to min-width of 3) \cr
}

}

\subsection{Day of Week in Month (big F)}{\tabular{ll}{
   Formatting String \tab Output \cr
   \code{"F"} \tab \code{"1"} \cr
}

}

\subsection{Modified Julian Day (little g)}{\tabular{ll}{
   Formatting String \tab Output \cr
   \code{"g"} \tab \code{"58303"} \cr
   \code{"gg"} \tab \code{"58303"} \cr
   \code{"ggg"} \tab \code{"58303"} \cr
   \code{"gggg"} \tab \code{"58303"} \cr
   \code{"ggggg"} \tab \code{"58303"} \cr
   \code{"gggggg"} \tab \code{"058303"} \cr
   \code{"ggggggg"} \tab \code{"0058303"} \cr
   \code{"gggggggg"} \tab \code{"00058303"} \cr
   \code{"ggggggggg"} \tab \code{"000058303"} \cr
}

}

\subsection{Day of Week Name (big E)}{\tabular{ll}{
   Formatting String \tab Output \cr
   \code{"E"} \tab \code{"Wed"} \cr
   \code{"EE"} \tab \code{"Wed"} \cr
   \code{"EEE"} \tab \code{"Wed"} \cr
   \code{"EEEE"} \tab \code{"Wednesday"} \cr
   \code{"EEEEE"} \tab \code{"W"} \cr
   \code{"EEEEEE"} \tab \code{"We"} \cr
}

}

\subsection{AM/PM Period of Day (little a)}{\tabular{lll}{
   Formatting String \tab Output \tab Note \cr
   \code{"a"}, \code{"aa"}, or \code{"aaa"} \tab \code{"PM"} \tab Abbreviated \cr
   \code{"aaaa"} \tab \code{"PM"} \tab Wide \cr
   \code{"aaaaa"} \tab \code{"p"} \tab Narrow \cr
}

}

\subsection{AM/PM Period of Day Plus Noon and Midnight (little b)}{

(a) \code{input_midnight}: \code{"2020-05-05T00:00:00"}

(b) \code{input_noon}: \code{"2020-05-05T12:00:00"}\tabular{lll}{
   Formatting String \tab Output \tab Note \cr
   \code{"b"}, \code{"bb"}, or \code{"bbb"} \tab (a) \code{"midnight"} \tab Abbreviated \cr
   \verb{"" "" ""} \tab (b) \code{"noon"} \tab \verb{"" "" ""} \cr
   \code{"bbbb"} \tab (a) \code{"midnight"} \tab Wide \cr
   \verb{"" "" ""} \tab (b) \code{"noon"} \tab \verb{"" "" ""} \cr
   \code{"bbbbb"} \tab (a) \code{"mi"} \tab Narrow \cr
   \verb{"" "" ""} \tab (b) \code{"n"} \tab \verb{"" "" ""} \cr
}

}

\subsection{Flexible Day Periods (big B)}{

(a) \code{input_morning}: \code{"2020-05-05T00:08:30"}

(b) \code{input_afternoon}: \code{"2020-05-05T14:00:00"}\tabular{lll}{
   Formatting String \tab Output \tab Note \cr
   \code{"B"}, \code{"BB"}, or \code{"BBB"} \tab (a) \code{"in the morning"} \tab Abbreviated \cr
   \verb{"" "" ""} \tab (b) \code{"in the afternoon"} \tab \verb{"" "" ""} \cr
   \code{"BBBB"} \tab (a) \code{"in the morning"} \tab Wide \cr
   \verb{"" "" ""} \tab (b) \code{"in the afternoon"} \tab \verb{"" "" ""} \cr
   \code{"BBBBB"} \tab (a) \code{"in the morning"} \tab Narrow \cr
   \verb{"" "" ""} \tab (b) \code{"in the afternoon"} \tab \verb{"" "" ""} \cr
}

}

\subsection{Hour 1-12 (little h)}{

Using: \code{"2015-08-01T08:35:09"}\tabular{lll}{
   Formatting String \tab Output \tab Note \cr
   \code{"h"} \tab \code{"8"} \tab Numeric, minimum digits \cr
   \code{"hh"} \tab \code{"08"} \tab Numeric, 2 digits (zero padded) \cr
}

}

\subsection{Hour 0-23 (big H)}{

Using: \code{"2015-08-01T08:35:09"}\tabular{lll}{
   Formatting String \tab Output \tab Note \cr
   \code{"H"} \tab \code{"8"} \tab Numeric, minimum digits \cr
   \code{"HH"} \tab \code{"08"} \tab Numeric, 2 digits (zero padded) \cr
}

}

\subsection{Hour 0-11 (big K)}{

Using: \code{"2015-08-01T08:35:09"}\tabular{lll}{
   Formatting String \tab Output \tab Note \cr
   \code{"K"} \tab \code{"7"} \tab Numeric, minimum digits \cr
   \code{"KK"} \tab \code{"07"} \tab Numeric, 2 digits (zero padded) \cr
}

}

\subsection{Hour 1-24 (little k)}{

Using: \code{"2015-08-01T08:35:09"}\tabular{lll}{
   Formatting String \tab Output \tab Note \cr
   \code{"k"} \tab \code{"9"} \tab Numeric, minimum digits \cr
   \code{"kk"} \tab \code{"09"} \tab Numeric, 2 digits (zero padded) \cr
}

}

\subsection{Minute (little m)}{\tabular{lll}{
   Formatting String \tab Output \tab Note \cr
   \code{"m"} \tab \code{"5"} \tab Numeric, minimum digits \cr
   \code{"mm"} \tab \code{"06"} \tab Numeric, 2 digits (zero padded) \cr
}

}

\subsection{Second (little s)}{\tabular{lll}{
   Formatting String \tab Output \tab Note \cr
   \code{"s"} \tab \code{"9"} \tab Numeric, minimum digits \cr
   \code{"ss"} \tab \code{"09"} \tab Numeric, 2 digits (zero padded) \cr
}

}

\subsection{Fractional Second (big S)}{\tabular{ll}{
   Formatting String \tab Output \cr
   \code{"S"} \tab \code{"2"} \cr
   \code{"SS"} \tab \code{"23"} \cr
   \code{"SSS"} \tab \code{"235"} \cr
   \code{"SSSS"} \tab \code{"2350"} \cr
   \code{"SSSSS"} \tab \code{"23500"} \cr
   \code{"SSSSSS"} \tab \code{"235000"} \cr
   \code{"SSSSSSS"} \tab \code{"2350000"} \cr
   \code{"SSSSSSSS"} \tab \code{"23500000"} \cr
   \code{"SSSSSSSSS"} \tab \code{"235000000"} \cr
}

}

\subsection{Milliseconds Elapsed in Day (big A)}{

Using: \code{"2011-07-27T00:07:19.7223"}\tabular{ll}{
   Formatting String \tab Output \cr
   \code{"A"} \tab \code{"439722"} \cr
   \code{"AA"} \tab \code{"439722"} \cr
   \code{"AAA"} \tab \code{"439722"} \cr
   \code{"AAAA"} \tab \code{"439722"} \cr
   \code{"AAAAA"} \tab \code{"439722"} \cr
   \code{"AAAAAA"} \tab \code{"439722"} \cr
   \code{"AAAAAAA"} \tab \code{"0439722"} \cr
   \code{"AAAAAAAA"} \tab \code{"00439722"} \cr
   \code{"AAAAAAAAA"} \tab \code{"000439722"} \cr
}

}

\subsection{TZ // Short and Long Specific non-Location Format (little z)}{\tabular{lll}{
   Formatting String \tab Output \tab Note \cr
   \code{"z"}, \code{"zz"}, or \code{"zzz"} \tab \code{"PDT"} \tab Short Specific \cr
   \code{"zzzz"} \tab \code{"Pacific Daylight Time"} \tab Long Specific \cr
}

}

\subsection{TZ // Short and Long Specific non-Location Formats (big Z)}{\tabular{lll}{
   Formatting String \tab Output \tab Note \cr
   \code{"Z"}, \code{"ZZ"}, or \code{"ZZZ"} \tab \code{"-0700"} \tab ISO 8601 basic format \cr
   \code{"ZZZZ"} \tab \code{"GMT-7:00"} \tab Long localized GMT format \cr
   \code{"ZZZZZ"} \tab \code{"-07:00"} \tab ISO 8601 extended format \cr
}

}

\subsection{TZ // Short and Long Localized GMT Formats (big O)}{\tabular{lll}{
   Formatting String \tab Output \tab Note \cr
   \code{"O"} \tab \code{"GMT-7"} \tab Short localized GMT format \cr
   \code{"OOOO"} \tab \code{"GMT-07:00"} \tab Long localized GMT format \cr
}

}

\subsection{TZ // Short and Long Localized GMT Formats (little v)}{\tabular{lll}{
   Formatting String \tab Output \tab Note \cr
   \code{"v"} \tab \code{"PT"} \tab Short generic non-location format \cr
   \code{"vvvv"} \tab \code{"Pacific Time"} \tab Long generic non-location format \cr
}

}

\subsection{TZ // Short Time Zone IDs and Exemplar City Formats (big V)}{\tabular{lll}{
   Formatting String \tab Output \tab Note \cr
   \code{"V"} \tab \code{"cavan"} \tab Short time zone ID \cr
   \code{"VV"} \tab \code{"America/Vancouver"} \tab Long time zone ID \cr
   \code{"VVV"} \tab \code{"Vancouver"} \tab The tz exemplar city \cr
   \code{"VVVV"} \tab \code{"Vancouver Time"} \tab Generic location format \cr
}

}

\subsection{TZ // ISO 8601 Formats with Z for +0000 (big X)}{\tabular{lll}{
   Formatting String \tab Output \tab Note \cr
   \code{"X"} \tab \code{"-07"} \tab ISO 8601 basic format (h; optional m) \cr
   \code{"XX"} \tab \code{"-0700"} \tab ISO 8601 basic format (h & m) \cr
   \code{"XXX"} \tab \code{"-07:00"} \tab ISO 8601 extended format (h & m) \cr
   \code{"XXXX"} \tab \code{"-0700"} \tab ISO 8601 basic format (h & m, optional s) \cr
   \code{"XXXXX"} \tab \code{"-07:00"} \tab ISO 8601 extended format (h & m, optional s) \cr
}

}

\subsection{TZ // ISO 8601 Formats (no use of Z for +0000) (little x)}{\tabular{lll}{
   Formatting String \tab Output \tab Note \cr
   \code{"x"} \tab \code{"-07"} \tab ISO 8601 basic format (h; optional m) \cr
   \code{"xx"} \tab \code{"-0700"} \tab ISO 8601 basic format (h & m) \cr
   \code{"xxx"} \tab \code{"-07:00"} \tab ISO 8601 extended format (h & m) \cr
   \code{"xxxx"} \tab \code{"-0700"} \tab ISO 8601 basic format (h & m, optional s) \cr
   \code{"xxxxx"} \tab \code{"-07:00"} \tab ISO 8601 extended format (h & m, optional s) \cr
}

}
}

\section{Examples}{

\subsection{Basics with \code{input} datetimes, formatting strings, and localization}{

With an input datetime of \code{"2018-07-04 22:05"} supplied as a string, we can
format to get just a date with the full year first, the month abbreviation
second, and the day of the month last (separated by hyphens):

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2018-07-04 22:05",
  format = "y-MMM-dd"
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "2018-Jul-04"
}\if{html}{\out{</div>}}

There are sometimes many options for each time part. Instead of using
\code{"y-MMM-dd"}, let's try a variation on that with \code{"yy-MMMM-d"}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2018-07-04 22:05",
  format = "yy-MMMM-d"
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "18-July-4"
}\if{html}{\out{</div>}}

The output is localizable and so elements will be translated when supplying
the appropriate locale code. Let's use \code{locale = es} to get the month written
in Spanish:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2018-07-04 22:05",
  format = "yy-MMMM-d",
  locale = "es"
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "18-julio-4"
}\if{html}{\out{</div>}}

\code{POSIXct} or \code{POSIXlt} datetimes can serve as an \code{input} to \code{fdt()}. Let's
create a single datetime value where the timezone is set as \code{"Asia/Tokyo"}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = lubridate::ymd_hms("2020-03-15 19:09:12", tz = "Asia/Tokyo"),
  format = "EEEE, MMMM d, y 'at' h:mm:ss B (VVVV)"
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "Sunday, March 15, 2020 at 7:09:12 in the evening (Tokyo Time)"
}\if{html}{\out{</div>}}

If you're going minimal, it's possible to supply an input datetime string
without a \code{format} directive. What this gives us is formatted datetime
output that conforms to ISO 8601. Note that the implied time zone is UTC.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(input = "2018-07-04 22:05")
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "2018-07-04T22:05:00Z"
}\if{html}{\out{</div>}}
}

\subsection{Using locales stored in the \link{fdt_locales_lst} list}{

The \link{fdt_locales_lst} object is provided in \strong{bigD} to make it easier to
choose one of supported locales. You can avoid typing errors and every
element of the list is meant to work. For example, we can use the \code{"it"}
locale by accessing it from \link{fdt_locales_lst} (autocomplete makes this
relatively simple).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2018-07-04 22:05",
  format = "yy-MMMM-d",
  locale = fdt_locales_lst$it
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "18-luglio-4"
}\if{html}{\out{</div>}}
}

\subsection{Omission of date or time in \code{input}}{

You don't have to supply a full datetime to \code{input}. Just supplying the date
portion implies midnight (and is just fine if you're only going to present
the date anyway).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(input = "2018-07-04")
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "2018-07-04T00:00:00Z"
}\if{html}{\out{</div>}}

If you omit the date and just supply a time, \code{fdt()} will correctly parse
this. The current date on the user system will be used because we need to
create some sort of datetime value internally. Again, this is alright if
you just intend to present a formatted time value.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(input = "22:05")
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "2022-08-16T22:05:00Z"
}\if{html}{\out{</div>}}

To see all of the supported locales, we can look at the vector supplied by
the \code{\link[=fdt_locales_vec]{fdt_locales_vec()}} function.
}

\subsection{Using standardized forms with the \verb{standard_*()} helper functions}{

With an input datetime of \code{"2018-07-04 22:05(America/Vancouver)"}, we can
format the date and time in a standardized way with \code{standard_date_time()}
providing the correct formatting string. This function is invoked in the
\code{format} argument of \code{fdt()}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2018-07-04 22:05(America/Vancouver)",
  format = standard_date_time(type = "full")
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "Wednesday, July 4, 2018 at 10:05:00 PM Pacific Daylight Time"
}\if{html}{\out{</div>}}

The locale can be changed and we don't have to worry about the particulars
of the formatting string (they are standardized across locales).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2018-07-04 22:05(America/Vancouver)",
  format = standard_date_time(type = "full"),
  locale = fdt_locales_lst$nl
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "woensdag 4 juli 2018 om 22:05:00 Pacific-zomertijd"
}\if{html}{\out{</div>}}

We can use different \code{type} values to control the output datetime string. The
default is \code{"short"}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2018-07-04 22:05(America/Vancouver)",
  format = standard_date_time()
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "7/4/18, 10:05 PM"
}\if{html}{\out{</div>}}

After that, it's \code{"medium"}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2018-07-04 22:05(America/Vancouver)",
  format = standard_date_time(type = "medium")
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "Jul 4, 2018, 10:05:00 PM"
}\if{html}{\out{</div>}}

The \code{"short"} and \code{"medium"} types don't display time zone information in the
output. Beginning with \code{"long"}, the tz is shown.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2018-07-04 22:05(America/Vancouver)",
  format = standard_date_time(type = "long")
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "July 4, 2018 at 10:05:00 PM PDT"
}\if{html}{\out{</div>}}

If you don't include time zone information in the input, the \code{"UTC"} time
zone will be assumed:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2018-07-04 22:05",
  format = standard_date_time(type = "full")
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "Wednesday, July 4, 2018 at 10:05:00 PM GMT+00:00"
}\if{html}{\out{</div>}}
}

\subsection{Using flexible date and time (12- and 24-hour) formatting}{

The \strong{bigD} package supplies a set of lists to allow for flexible date and
time formatting (\link{flex_d_lst}, \link{flex_t24_lst}, and \link{flex_t12_lst}). These
are useful when you need a particular format that works across all locales.
Here's an example that uses the \code{"yMMMEd"} flexible date type by accessing it
from the \code{flex_d_lst} object, yielding a formatted date.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2021-01-09 16:32(America/Toronto)",
  format = flex_d_lst$yMMMEd,
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "Sat, Jan 9, 2021"
}\if{html}{\out{</div>}}

If we wanted this in a different locale, the locale-specific \code{format} pattern
behind the flexible date identifier would ensure consistency while moving to
that locale.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2021-01-09 16:32(America/Toronto)",
  format = flex_d_lst$yMMMEd,
  locale = "fr_CA"
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "sam. 9 janv. 2021"
}\if{html}{\out{</div>}}

Formatting as a 12-hour time with the \link{flex_t12_lst} list and using the
\code{"hms"} flexible type:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2021-01-09 16:32(America/Toronto)",
  format = flex_t12_lst$hms
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "4:32:00 PM"
}\if{html}{\out{</div>}}

The 24-hour variant, \link{flex_t24_lst}, has a similar \code{"Hms"} flexible type that
will give us a 24-hour version of the same clock time:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2021-01-09 16:32(America/Toronto)",
  format = flex_t24_lst$Hms
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "16:32:00"
}\if{html}{\out{</div>}}

A flexible date and time can be used together by enveloping the two in a
list (\strong{bigD} will handle putting the date and time together in a sensible
manner).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2021-01-09 16:32(America/Toronto)",
  format = list(flex_d_lst$yMMMEd, flex_t24_lst$Hmv)
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> "Sat, Jan 9, 2021, 16:32 ET"
}\if{html}{\out{</div>}}
}
}

