################################################################################
#
#' bootPROBIT
#'
#' PROBIT statistics function for bootstrap estimation
#'
#' @param x A data frame with \code{primary sampling unit (PSU)} in column named
#'   \code{psu} and with data column/s containing the continuous variable/s of
#'   interest with column names corresponding to \code{params} values
#' @param params A vector of column names corresponding to the continuous
#'   variables of interest contained in \code{x}
#' @param threshold cut-off value for continuous variable to differentiate
#'   case and non-case
#' @return A numeric vector of the PROBIT estimate of each continuous variable
#'   of interest with length equal to \code{length(params)}
#' @examples
#'
#' # Example call to bootBW function:
#'
#' bootPROBIT(x = indicatorsCH1,
#'            params = "muac1",
#'            threshold = 115)
#'
bootPROBIT <- function(x, params, threshold = THRESHOLD) {
  #
  # Get data
  #
  d <- x[[params[1]]]
  #
  # Shift data to the left to avoid "comutation instability" when :
  #
  #   max(x) / min(x)
  #
  # is small (i.e. close to unity).
  #
  shift <- min(min(d, na.rm = TRUE), threshold) - 1
  d <- d - shift
  threshold <- threshold - shift
  #
  # Box-cox transformation
  #
  lambda <- powerTransform(d)$lambda
  d <- bcPower(d, lambda)
  threshold <- bcPower(threshold, lambda)
  m <- mean(d, na.rm = TRUE)
  s <- sd(d, na.rm = T)
  #
  # PROBIT estimate
  #
  x <- pnorm(q = threshold, mean = m, sd = s)
  return(x)
}
