test_that("sample_G must be a zero-argument function", {
  expect_error(init_avseqmc_progress(sample_G = 5, epsilon = 0.05),
               "sample_G must be a function")
  expect_error(init_avseqmc_progress(sample_G = function(x) x, epsilon = 0.05),
               "sample_G must be a function that takes no arguments")
  expect_silent(init_avseqmc_progress(sample_G = function() sample(0:1,1), epsilon = 0.05))
})

test_that("epsilon must be numeric scalar strictly between 0 and 1", {
  expect_error(init_avseqmc_progress(sample_G = function() 0, epsilon = "a"),
               "epsilon must be a single number strictly between 0 and 1")
  expect_error(init_avseqmc_progress(sample_G = function() 0, epsilon = c(0.1, 0.2)),
               "epsilon must be a single number strictly between 0 and 1")
  expect_error(init_avseqmc_progress(sample_G = function() 0, epsilon = 0),
               "epsilon must be a single number strictly between 0 and 1")
  expect_error(init_avseqmc_progress(sample_G = function() 0, epsilon = 1),
               "epsilon must be a single number strictly between 0 and 1")
  expect_silent(init_avseqmc_progress(sample_G = function() 0, epsilon = 0.5))
})

test_that("ptilde must be numeric scalar between 0 and 1", {
  expect_error(init_avseqmc_progress(sample_G=function() 0, epsilon=0.05, ptilde="x"),
               "ptilde must be a single number between 0 and 1")
  expect_error(init_avseqmc_progress(sample_G=function() 0, epsilon=0.05, ptilde=c(0.2,0.3)),
               "ptilde must be a single number between 0 and 1")
  expect_error(init_avseqmc_progress(sample_G=function() 0, epsilon=0.05, ptilde=-0.1),
               "ptilde must be a single number between 0 and 1")
  expect_error(init_avseqmc_progress(sample_G=function() 0, epsilon=0.05, ptilde=1.1),
               "ptilde must be a single number between 0 and 1")
  expect_warning(init_avseqmc_progress(sample_G=function() 0, epsilon=0.05, ptilde=0.5, n=1, S=1),
              "Initializing avseqmc_progress with a previous value of n, S and ptilde. Note that ptilde is not recomputed and is assumed correct.")
})

test_that("n must be an integer >= 1", {
  expect_error(init_avseqmc_progress(sample_G=function() 0, epsilon=0.05, n=0),
               "n must be a single integer greater than or equal to 1")
  expect_error(init_avseqmc_progress(sample_G=function() 0, epsilon=0.05, n=1.5),
               "n must be a single integer greater than or equal to 1")
  expect_error(init_avseqmc_progress(sample_G=function() 0, epsilon=0.05, n="1"),
               "n must be a single integer greater than or equal to 1")
  expect_silent(init_avseqmc_progress(sample_G=function() 0, epsilon=0.05, n=5, S=1))
})

test_that("S must be an integer >= 0", {
  expect_error(init_avseqmc_progress(sample_G=function() 0, epsilon=0.05, S=-1),
               "S must be a single integer greater than or equal to 1")
  expect_error(init_avseqmc_progress(sample_G=function() 0, epsilon=0.05, S=1.2),
               "S must be a single integer greater than or equal to 1")
  expect_error(init_avseqmc_progress(sample_G=function() 0, epsilon=0.05, S="1"),
               "S must be a single integer greater than or equal to 1")
  expect_silent(init_avseqmc_progress(sample_G=function() 0, epsilon=0.05, S=0, n=10))
  expect_silent(init_avseqmc_progress(sample_G=function() 0, epsilon=0.05, S=10,n=10))
})

test_that("sample_G and epsilon must be provided", {
  expect_error(init_avseqmc_progress(epsilon=0.05),
               "sample_G and risk of overestimated significance epsilon must be specified")
  expect_error(init_avseqmc_progress(sample_G=function() 0),
               "sample_G and risk of overestimated significance epsilon must be specified")
  expect_silent(init_avseqmc_progress(sample_G=function() 0, epsilon=0.05))
})

test_that("Valid combinations of n, S, ptilde argument presence", {
  # None of n, S, ptilde - empty object
  expect_silent(init_avseqmc_progress(sample_G=function() 0, epsilon=0.05))

  # n + ptilde (warning expected)
  expect_warning(init_avseqmc_progress(sample_G=function() 0, epsilon=0.05, n=10, ptilde=0.1),
                 "Initializing avseqmc_progress with a previous value of ptilde, without the previously observed number of ones S")

  # n + S (no warning)
  expect_silent(init_avseqmc_progress(sample_G=function() 0, epsilon=0.05, n=10, S=5))

  # n + S + ptilde (warning expected)
  expect_warning(init_avseqmc_progress(sample_G=function() 0, epsilon=0.05, n=10, S=5, ptilde=0.1),
                 "Initializing avseqmc_progress with a previous value of n, S and ptilde")

  # Invalid combinations (only S)
  expect_error(init_avseqmc_progress(sample_G=function() 0, epsilon=0.05, S=5),
               "Invalid input. To initialize a avseqmc_progress object with earlier simulation progress")

  # Invalid combinations (only ptilde)
  expect_error(init_avseqmc_progress(sample_G=function() 0, epsilon=0.05, ptilde=0.1),
               "Invalid input. To initialize a avseqmc_progress object with earlier simulation progress")

  # Invalid combinations (ptilde + S only)
  expect_error(init_avseqmc_progress(sample_G=function() 0, epsilon=0.05, ptilde=0.1, S=5),
               "Invalid input. To initialize a avseqmc_progress object with earlier simulation progress")
})

test_that("Returned object has correct class and components", {
  res <- init_avseqmc_progress(sample_G=function() 0, epsilon=0.05)
  expect_s3_class(res, "avseqmc_progress")
  expect_equal(res$epsilon, 0.05)
  expect_equal(res$sample_G(), 0)
  expect_equal(res$S, 0)
  expect_equal(res$n, 0)
  expect_null(res$ptilde)
})

test_that("Lengths of B, Bn, ptilde, Ltilde are consistent", {
  # Case 1: no history (empty)
  res_empty <- init_avseqmc_progress(sample_G = function() 0, epsilon = 0.05)
  expect_true(is.null(res_empty$B))
  expect_true(is.null(res_empty$Bn))
  expect_null(res_empty$ptilde)  # can be NULL
  expect_null(res_empty$Ltilde)  # can be NULL

  # lengths
  expect_length(res_empty$B, 0)
  expect_length(res_empty$Bn, 0)
  expect_length(res_empty$ptilde, 0)
  expect_length(res_empty$Ltilde, 0)

  # Case 2: with history (n + S, ptilde recomputed)
  res_hist <- init_avseqmc_progress(sample_G = function() 0, epsilon = 0.05, n = 10, S = 5)
  expect_equal(length(res_hist$B), 1)
  expect_equal(length(res_hist$Bn), 1)
  expect_equal(length(res_hist$ptilde), 1)
  expect_equal(length(res_hist$Ltilde), 1)

  expect_equal(length(res_hist$B), length(res_hist$Bn))
  expect_equal(length(res_hist$Bn), length(res_hist$ptilde))
  expect_equal(length(res_hist$ptilde), length(res_hist$Ltilde))

  # Case 3: with n + ptilde only (S computed)
  res_hist2 <- init_avseqmc_progress(sample_G = function() 0, epsilon = 0.05, n = 10, S = 5)
  expect_equal(length(res_hist2$B), 1)
  expect_equal(length(res_hist2$Bn), 1)
  expect_equal(length(res_hist2$ptilde), 1)
  expect_equal(length(res_hist2$Ltilde), 1)

  expect_equal(length(res_hist2$B), length(res_hist2$Bn))
  expect_equal(length(res_hist2$Bn), length(res_hist2$ptilde))
  expect_equal(length(res_hist2$ptilde), length(res_hist2$Ltilde))
})
