#' Throws an error if a condition isn't met.
#'
#' The workhorse of the package.  If a condition isn't met, then an error
#' is thrown.
#'
#' @param x Input to check.  If missing, pass no args to \code{predicate}.
#' @param predicate Function that returns a logical value (possibly 
#' a vector).
#' @param msg The error message, in the event of failure.
#' @param what Either 'all' or 'any', to reduce vectorised tests to a 
#' single value.
#' @param ... Passed to the \code{predicate} function.
#' @return \code{FALSE} with the attribute \code{message}, as provided
#' in the input.
assert_engine <- function(x, predicate, msg, what = c("all", "any"), ...)
{
  handler <- match.fun(match.arg(
    getOption("assertive.severity"),
    c("stop", "warning", "message")
    ))
  what <- match.fun(match.arg(what))
  #Some functions, e.g., is.R take no args
  ok <- if(missing(x)) predicate() else predicate(x, ...)
  if(!what(ok))
  {
    if(missing(msg)) 
    {
      if(is_scalar(ok))
      {
        msg <- cause(ok)
      } else
      {
        stop("Bug in assertive; error message is missing")
      }
    }
    handler(msg, call. = FALSE)
  }
}

#' Wrapper to vapply that returns booleans.
#' 
#' Wrapper to \code{\link{vapply}} for functions that return a boolean (logical scalar) value.
#' 
#' @param x A vector (atomic or list).
#' @param predicate A predicate (function that returns a bool) to apply elementwise to \code{x}.
#' @param USE.NAMES Passed to \code{vapply}.
#' @param ... Passed to \code{vapply}.
#' @return A logical vector.
#' @seealso \code{\link{vapply}}.
bapply <- function(x, predicate, ..., USE.NAMES = TRUE)
{
  vapply(x, predicate, logical(1L), ..., USE.NAMES = TRUE)
}

#' Call a function, and give the result names.
#'
#' Calls a function, and names the result with the first argument.
#'
#' @param fn A function to call.  See note below.
#' @param x The first input to \code{fn}.
#' @param ... Optional additional inputs to \code{fn}.
#' @return The result of \code{fn(x, ...)}, with names given by the
#' argument \code{x}.
#' @note The function, \code{fn}, should return an object with the 
#' same length as the input \code{x}.
#' @examples
#' \dontrun{
#' call_and_name(is.finite, c(1, Inf, Na))
#' }
#' @seealso \code{\link{cause}} and \code{\link{na}}.
call_and_name <- function(fn, x, ...)
{
  y <- fn(x, ...)
  if(!is_identical_to_true(length(y) == length(x)))
  {
    warning("Vector of names is different length to results.  Trying to resize.")
    length(x) <- length(y)
  }
  names(y) <- x
  y
}

#' Convert a character vector to a list of numeric vectors.
#'
#' Split strings by character, then convert to numeric.
#' @param x Input to convert.
#' @return A list of numeric vectors.
#' @examples
#' \dontrun{
#' character_to_list_of_numeric_vectors(c("123", "4567a"))
#' }
#' @seealso \code{\link[base]{strsplit}} and \code{\link[base]{as.numeric}}.
character_to_list_of_numeric_vectors <- function(x)
{
  x <- coerce_to(x, "character")
  lapply(strsplit(x, ""), as.numeric)
}

#' Create a regex from components.
#'
#' Creates a regex from regex components.
#' @param ... Character vectors of regex components.
#' @param l A list of character vectors for alternate specification.
#' @param sep Regex for separating components of complete regex.
#' Defaults to "an optional space or hyphen".
#' @return A string containing a regex.
#' Each element in the vectors are pasted together, separated by the
#' \code{sep} value.  Those character vectors are then preceded by "^"
#' (regex for 'start of string'() and followed by "$" (regex for end
#' of string).  Finally, the regexes are collapsed with "|" (regex for
#' 'or').
#' @examples
#' \dontrun{
#' cas_number_components <- c("[[:digit:]]{1,7}", "[[:digit:]]{2}", "[[:digit:]]")
#' cas_number_rx <- create_regex(rx_components, sep = "-")
#' }
create_regex <- function(..., l = list(), sep = "[- ]?")
{
  x <- merge_dots_with_list(..., l = l)
  rx <- vapply(x, function(x) paste0(x, collapse = sep), character(1))
  paste0("^", rx, "$", collapse = "|")
}

#' Create regex for repeated digits
#' 
#' Creates a regex string for repeated digits.
#' 
#' @param lo Minimum number of digits to match.
#' @param hi Optional maximum number of digits to match.
#' @param optional If \code{TRUE}, the digits are optional.
#' @note If \code{hi} is omitted, the returned regex will only match the exact number
#' of digits given by \code{lo}.
#' @return A character vector of regexes.
#' @examples
#' \dontrun{
#' d(3)
#' d(3, 4)
#' d(1:5, 6)
#' }
d <- function(lo, hi, optional = FALSE)
{
  lo <- as.integer(lo)
  assert_all_are_non_negative(lo)
  if(!missing(hi))
  {
    hi <- as.integer(hi)
    assert_all_are_true(hi > lo)
    lo <- paste(lo, hi, sep = ",")
  }
  rx <- paste0("[[:digit:]]{", lo, "}")
  rx <- sub("{1}", "", rx, fixed = TRUE)
  if(optional)
  {
    rx <- paste0("(", rx, ")?")
  }
  rx
}

#' FALSE, with a cause of failure.
#'
#' Always returns the value \code{FALSE}, with a cause attribute.
#'
#' @param ... Passed to sprintf to create a cause of failure message.
#' @return \code{FALSE} with the attribute \code{cause}, as provided
#' in the input.
#' @seealso \code{\link{cause}} and \code{\link{na}}.
false <- function(...)
{
  msg <- if(length(list(...)) > 0L) sprintf(...) else ""
  x <- FALSE
  cause(x) <- msg
  x
}

#' Allowed locale categories.
#'
#' The categories of locale that can be gotten/set.
#'
#' @param include_all If \code{TRUE}, the value \code{LC_ALL} is included.
#' @return A character vector of locale categories.
#' @seealso \code{\link{sys_get_locale}}.
locale_categories <- function(include_all = TRUE)
{
  allowed_categories <- c(
    if(include_all) "ALL",
    "COLLATE", "CTYPE", "MONETARY", "NUMERIC", "TIME",
    if(is_unix()) c("MESSAGES", "PAPER", "MEASUREMENT")
  )
  paste0("LC_", allowed_categories)
}

#' Does the input match the regular expression?
#' 
#' Checks that the input matches the regular expression.
#'
#' @param x Input to check.
#' @param rx A regular expression.
#' @param ignore.case Should the case of alphabetic chracters be ignored?
#' @param ... Passed to \code{\link{grepl}}.
#' @note The default for \code{ignore.case} is different to the default in \code{grepl}.
#' @return A logical vector that is \code{TRUE} when the input matches the regular expression.
#' @seealso \code{\link{regex}} and \code{\link{regexpr}}.
matches_regex <- function(x, rx, ignore.case = TRUE, ...)
{
  call_and_name(function(x) grepl(rx, x, ignore.case = ignore.case, ...), x)
}

#' The most common value in a vector.
#'
#' The modal value of a vector.
#' @param x vector to find the modal value of.
#' @note Probably very inefficient; not suitable for general use.
#' @return The modal value of \code{x}.
modal_value <- function(x)
{
  names(sort(table(x), descending = TRUE))[1]
}

#' NA, with a cause of failure.
#'
#' Always returns the value (logical) \code{NA}, with a cause attribute.
#'
#' @param ... Passed to sprintf to create a cause of failure message.
#' @return \code{NA} with the attribute \code{cause}, as provided
#' in the input.
#' @seealso \code{\link{cause}} and \code{\link{false}}.
na <- function(...)
{
  msg <- if(length(list(...)) > 0L) sprintf(...) else ""
  x <- NA
  cause(x) <- msg
  x
}


#' Removes invalid characters from a string.
#'
#' Removes invalid characters from a string, leaving only digits.
#' @param x Input to strip.
#' @param invalid_chars A regular expression detailing characters to remove.
#' @param char_desc A string describing the characters to remove.
#' @param allow_x If \code{TRUE}, the letter "X" is allowed - useful for check digits.
#' @param allow_plus If \code{TRUE}, the symbol "+" is allowed - useful for phone numbers.
#' @return A character vector of the same length as \code{x}, consisting of strings without
#' the characters detailed in the \code{invalid_chars}.
#' @examples
#' \dontrun{
#' strip_invalid_chars("  We're floating\tin    space\n\n\n", "[[:space:]]", "whitespace")
#' strip_non_numeric(" +44 800-123-456 ", allow_plus = TRUE)
#' #Inputs such as factors as coerced to character.
#' strip_non_alphanumeric(factor(c(" A1\t1AA.", "*(B2^2BB)%")))
#' }
strip_invalid_chars <- function(x, invalid_chars, char_desc = "invalid")
{
  x <- coerce_to(x, "character")
  if(any(grepl(invalid_chars, x)))
  {
    warning("Removing ", char_desc, " characters from input.")
    x <- gsub(invalid_chars, "", x)
  }
  x
}

#' @rdname strip_invalid_chars
strip_non_alphanumeric <- function(x)
{
  strip_invalid_chars(x, "[^[:alnum:]]+", "non-alphanumeric")
}

#' @rdname strip_invalid_chars
strip_non_numeric <- function(x, allow_x = FALSE, allow_plus = FALSE)
{
  invalid_chars <-paste0("[^[:digit:]", if(allow_x) "X", if(allow_plus) "\\+", "]+", collapse = "")
  strip_invalid_chars(x, invalid_chars, "non-numeric")
}
