% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ara_unconstrained_l2.R
\name{ara_unconstrained_l2}
\alias{ara_unconstrained_l2}
\title{Unconstrained Adaptable Radial Axes (ARA) mappings using the L2 norm}
\usage{
ara_unconstrained_l2(X, V, weights = rep(1, ncol(X)), solver = "formula")
}
\arguments{
\item{X}{Numeric data matrix of dimensions N x n, where N is the number of
observations, and n is the number of variables.}

\item{V}{Numeric matrix defining the axes or "axis vectors". Its dimensions are n x m,
where 1<=m<=3 is the dimension of the visualization space. Each row of
\code{V} defines an axis vector.}

\item{weights}{Numeric array specifying optional non-negative weights associated with each
variable. The function only considers them if they do not share the same
value. Default: array of n ones.}

\item{solver}{String indicating a package or method for solving the optimization problem.
It can be "formula" (default), where the solution is obtained through a
closed-form formula, or "CVXR".}
}
\value{
A list with the three following entries:
\describe{
\item{\code{P}}{A numeric N x m matrix containing the mapped points. Each
row is the low-dimensional representation of a data observation in X.}
\item{\code{status}}{A vector of length N where the i-th element contains
the status of the chosen solver when calculating the mapping of the i-th
data observation. The type of the elements depends on the particular chosen
solver.}
\item{\code{objval}}{The numeric objective value associated with the
solution to the optimization problem, considering matrix norms, and
ignoring weights.}
}
When solver is "formula" this function always produces valid solutions
(\code{P}), since the pseudo-inverse matrix always exists. Thus, the output
status vector is not relevant, but is returned in consonance with other
adaptable radial axes functions in the package. If \pkg{CVRX} were used and
failed to map the data observations (i.e., failed to solve the related
optimization problem), \code{P} would be a matrix containing \code{NA} (not
available) values, and \code{objval} would be also be \code{NA}.
}
\description{
\code{ara_unconstrained_l2()} computes \strong{unconstrained}
\strong{Adaptable Radial Axes} (ARA) mappings for the \strong{L2 norm}
}
\details{
\code{ara_unconstrained_l2()} computes low-dimensional point representations
of high-dimensional numerical data (\code{X}) according to the data
visualization method "Adaptable Radial Axes" (M. Rubio-Sánchez, A. Sanchez,
and D. J. Lehmann (2017), doi: 10.1111/cgf.13196), which
describes a collection of convex norm optimization problems aimed at
minimizing estimates of original values in \code{X} through dot products of
the mapped points with the axis vectors (rows of \code{V}). This particular
function solves the unconstrained optimization problem in Eq. (10), for the
squared-Euclidean norm. Optional non-negative weights (\code{weights})
associated with each data variable can be supplied to solve the problem in
Eq. (15).
}
\examples{
# Load data
data("auto_mpg", package = "ascentTraining")

# Define subset of (numerical) variables
# 1:"mpg", 4:"horsepower", 5:"weight", 6:"acceleration"
selected_variables <- c(1, 4, 5, 6)

# Retain only selected variables and rename dataset as X
X <- auto_mpg[, selected_variables] # Select a subset of variables
rm(auto_mpg)

# Remove rows with missing values from X
N <- nrow(X)
rows_to_delete <- NULL
for (i in 1:N) {
  if (sum(is.na(X[i, ])) > 0) {
    rows_to_delete <- c(rows_to_delete, -i)
  }
}
X <- X[rows_to_delete, ]

# Convert X to matrix
X <- apply(as.matrix.noquote(X), 2, as.numeric)

# Standardize data
Z <- scale(X)

# Define axis vectors (2-dimensional in this example)
r <- c(0.8, 1, 1.2, 1)
theta <- c(225, 100, 315, 80) * 2 * pi / 360
V <- geometry::pol2cart(theta, r)

# Define weights
weights <- c(1, 0.75, 0.75, 1)

# Compute the mapping
mapping <- ara_unconstrained_l2(
  Z,
  V,
  weights = weights,
  solver = "formula"
)

# Select variables with labeled axis lines on ARA plot
axis_lines <- c(1, 4) # 1:"mpg", 4:"acceleration")

# Select variable used for coloring embedded points
color_variable <- 1 # "mpg"

# Draw the ARA plot
draw_ara_plot_2d_standardized(
  Z,
  X,
  V,
  mapping$P,
  weights = weights,
  axis_lines = axis_lines,
  color_variable = color_variable
)

}
\references{
M. Rubio-Sánchez, A. Sanchez, D. J. Lehmann: Adaptable radial axes plots for
improved multivariate data visualization. Computer Graphics Forum 36, 3
(2017), 389–399. \href{https://onlinelibrary.wiley.com/doi/10.1111/cgf.13196}{doi:10.1111/cgf.13196}
}
