#' Run an APSIM-X Simulation
#' 
#' A valid apsimx file can be run from within R. The main goal is to make running APSIM-X
#' simple, especially for large scale simulations or parameter optimization
#' 
#' @title Run an APSIM-X simulation
#' @name apsimx
#' @description Run apsimx from R. It uses \sQuote{system} (unix) or \sQuote{shell} (windows) and it attempts to be platform independent.
#' @param file file name to be run (the extension .apsimx is optional)
#' @param src.dir directory containing the .apsimx file to be run (defaults to the current directory)
#' @param silent whether to print messages from apsim simulation
#' @param value how much output to return: \cr
#'              option \sQuote{all} returns all components of the simulation; \cr
#'              option \sQuote{report} returns only the \sQuote{main} report component; \cr
#'              option \sQuote{none} does not create a data.frame but it generates the databases from APSIM-X
#' @param cleanup logical. Whether to delete the .db file generated by APSIM-X. Default is FALSE
#' @return a data frame with the \sQuote{Report} from the APSIM-X simulation. To change the variables being reported use the GUI.
#' @export
#' @examples 
#' \dontrun{
#' ## See function 'apsimx_example' and vignette 'apsimx'
#' }
#'

apsimx <- function(file = "", src.dir = ".",
                   silent = FALSE, 
                   value = c("all", "report", "none"),
                   cleanup = FALSE){
  
  value <- match.arg(value)
  
  if(file == "") stop("need to specify file name")
  
  .check_apsim_name(file)
  
  ## The might offer suggestions in case there is a typo in 'file'
  file.names <- dir(path = src.dir, pattern=".apsimx$", ignore.case=TRUE)
  
  if(length(file.names) == 0){
    stop("There are no .apsimx files in the specified directory to run.")
  }
  
  file <- match.arg(file, file.names)
  
  file.name.path <- file.path(src.dir, file)
  
  ada <- auto_detect_apsimx()

  if(.Platform$OS.type == "unix"){
    mono <- system("which mono", intern = TRUE)
    run.strng <- paste0(mono, " ", ada, " ", file.name.path) ## Could use 'paste' instead I guess...
    ## Run APSIM-X on the command line
    system(command = run.strng, ignore.stdout = silent)
  }
  
  if(.Platform$OS.type == "windows"){
    ## Should probably delete the line below
    ## As of 2020-07-04, this seems to be working fine
    ## I will leave the line below until I'm sure it is not needed
    ## if(src.dir != ".") stop("In Windows you can only run a file from the current directory.")
    run.strng <- paste0(ada, " ", file.name.path)
    shell(cmd = run.strng, translate = TRUE, intern = TRUE)
  }

  if(value != "none"){
    ans <- read_apsimx(file = sub("apsimx","db",file), src.dir = src.dir, value = value)
  }else{
    if(value == "none" && !silent){
      cat("APSIM created .db files, but nothing is returned \n")
    }
  }
  
  if(cleanup){
    ## Default is not to cleanup
    if(value == "none") stop("do not clean up if you choose value = 'none' ")
    ## Delete the apsim-generated sql database 
    file.remove(paste0(src.dir, "/", sub("apsimx", "db", file)))
  }
  
  if(value != "none")
    return(ans)
}

## This is an internal function so I won't export/document it
#' @noRd
auto_detect_apsimx <- function(){

  if(.Platform$OS.type == "unix"){

    if(grepl("Darwin", Sys.info()[["sysname"]])){
      laf <- list.files("/Applications/")
      find.apsim <- grep("APSIM", laf)
      ## This deals with the fact that APSIM-X might not be present but perhaps a
      ## custom version is available
      if(length(find.apsim) == 0){
        ## I only throw a warning because maybe the user has a custom version of APSIM-X only
        if(!is.na(apsimx::apsim.options$exe.path) && apsimx::apsimx.options$warn.find.apsimx){
          warning("APSIM-X not found, but a custom one is present")
        }else{
          stop("APSIM-X not found and no 'exe.path' exists.")
        }
      }
      ## If only one version of APSIM-X is present
      ## APSIM executable
      st1 <- "/Applications/"
      st3 <- "/Contents/Resources/Bin/Models.exe" 
      if(length(find.apsim) == 1){
        apsimx.name <- laf[find.apsim]
        apsimx_dir <- paste0(st1, apsimx.name, st3)
        ## I could use 'file.path' instead, but it this is not a 'file'
        ## so it could be confusing
      }
      ## If there is more than one version of APSIM-X
      if(length(find.apsim) > 1){
        ## Originally I was sorting by #issue number but this
        ## does not give you the latest version
        len.fa <- length(find.apsim)
        ## This extracts the date from the APSIM name but I 
        ## only need this for debugging in case there is a problem
        fa.dt <- as.numeric(sapply(laf[find.apsim], .favd))
        newest.version <- laf[find.apsim][len.fa]
        if(apsimx::apsimx.options$warn.versions &&
           is.na(apsimx::apsim.options$exe.path)){
           warning(paste("Multiple versions of APSIM-X installed. \n
                    Choosing the newest one:", newest.version))
        }
        apsimx.name <- newest.version
        apsimx_dir <- paste0(st1, apsimx.name, st3)
      }
    }
  
    if(grepl("Linux", Sys.info()[["sysname"]])){
      
      find.apsim <- grep("apsim", list.files("/usr/local/lib"))
      ## What if length equals zero?
      if(length(find.apsim) == 0){
        ## I only throw a warning because maybe the user has a custom version of APSIM-X only
        if(!is.na(apsimx::apsim.options$exe.path) && apsimx::apsimx.options$warn.find.apsimx){
          warning("APSIM-X not found, but a custom one is present")
        }else{
          stop("APSIM-X not found and no 'exe.path' exists.")
        }
      }
      ## APSIM executable
      st1 <- "/usr/local/lib/apsim/"
      st3 <- "/Bin/Models.exe" 
      if(length(find.apsim) == 1){
        apsimx.versions <- list.files("/usr/local/lib/apsim")
        apsimx.name <- apsimx.versions
        apsimx_dir <- paste0(st1,apsimx.name,st3)
      }
      
      ## Note: Apparently Debian does not tolerate multiple 
      ## versions of APSIM-X installed, date 2019-12-12
      if(length(apsimx.versions) > 1){
        len.fa <- length(find.apsim)
        newest.version <- apsimx.versions[find.apsim]
        if(apsimx::apsimx.options$warn.versions){
          warning(paste("Multiple versions of APSIM-X installed. \n
                    Choosing the newest one:",newest.version))
        }
        apsimx.name <- newest.version
        apsimx_dir <- paste0(st1,apsimx.name,st3)
      }
    }
  }
  
  if(.Platform$OS.type == "windows"){
    st1 <- "C:/PROGRA~1/"
    laf <- list.files(st1)
    find.apsim <- grep("APSIM", laf)
    
    ## What if length equals zero?
    if(length(find.apsim) == 0){
      ## I only throw a warning because maybe the user has a custom version of APSIM-X only
      if(!is.na(apsimx::apsim.options$exe.path) && apsimx::apsimx.options$warn.find.apsimx){
        warning("APSIM-X not found, but a custom one is present")
      }else{
        stop("APSIM-X not found and no 'exe.path' exists.")
      }
    }
    
    st3 <- "/Bin/Models.exe" 
    if(length(find.apsim) == 1){
      apsimx.versions <- laf[find.apsim]
      apsimx.name <- apsimx.versions
      apsimx_dir <- paste0(st1, apsimx.name, st3)
    }
    
    if(length(find.apsim) > 1){
      apsimx.versions <- laf[find.apsim]
      newest.version <- apsimx.versions[length(find.apsim)]
      if(apsimx::apsimx.options$warn.versions){
        warning(paste("Multiple versions of APSIM-X installed. \n
                    Choosing the newest one:", newest.version))
      }
      apsimx.name <- newest.version
      apsimx_dir <- paste0(st1, apsimx.name, st3)
    }
  }
  
  if(!is.na(apsimx::apsimx.options$exe.path)){
    ## Windows paths might contain white spaces which are
    ## problematic when running them at the command line
    if(grepl("\\s", apsimx::apsimx.options$exe.path))
      stop("White spaces are not allowed in APSIM-X executable path")
    apsimx_dir <- apsimx::apsimx.options$exe.path
  }
  return(apsimx_dir)
}

#' Auto detect where apsimx examples are located 
#' 
#' @title Auto detect where apsimx examples are located
#' @name auto_detect_apsimx_examples
#' @description simple function to detect where APSIM-X examples are located
#' @return will create a directory pointing to APSIM-X distributed examples
#' @export
#' @examples 
#' \dontrun{
#' ex.dir <- auto_detect_apsimx_examples()
#' }
#' 

auto_detect_apsimx_examples <- function(){
  
  if(.Platform$OS.type == "unix"){
    apsim.name <- NULL # Why did I create this NULL variable?
    if(grepl("Darwin", Sys.info()[["sysname"]])){
      ## If APSIM-X is installed it will be in /Applications/
      ## look into Applications folder
      laf <- list.files("/Applications/")
      find.apsim <- grep("APSIM",laf)
      
      if(length(find.apsim) == 0) stop("APSIM-X examples not found")
      
      if(length(find.apsim) > 1){
        newest.version <- laf[find.apsim][length(find.apsim)]
        if(apsimx::apsimx.options$warn.versions){
          warning(paste("Multiple versions of APSIM-X installed. \n
                    Choosing the newest one:",newest.version))
        }
        apsimx.name <- newest.version
      }else{
        apsimx.name <- laf[find.apsim]
      }
      st1 <- "/Applications/"
      st3 <- "/Contents/Resources/Examples" 
      apsimx_ex_dir <- paste0(st1, apsimx.name, st3)
      return(apsimx_ex_dir)
    }
  
    if(grepl("Linux", Sys.info()[["sysname"]])){
      st1 <- "/usr/local/lib/apsim/"
      apsimx.versions <- list.files(st1) 
      ## In Ubuntu it looks like you can only have one version 
      ## of APSIM-X installed
      if(length(apsimx.versions) > 1){
        len.fa <- length(find.apsim)
        newest.version <- apsimx.versions[find.apsim]
        if(apsimx.options$warn.versions){
          warning(paste("Multiple versions of APSIM-X installed. \n
                    Choosing the newest one:", newest.version))
        }
        apsimx.name <- newest.version
      }else{
        apsimx.name <- apsimx.versions
      }
      apsimx_ex_dir <- paste0(st1, apsimx.name, "/Examples")
    }
  }
  
  if(.Platform$OS.type == "windows"){
      st1 <- "C:/PROGRA~1"
      laf <- list.files(st1)
      find.apsim <- grep("APSIM", laf)
      
      if(length(find.apsim) == 0) stop("APSIM-X not found")
      
      apsimx.versions <- laf[find.apsim]
      if(length(apsimx.versions) > 1){
        newest.version <- laf[find.apsim][length(find.apsim)]
        if(apsimx.options$warn.versions){
          warning(paste("Multiple versions of APSIM-X installed. \n
                        Choosing the newest one:", newest.version))
        }
        apsimx.name <- newest.version
      }else{
        apsimx.name <- apsimx.versions
      }
      ## APSIM path to examples
      st3 <- "/Examples" 
      apsimx_ex_dir <- paste0(st1, "/", apsimx.name, st3)
  }
  
  if(!is.na(apsimx::apsimx.options$examples.path)){
    ## I dislike white spaces in paths!
    ## I'm looking at you Windows!
    if(grepl("\\s", apsimx::apsimx.options$examples.path))
      stop("White spaces are not allowed in examples.path")
    apsimx_ex_dir <- apsimx::apsimx.options$examples.path
  }
  return(apsimx_ex_dir)
}

#'
#' @title Access Example APSIM-X Simulations
#' @name apsimx_example
#' @description simple function to run some of the built-in APSIM-X examples
#' @param example run an example from built-in APSIM-X. Options are all of the ones included with the APSIM-X distribution, except \sQuote{Graph}.
#' @param silent whether to print standard output from the APSIM-X execution
#' @note This function creates a new column \sQuote{Date} which is in the R \sQuote{Date} format which is convenient for graphics.
#' @details This function creates a temporary copy of the example file distributed with APSIM-X to avoid writing a .db file 
#'          to the directory where the \sQuote{Examples} are located. It is not a good practice and there is no guarantee that 
#'          the user has read/write permissions in that directory.
#' @export
#' @examples 
#' \dontrun{
#' wheat <- apsimx_example("Wheat")
#' maize <- apsimx_example("Maize")
#' barley <- apsimx_example("Barley")
#' ## The 'Date' column is created by this function, based on apsim output.
#' require(ggplot2)
#' ggplot(data = wheat , aes(x = Date, y = Yield)) + 
#'   geom_point()
#' }
#' 

apsimx_example <- function(example = "Wheat", silent = FALSE){

  ## Write to a temp directory only
  tmp.dir <- tempdir()
  ## Run a limited set of examples
  ## Now the only one missing is Graph, which I assume is about
  ## graphics and not that relevant to apsimx
  ## Several examples are not supported because they do not use
  ## relative paths for the weather file
  ## Examples which do not run: Chicory
  ex.ch <- c("Barley","ControlledEnvironment","Eucalyptus",
             "EucalyptusRotation",
             "Maize","Oats","Sugarcane","Wheat")

  example <- match.arg(example, choices = ex.ch)
  
  ex.dir <- auto_detect_apsimx_examples()
  ex <- file.path(ex.dir, paste0(example, ".apsimx"))
  
  if(!file.exists(ex)) stop(paste0("cannot find example file ", example))
  ## Make a temporary copy of the file to the current directory
  ## Do not transfer permissions?
  file.copy(from = ex, to = tmp.dir, copy.mode = FALSE)
  
  sim <- apsimx(paste0(example, ".apsimx"), src.dir = tmp.dir, value = "report")

  ## OS independent cleanup (risky?)
  file.remove(paste0(tmp.dir, "/", example, ".db"))
  file.remove(paste0(tmp.dir, "/", example, ".apsimx"))

  return(sim)
}

#' Read APSIM-X generated .db files
#' 
#' @title Read APSIM-X generated .db files
#' @name read_apsimx
#' @description read SQLite databases created by APSIM-X runs. One file at a time.
#' @param file file name
#' @param src.dir source directory where file is located
#' @param value either \sQuote{report} (data.frame) or \sQuote{all} (list)
#' @export
#' 

read_apsimx <- function(file = "", src.dir = ".", value = c("report","all")){
  
  if(file == "") stop("need to specify file name")
  
  file.names <- dir(path = src.dir, pattern=".db$", ignore.case=TRUE)
  
  if(length(file.names) == 0){
    stop("There are no .db files in the specified directory to read.")
  }
  
  value <- match.arg(value)
  
  file.name.path <- file.path(src.dir, file)
  
  con <- DBI::dbConnect(RSQLite::SQLite(), file.name.path)
  ## create data frame for each table
  tbl0 <- DBI::dbGetQuery(con, "SELECT * FROM Report")
  tbl1 <- DBI::dbGetQuery(con, "SELECT * FROM _Checkpoints")
  tbl2 <- DBI::dbGetQuery(con, "SELECT * FROM _InitialConditions")
  tbl3 <- DBI::dbGetQuery(con, "SELECT * FROM _Messages")
  tbl4 <- DBI::dbGetQuery(con, "SELECT * FROM _Simulations")
  tbl5 <- DBI::dbGetQuery(con, "SELECT * FROM _Units")
  ## Disconnect
  DBI::dbDisconnect(con)
  
  if(any(grepl("Clock.Today",names(tbl0)))){
    tbl0$Date <- try(as.Date(sapply(tbl0$Clock.Today, function(x) strsplit(x, " ")[[1]][1])), silent = TRUE)
  }
  ## Return list
  if(value == "all"){
    ans <- list(Report = tbl0, Checkpoints = tbl1, InitialConditions = tbl2,
                Messages = tbl3, Simulations = tbl4, Units = tbl5)
  }
  ## Return data.frame
  if(value == "report"){
    ans <- tbl0
  }
  return(ans)
}

#' Read all APSIM-X generated .db files in a directory
#' 
#' @title Read all APSIM-X generated .db files in a directory
#' @name read_apsimx_all
#' @description Like \code{\link{read_apsimx}}, but it reads all .db files in a directory. 
#' @param src.dir source directory where files are located
#' @param value either \sQuote{report} or \sQuote{all} (only \sQuote{report} implemented at the moment)
#' @note Warning: very simple function at the moment, not optimized for memory or speed.
#' @export
#' 

read_apsimx_all <- function(src.dir = ".", value = c("report", "all")){
  
  ## This is super memorey hungry and not efficient at all, but it might work 
  ## for now
  
  value <- match.arg(value)
  
  file.names <- dir(path = src.dir, pattern=".db$", ignore.case=TRUE)
  
  ans <- NULL
  
  for(i in file.names){
    
    tmp <- read_apsimx(file.names[i], value = value)
    tmp.d <- data.frame(file.name = file.names[i], tmp)
    ans <- rbind(ans, tmp)
    
  }
  return(ans)
}

#' Set apsimx options
#' 
#' @title Setting some options for the package
#' @name apsimx_options
#' @description Set the path to the APSIM-X executable, examples and warning suppression. 
#' @param exe.path path to apsim executable. White spaces are not allowed.
#' @param examples.path path to apsim examples
#' @param warn.versions logical. warning if multiple versions of APSIM-X are detected.
#' @param warn.find.apsimx logical. By default a warning will be thrown if APSIM-X is not found. 
#' If \sQuote{exe.path} is \sQuote{NA} an error will be thrown instead.
#' @note It is possible that APSIM-X is installed in some alternative location other than the 
#'       defaults ones. Guessing this can be difficult and then the auto_detect functions might
#'       fail. Also, if multiple versions of APSIM-X are installed apsimx will choose the newest
#'       one but it will issue a warning. Suppress the warning by setting warn.versions = FLASE.
#' @export
#' @examples 
#'\dontrun{
#' names(apsimx.options)
#' apsimx_options(exe.path = "some-new-path-to-executable")
#' apsimx.options$exe.path
#' }

apsimx_options <- function(exe.path = NA, examples.path = NA, 
                           warn.versions = TRUE, warn.find.apsimx = TRUE){
  assign('exe.path', exe.path, apsimx.options)
  assign('examples.path', examples.path, apsimx.options)
  assign('warn.versions', warn.versions, apsimx.options)
  assign('warn.find.apsimx', warn.find.apsimx, apsimx.options)
}

#' Environment which stores APSIM-X options
#' 
#' @title Environment which stores APSIM-X options
#' @name apsimx.options
#' @description Environment which can store the path to the executable, warning settings and 
#'              where examples are located.
#'              Creating an environment avoids the use of global variables or other similar practices
#'              which would have possible undesriable consequences. 
#' @export
#' @examples 
#' \dontrun{
#' names(apsimx.options)
#' apsimx_options(exe.path = "some-new-path-to-executable")
#' apsimx.options$exe.path
#' }
#' 

apsimx.options <- new.env(parent = emptyenv())
assign('exe.path', NA, apsimx.options)
assign('examples.path', NA, apsimx.options)
assign('warn.versions', TRUE, apsimx.options)
assign('warn.find.apsimx', TRUE, apsimx.options)
assign('.run.local.tests', FALSE, apsimx.options)

## I'm planning to use '.run.local.tests' for running tests
## which do not require an APSIM install

## Import packages needed for apsimx to work correctly
#' @import DBI jsonlite knitr RSQLite xml2 
#' @importFrom utils read.table
#' @importFrom utils write.table
#' @importFrom tools file_path_sans_ext
NULL

