\name{logit}
\alias{logit}
\alias{logit.nests}
\alias{logit.cap}
\alias{logit.alm}
\title{(Nested) Logit Demand Calibration and Merger Simulation}

\description{Calibrates consumer demand using (Nested) Logit and then simulates the price effect of a merger between two firms
  under the assumption that all firms in the market are playing a
  differentiated products Bertrand pricing game.}

\usage{
logit(prices,shares,margins,
                ownerPre,ownerPost,
                normIndex=ifelse(sum(shares)<1,NA,1),
                mcDelta=rep(0,length(prices)),
                priceStart = prices,
                isMax=FALSE,
                labels=paste("Prod",1:length(prices),sep=""),
                ...
                )
logit.alm(prices,shares,margins,
                ownerPre,ownerPost,
                mcDelta=rep(0,length(prices)),
                priceStart = prices,
                isMax=FALSE,
                parmsStart,
                labels=paste("Prod",1:length(prices),sep=""),
                ...
                )

logit.nests(prices,shares,margins,
                ownerPre,ownerPost,
                nests=rep(1,length(shares)),
                normIndex=ifelse(sum(shares) < 1,NA,1),
                mcDelta=rep(0,length(prices)),
                priceStart = prices,
                isMax=FALSE,
                constraint = TRUE,
                parmsStart,
                labels=paste("Prod",1:length(prices),sep=""),
                ...
                )

logit.cap(prices,shares,margins,
                ownerPre,ownerPost,
                capacities,
                mktSize,
                normIndex=ifelse(sum(shares)<1,NA,1),
                mcDelta=rep(0,length(prices)),
                priceStart = prices,
                isMax=FALSE,
                labels=paste("Prod",1:length(prices),sep=""),
                ...
                )


             }
\arguments{
  \item{}{Let k denote the number of products produced by all firms playing the
    Bertrand pricing game.}
  \item{prices}{A length k vector of product prices.}
  \item{shares}{A length k vector of product (quantity) shares. Values must be
    between 0 and 1.}
  \item{margins}{A length k vector of product margins, some of which may
    equal NA.}
  \item{nests}{A length k vector identifying the nest that each
    product belongs to.}
  \item{capacities}{A length k vector of product capacities. Capacities
    must be at least as great as shares * mktSize.}
  \item{mktSize}{An integer equal to the number of potential
    customers. If an outside option is present, should include
    individuals who chose that option.}
  \item{normIndex}{An integer equalling the index (position) of the
    inside product whose mean valuation will be normalized to 1. Default
    is 1, unless `shares' sum to less than 1, in which case the default is
    NA and an outside good is assumed to exist.}
  \item{ownerPre}{EITHER a vector of length k whose values
    indicate which firm produced a product pre-merger OR
    a k x k matrix of pre-merger ownership shares.}
  \item{ownerPost}{EITHER a vector of length k whose values
    indicate which firm produced a product after the merger OR
    a k x k matrix of post-merger ownership shares.}
  \item{mcDelta}{A vector of length k where each element equals the
    proportional change in a product's marginal costs due to
    the merger. Default is 0, which assumes that the merger does not
    affect any products' marginal cost.}
  \item{constraint}{if TRUE, then the nesting parameters for all
    non-singleton nests are assumed equal. If FALSE, then each
    non-singleton nest is permitted to have its own value. Default is TRUE.}
  \item{priceStart}{A length k vector of starting values used to solve for
    equilibrium price. Default is the `prices' vector.}
  \item{isMax}{If TRUE, checks to see whether computed price equilibrium
    locally maximizes firm profits and returns a warning if not. Default is FALSE.}
  \item{parmsStart}{A vector of starting values used to solve for
    price coefficient and nest parameters. The first element should
    always be the price coefficient and the remaining elements should be
    nesting parameters. Theory requires the nesting parameters to be
    greater than the price coefficient. If missing then the random
    draws with the appropriate restrictions are employed.}
  \item{labels}{A k-length vector of labels. Default is "Prod#", where
    `#' is a number between 1 and the length of `prices'.}
  \item{...}{Additional options to feed to the \code{\link[nleqslv]{nleqslv}}
    optimizer used to solve for equilibrium prices.}

}



\details{Using product prices, quantity shares and all of the
  product margins from at least one firm, \code{logit} is able to
  recover the price coefficient and product mean valuations in a
  Logit demand model. \code{logit} then uses these
  calibrated parameters to simulate a merger between two firms.

  \code{logit.alm} is identical to \code{logit} except that it assumes
  that an outside product is included and uses additional margin
  information to estimate the share of the outside good.

  \code{logit.nests} is identical to \code{logit} except that it includes the `nests'
  argument which may be used to assign products to different
  nests. Nests are useful because they allow for richer substitution
  patterns between products. Products within the same nest are assumed
  to be closer substitutes than products in different nests. The degree
  of substitutability between products located in different nests is
  controlled by the value of the nesting parameter sigma.
  The nesting parameters for singleton nests (nests containing
  only one product) are not identified and normalized to 1.  The vector of
  sigmas is calibrated from the prices, revenue shares, and margins supplied
  by the user.

  By default, all non-singleton nests are assumed to have a common value for sigma.
  This constraint may be relaxed  by setting `constraint' to
  FALSE. In this case, at least one product margin must be supplied from
  a product within each nest.


  \code{logit.cap} is identical to  \code{logit}  except that firms are
  playing the Bertrand pricing game under exogenously supplied capacity
  constraints. Unlike \code{logit},  \code{logit.cap} requires users to
  specify capacity constraints via `capacities' and  the number of
  potential customers in a market via `mktSize'. `mktSize' is needed to
  transform `shares' into quantities that must be directly compared to `capacities'.

  In \code{logit}, \code{logit.nests} and \code{logit.cap},  if quantity shares sum to 1,
  then one product's mean value is not identified and must be normalized
  to 0. `normIndex' may be used to specify the index (position) of the
  product whose mean value is to be normalized. If the sum of revenue shares
  is less than 1, both of these functions assume that the exists a k+1st
  product in the market whose price and mean value are both normalized
  to 0.


}

\value{\code{logit} returns an instance of class
  \code{\linkS4class{Logit}}.
   \code{logit.alm} returns an instance of \code{\linkS4class{LogitALM}}, a
  child class of \code{\linkS4class{Logit}.}.
  \code{logit.nests} returns an instance of \code{\linkS4class{LogitNests}}, a
  child class of \code{\linkS4class{Logit}}.
  \code{logit.cap} returns an instance of \code{\linkS4class{LogitCap}}, a
  child class of \code{\linkS4class{Logit}.}}






\seealso{\code{\link{ces}}}

\author{Charles Taragin \email{charles.taragin@usdoj.gov}}

\references{
Anderson, Simon, Palma, Andre, and Francois Thisse (1992).
\emph{Discrete Choice Theory of Product Differentiation}.
The MIT Press, Cambridge, Mass.


Epstein, Roy and Rubinfeld, Daniel (2004).
``Effects of Mergers Involving Differentiated Products.''

Werden, Gregory and Froeb, Luke (1994).
``The Effects of Mergers in
Differentiated Products Industries: Structural Merger Policy and the
Logit Model'',
\emph{Journal of Law, Economics, \& Organization}, \bold{10}, pp. 407-426.

Froeb, Luke, Tschantz, Steven  and Phillip Crooke (2003).
``Bertrand Competition and Capacity Constraints: Mergers Among Parking Lots'',
\emph{Journal of Econometrics}, \bold{113}, pp. 49-67.

Froeb, Luke and Werden, Greg (1996).
``Computational Economics and Finance: Modeling and Analysis with Mathematica, Volume 2.''
In Varian H (ed.), chapter Simulating Mergers among Noncooperative Oligopolists, pp. 177-95.
Springer-Verlag, New York.
}




\examples{

## Calibration and simulation results from a merger between Budweiser and
## Old Style.
## Source: Epstein/Rubenfeld 2004, pg 80


prodNames <- c("BUD","OLD STYLE","MILLER","MILLER-LITE","OTHER-LITE","OTHER-REG")
ownerPre <-c("BUD","OLD STYLE","MILLER","MILLER","OTHER-LITE","OTHER-REG")
ownerPost <-c("BUD","BUD","MILLER","MILLER","OTHER-LITE","OTHER-REG")
nests <- c("Reg","Reg","Reg","Light","Light","Reg")

price    <- c(.0441,.0328,.0409,.0396,.0387,.0497)
shares   <- c(.066,.172,.253,.187,.099,.223)
margins <- c(.3830,.5515,.5421,.5557,.4453,.3769)

names(price) <-
    names(shares) <-
    names(margins) <-
    prodNames

result.logit <- logit(price,shares,margins,ownerPre=ownerPre,ownerPost=ownerPost,labels=prodNames)



print(result.logit)           # return predicted price change
summary(result.logit)         # summarize merger simulation

elast(result.logit,TRUE)      # returns premerger elasticities
elast(result.logit,FALSE)     # returns postmerger elasticities

diversion(result.logit,TRUE)  # return premerger diversion ratios
diversion(result.logit,FALSE) # return postmerger diversion ratios


cmcr(result.logit)            #calculate compensating marginal cost reduction
upp(result.logit)            #calculate Upwards Pricing Pressure Index

CV(result.logit)              #calculate representative agent compensating variation

## Construct a matrix containing all candidate product markets satisfying a 5\% SSNIP

## Implement the Hypothetical Monopolist Test
## for BUD and OLD STYLE using a 5\% SSNIP

HypoMonTest(result.logit,prodIndex=1:2)


#
# Logit With capacity Constraints
#

mktSize <- 1000
cap     <- c(66,200,300,200,99,300) # BUD and OTHER-LITE are capacity constrained
result.cap <- logit.cap(price,shares,margins,capacities=cap,mktSize=mktSize,ownerPre=ownerPre,ownerPost=ownerPost,labels=prodNames)
print(result.cap)

}