% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/readAntares.R
\name{readAntares}
\alias{readAntares}
\title{Read the data of an Antares simulation}
\usage{
readAntares(
  areas = NULL,
  links = NULL,
  clusters = NULL,
  districts = NULL,
  misc = FALSE,
  thermalAvailabilities = FALSE,
  hydroStorage = FALSE,
  hydroStorageMaxPower = FALSE,
  reserve = FALSE,
  linkCapacity = FALSE,
  mustRun = FALSE,
  thermalModulation = FALSE,
  select = NULL,
  mcYears = NULL,
  timeStep = c("hourly", "daily", "weekly", "monthly", "annual"),
  opts = simOptions(),
  parallel = FALSE,
  simplify = TRUE,
  showProgress = TRUE
)
}
\arguments{
\item{areas}{Vector containing the names of the areas to import. If
\code{NULL} no area is imported. The special value \code{"all"} tells the
function to import all areas. By default, the value is "all" when no other argument is enter and "NULL" when other arguments are enter.}

\item{links}{Vector containing the name of links to import. If \code{NULL} no
area is imported. The special value \code{"all"} tells the function to
import all areas. Use function \code{\link{getLinks}} to import all links
connected to some areas.}

\item{clusters}{Vector containing the name of the areas for which you want to
import results at cluster level. If \code{NULL} no cluster is imported. The
special value \code{"all"} tells the function to import clusters from all
areas.}

\item{districts}{Vector containing the names of the districts to import. If \code{NULL},
no district is imported. The special value \code{"all"} tells the function to import all
districts.}

\item{misc}{Vector containing the name of the areas for which you want to
import misc.}

\item{thermalAvailabilities}{Should thermal availabilities of clusters be imported ? If TRUE, the column
"thermalAvailability" is added to the result and a new column "availableUnits"
containing the number of available units in a cluster is created.If synthesis is set to TRUE then
"availableUnits" contain the mean of avaible units on all MC Years.}

\item{hydroStorage}{Should hydro storage be imported ?}

\item{hydroStorageMaxPower}{Should hydro storage maximum power be imported ?}

\item{reserve}{Should reserve be imported ?}

\item{linkCapacity}{Should link capacities be imported ?}

\item{mustRun}{Should must run productions be added to the result? If TRUE,
then four columns are added: \code{mustRun} contains the production of 
clusters that are in complete must run mode; \code{mustRunPartial}
contains the partial must run production of clusters; \code{mustRunTotal}
is the sum of the two previous columns. Finally \code{thermalPmin} is
similar to mustRunTotal except it also takes into account the production
induced by the minimum stable power of the units of a cluster. More
precisely, for a given cluster and a given time step, it is equal to 
\code{min(NODU x min.stable.power, mustRunTotal)}.}

\item{thermalModulation}{Should thermal modulation time series be imported ? If \code{TRUE}, the
columns "marginalCostModulation", "marketBidModulation", "capacityModulation"
and "minGenModulation" are added to the cluster data.}

\item{select}{Character vector containing the name of the columns to import. If this 
argument is \code{NULL}, all variables are imported. Special names 
\code{"allAreas"} and \code{"allLinks"} indicate to the function to import 
all variables for areas or for links. Since version 1.0, values "misc", 
"thermalAvailabilities", "hydroStorage", "hydroStorageMaxPower", "reserve",
"linkCapacity", "mustRun", "thermalModulation" are also accepted and can
replace the corresponding arguments. The list of available variables can be
seen with the command \code{simOptions()$variables}.  Id variables like
\code{area}, \code{link} or \code{timeId} are automatically imported.
Note that \code{select} is *not* taken into account when importing cluster
data.}

\item{mcYears}{Index of the Monte-Carlo years to import. If \code{NULL}, synthetic results
are read, else the specified Monte-Carlo simulations are imported. The 
special value \code{all} tells the function to import all Monte-Carlo
simulations.}

\item{timeStep}{Resolution of the data to import: hourly (default), daily,
weekly, monthly or annual.}

\item{opts}{list of simulation parameters returned by the function
\code{\link{setSimulationPath}}}

\item{parallel}{Should the importation be parallelized ? (See details)}

\item{simplify}{If TRUE and only one type of output is imported then a
data.table is returned. If FALSE, the result will always be a list of class
"antaresData".}

\item{showProgress}{If TRUE the function displays information about the progress of the
importation.}
}
\value{
If \code{simplify = TRUE} and only one type of output is imported
then the result is a data.table.

Else an object of class "antaresDataList" is returned. It is a list of
data.tables, each element representing one type of element (areas, links,
clusters)
}
\description{
\code{readAntares} is a swiss-army-knife function used to read almost every
possible time series of an antares Project at any desired time resolution
(hourly, daily, weekly, monthly or annual).

It was first designed to read
output time series, but it can also read input time series. The input time
series are processed by the function to fit the query of the user (timeStep,
synthetic results or Monte-Carlo simulation, etc.). The few data that are not
read by \code{readAntares} can generally by read with other functions of the
package starting with "read" (\code{\link{readClusterDesc}},
\code{\link{readLayout}}, \code{\link{readBindingConstraints}})
}
\details{
If parameters \code{areas}, \code{links}, \code{clusters} and \code{districts}
are all \code{NULL}, \code{readAntares} will read output for all areas.
By default the function reads synthetic results if they are available.

\code{readAntares} is able to read input time series, but when they are not
stored in output, these time series may have changed since a simulation has
been run. In such a case the function will remind you this danger with a
warning.

When individual Monte-Carlo simulations are read, the function may crash
because of insufficient memory. In such a case, it is necessary to reduce
size of the output. Different strategies are available depending on your
objective:

\itemize{
  \item Use a larger time step (parameter \code{timeStep})
  \item Filter the elements to import (parameters \code{areas},\code{links},
    \code{clusters} and \code{districts})
  \item Select only a few columns (parameter \code{select})
  \item read only a subset of Monte-Carlo simulations (parameter
    \code{mcYears}). For instance one can import a random sample of
    100 simulations with \code{mcYears = sample(simOptions()$mcYears, 100)}
}
}
\section{Parallelization}{


If you import several elements of the same type (areas, links, clusters), you
can use parallelized importation to improve performance. Setting the
parameter \code{parallel = TRUE} is not enough to parallelize the
importation, you also have to install the package
\href{https://CRAN.R-project.org/package=foreach}{foreach}
and a package that provides a parallel backend (for instance the package
\href{https://CRAN.R-project.org/package=doParallel}{doParallel}).

Before running the function with argument \code{parallel=TRUE}, you need to
register your parallel backend. For instance, if you use package "doParallel"
you need to use the function \code{registerDoParallel} once per
session.
}

\examples{
\dontrun{
# Import areas and links separately

areas <- readAntares() # equivalent to readAntares(areas="all")
links <- readAntares(links="all")

# Import areas and links at same time

output <- readAntares(areas = "all", links = "all")

# Add input time series to the object returned by the function
areas <- readAntares(areas = "all", misc = TRUE, reserve = TRUE)

# Get all output for one area

myArea <- sample(simOptions()$areaList, 1)
myArea

myAreaOutput <- readAntares(area = myArea,
                            links = getLinks(myArea, regexpSelect=FALSE),
                            clusters = myArea)

# Or equivalently:
myAreaOutput <- readAntaresAreas(myArea)

# Use parameter "select" to read only some columns.

areas <- readAntares(select = c("LOAD", "OV. COST"))

# Aliases can be used to select frequent groups of columns. use showAliases()
# to view a list of available aliases

areas <- readAntares(select="economy")

}
}
\seealso{
\code{\link{setSimulationPath}}, \code{\link{getAreas}},
\code{\link{getLinks}}, \code{\link{getDistricts}}
}
