% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/01-optimization.R
\name{optimize_theta}
\alias{optimize_theta}
\title{Obtain function information necessary for performing quadrature}
\usage{
optimize_theta(ff, startingvalue, control = default_control(), ...)
}
\arguments{
\item{ff}{A list with three elements:
\itemize{
\item{\code{fn}}{: function taking argument \code{theta} and returning a numeric
value representing the log-posterior at \code{theta}}
\item{\code{gr}}{: function taking argument \code{theta} and returning a numeric
vector representing the gradient of the log-posterior at \code{theta}}
\item{\code{he}}{: function taking argument \code{theta} and returning a numeric
matrix representing the hessian of the log-posterior at \code{theta}}
}
The user may wish to use \code{numDeriv::grad} and/or \code{numDeriv::hessian} to
obtain these. Alternatively, the user may consider the \code{TMB} package. This
list is deliberately formatted to match the output of \code{TMB::MakeADFun}.}

\item{startingvalue}{Value to start the optimization. \code{ff$fn(startingvalue)},
\code{ff$gr(startingvalue)}, and \code{ff$he(startingvalue)} must all return
appropriate values without error.}

\item{control}{A list with elements
\itemize{
\item{\code{method}: }{optimization method to use:
\itemize{
\item{'sparse_trust' (default): }{\code{trustOptim::trust.optim} with \code{method = 'sparse'}}
\item{'SR1' (default): }{\code{trustOptim::trust.optim} with \code{method = 'SR1'}}
\item{'trust': }{\code{trust::trust}}
\item{'BFGS': }{\code{optim(...,method = "BFGS")}}
}
Default is 'sparse_trust'.
}
\item{\code{optcontrol}: }{optional: a list of control parameters to pass to the
internal optimizer you chose. The \code{aghq} package uses sensible defaults.}
}}

\item{...}{Additional arguments to be passed to \code{ff$fn}, \code{ff$gr}, and \code{ff$he}.}
}
\value{
A list with elements
\itemize{
\item{\code{ff}: }{the function list that was provided}
\item{\code{mode}: }{the mode of the log posterior}
\item{\code{hessian}: }{the hessian of the log posterior at the mode}
\item{\code{convergence}: }{specific to the optimizer used, a message indicating whether it converged}
}
}
\description{
This function computes the two pieces of information needed about
the log posterior to do adaptive quadrature: the mode, and the hessian at the mode.
It is designed for use within \code{aghq::aghq}, but is exported in case users
need to debug the optimization process and documented in case users want to
write their own optimizations.
}
\examples{
# Poisson/Exponential example
logfteta <- function(eta,y) {
  sum(y) * eta - (length(y) + 1) * exp(eta) - sum(lgamma(y+1)) + eta
}

y <- rpois(10,5) # Mode should be (sum(y) + 1) / (length(y) + 1)

objfunc <- function(x) logfteta(x,y)
funlist <- list(
  fn = objfunc,
  gr = function(x) numDeriv::grad(objfunc,x),
  he = function(x) numDeriv::hessian(objfunc,x)
)

optimize_theta(funlist,1.5)
optimize_theta(funlist,1.5,control = list(method = "trust"))
optimize_theta(funlist,1.5,control = list(method = "BFGS"))

}
\seealso{
Other quadrature: 
\code{\link{aghq}()},
\code{\link{laplace_approximation}()},
\code{\link{marginal_laplace_tmb}()},
\code{\link{marginal_laplace}()},
\code{\link{normalize_logpost}()},
\code{\link{plot.aghq}()},
\code{\link{print.aghqsummary}()},
\code{\link{print.aghq}()},
\code{\link{print.laplacesummary}()},
\code{\link{print.laplace}()},
\code{\link{summary.aghq}()},
\code{\link{summary.laplace}()}
}
\concept{quadrature}
