% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/seasonal_burden_levels.R
\name{seasonal_burden_levels}
\alias{seasonal_burden_levels}
\title{Compute burden levels from seasonal time series observations of current season.}
\usage{
seasonal_burden_levels(
  tsd,
  family = c("lnorm", "weibull", "exp"),
  season_start = 21,
  season_end = season_start - 1,
  method = c("intensity_levels", "peak_levels"),
  conf_levels = 0.95,
  decay_factor = 0.8,
  disease_threshold = 20,
  n_peak = 6,
  only_current_season = TRUE,
  ...
)
}
\arguments{
\item{tsd}{An object containing time series data with 'time' and 'observation.'}

\item{family}{A character string specifying the family for modeling}

\item{season_start, season_end}{Integers giving the start and end weeks of the seasons to
stratify the observations by.}

\item{method}{A character string specifying the model to be used in the level calculations.
Both model predict the levels of the current series of
observations.
\itemize{
\item \code{intensity_levels}: models the risk compared to what has been observed in previous seasons.
\item \code{peak_levels}: models the risk compared to what has been observed in the \code{n_peak} observations each season.
}}

\item{conf_levels}{A numeric vector specifying the confidence levels for parameter estimates. The values have
to be unique and in ascending order, (i.e. the lowest level is first and highest level is last).
The \code{conf_levels} are specific for each method:
\itemize{
\item for \code{intensity_levels} only specify the highest confidence level e.g.: \code{0.95}, which is the highest intensity
that has been observed in previous seasons.
\item for \code{peak_levels} specify three confidence levels e.g.: \code{c(0.4, 0.9, 0.975)}, which are the three confidence
levels low, medium and high that reflect the peak severity relative to those observed in previous seasons.
}}

\item{decay_factor}{A numeric value between 0 and 1, that specifies the weight applied to previous seasons in level
calculations. It is used as \code{decay_factor}^(number of seasons back), whereby the weight for the most recent season
will be \code{decay_factor}^0 = 1. This parameter allows for a decreasing weight assigned to prior seasons, such that
the influence of older seasons diminishes exponentially.}

\item{disease_threshold}{An integer specifying the threshold for considering a disease outbreak. It defines the per time-step disease threshold that has to be surpassed for the observation to be
included in the level calculations.}

\item{n_peak}{A numeric value specifying the number of peak observations to be selected from each season in the
level calculations. The \code{n_peak} observations have to surpass the \code{disease_threshold} to be included.}

\item{only_current_season}{Should the output only include results for the current season?}

\item{...}{Arguments passed to the \code{fit_percentiles()} function.}
}
\value{
A list containing:
\itemize{
\item 'season': The season that burden levels are calculated for.
\item 'high_conf_level': (only for intensity_level method) The conf_level chosen for the high level.
\item 'conf_levels': (only for peak_level method) The conf_levels chosen to fit the 'low', 'medium', 'high' levels.
\item 'values': A named vector with values for 'very low', 'low', 'medium', 'high' levels.
\item 'par': The fit parameters for the chosen family.
\itemize{
\item par_1:
\itemize{
\item For 'weibull': Shape parameter.
\item For 'lnorm': Mean of the log-transformed observations.
\item For 'exp': Rate parameter.
}
\item 'par_2':
\itemize{
\item For 'weibull': Scale parameter.
\item For 'lnorm': Standard deviation of the log-transformed observations.
\item For 'exp': Not applicable (set to NA).
}
}
\item 'obj_value': The value of the objective function - (negative log-likelihood), which represent the minimized
objective function value from the optimisation. Smaller value equals better optimisation.
\item 'converged': Logical. TRUE if the optimisation converged.
\item 'family': The distribution family used for the optimization.
\itemize{
\item 'weibull': Uses the Weibull distribution for fitting.
\item 'lnorm': Uses the Log-normal distribution for fitting.
\item 'exp': Uses the Exponential distribution for fitting.
\item 'disease_threshold': The input disease threshold, which is also the very low level.
}
}
}
\description{
This function estimates the burden levels of time series observations that are stratified by season.
It uses the previous seasons to estimate the levels of the current season.
The output is results regarding the current season in the time series observations.
NOTE: The data must include data for a complete previous season to make predictions for the current season.
}
\examples{
# Generate random flu season
generate_flu_season <- function(start = 1, end = 1000) {
  random_increasing_obs <- round(sort(runif(24, min = start, max = end)))
  random_decreasing_obs <- round(rev(random_increasing_obs))

  # Generate peak numbers
  add_to_max <- c(50, 100, 200, 100)
  peak <- add_to_max + max(random_increasing_obs)

  # Combine into a single observations sequence
  observations <- c(random_increasing_obs, peak, random_decreasing_obs)

 return(observations)
}

season_1 <- generate_flu_season()
season_2 <- generate_flu_season()

start_date <- as.Date("2022-05-29")
end_date <- as.Date("2024-05-20")

weekly_dates <- seq.Date(from = start_date,
                         to = end_date,
                         by = "week")

tsd_data <- tsd(
  observation = c(season_1, season_2),
  time = as.Date(weekly_dates),
  time_interval = "week"
)

# Print seasonal burden results
seasonal_burden_levels(tsd_data, family = "lnorm")
}
