%% File Name: tamaan.Rd
%% File Version: 0.47

\name{tamaan}
\alias{tamaan}
\alias{summary.tamaan}
\alias{print.tamaan}

%- Also NEED an '\alias' for EACH other topic documented here.
\title{
Wrapper Function for \pkg{TAM} Language
}

\description{
This function is a convenience wrapper function for
several item response models in \pkg{TAM}. Using the
\code{\link{tamaanify}} framework, multidimensional item response models,
latent class models, located and ordered latent class models
and mixture item response models can be estimated.
}

\usage{
tamaan(tammodel, resp, tam.method = NULL, control = list(), doparse=TRUE, ...)

\method{summary}{tamaan}(object,file=NULL,\dots)

\method{print}{tamaan}(x,\dots)
}

%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{tammodel}{
String for specification in \pkg{TAM}, see also \code{\link{tamaanify}}.
}
  \item{resp}{
Dataset with item responses
}
  \item{tam.method}{
One of the \pkg{TAM} methods \code{tam.mml}, \code{tam.mml.2pl}
or \code{tam.mml.3pl}.
}
  \item{control}{
List with control arguments. See \code{\link{tam.mml}}.
}
\item{doparse}{Optional logical indicating whether \code{lavmodel}
should be parsed for \code{DO} statements, see \code{\link{doparse}}.
}
  \item{\dots}{
Further arguments to be passed to
\code{tam.mml}, \code{tam.mml.2pl}
or \code{tam.mml.3pl}.
}
  \item{object}{
Object of class \code{tamaan}
}
\item{file}{
A file name in which the summary output will be written
    }
\item{x}{Object of class \code{tamaan}}
}

%\details{
%%  ~~ If necessary, more details than the description above ~~
%}

\value{
Values generated by \code{tam.mml}, \code{tam.mml.2pl}
or \code{tam.mml.3pl}. In addition, the list also contains the (optional) entries

\item{tamaanify}{Output produced by \code{\link{tamaanify}}}
\item{lcaprobs}{Matrix with probabilities for latent class models}
\item{locs}{Matrix with cluster locations (for \code{TYPE="LOCLCA"})
            }
\item{probs_MIXTURE}{Class probabilities (for \code{TYPE="MIXTURE"})}
\item{moments_MIXTURE}{Distribution parameters (for \code{TYPE="MIXTURE"})}
\item{itempartable_MIXTURE}{Item parameters (for \code{TYPE="MIXTURE"})}
\item{ind_classprobs}{Individual posterior probabilities for
  latent classes (for \code{TYPE="MIXTURE"})}
}


%\references{
%% ~put references to the literature/web site here ~
%}
%\author{
%%  ~~who you are~~
%}
%\note{
%%  ~~further notes~~
%}

%% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{
See \code{\link{tamaanify}} for more details about model specification
using \code{tammodel}.

See \code{\link{tam.mml}} or \code{\link{tam.mml.3pl}}
for more examples.
}


\examples{
#############################################################################
# EXAMPLE 1: Examples dichotomous data data.read
#############################################################################

library(sirt)
data(data.read,package="sirt")
dat <- data.read

#*********************************************************************
#*** Model 1: Rasch model

tammodel <- "
LAVAAN MODEL:
  F1 =~ A1__C4
  F1 ~~ F1
ITEM TYPE:
  ALL(Rasch);
    "
# estimate model
mod1 <- TAM::tamaan( tammodel , resp=dat)
summary(mod1)

\dontrun{
#*********************************************************************
#*** Model 2: 2PL model with some selected items

tammodel <- "
LAVAAN MODEL:
  F1 =~ A1__B1 + B3 + C1__C3
  F1 ~~ F1
    "
mod2 <- TAM::tamaan( tammodel , resp=dat)
summary(mod2)

#*********************************************************************
#*** Model 3: Multidimensional IRT model

tammodel <- "
LAVAAN MODEL:
  G  =~ A1__C4
  F1 =~ A1__B4
  F2 =~ C1__C4
  F1 ~~ F2
  # specify fixed entries in covariance matrix
  F1 ~~ 1*F1
  F2 ~~ 1*F2
  G  ~~ 0*F1
  G  ~~ 0.3*F2
  G  ~~ 0.7*G
    "
mod3 <- TAM::tamaan( tammodel , resp=dat , control=list(maxiter=30))
summary(mod3)

#*********************************************************************
#*** Model 4: Some linear constraints for item slopes and intercepts

tammodel <- "
LAVAAN MODEL:
  F =~ lam1__lam10*A1__C2
  F =~ 0.78*C3
  F ~~ F
  A1 | a1*t1
  A2 | a2*t1
  A3 | a3*t1
  A4 | a4*t1
  B1 | b1*t1
  B2 | b2*t1
  B3 | b3*t1
  C1 | t1
MODEL CONSTRAINT:
  # defined parameters
  # only linear combinations are permitted
  b2 == 1.3*b1 + (-0.6)*b3
  a1 == q1
  a2 == q2 + t
  a3 == q1 + 2*t
  a4 == q2 + 3*t
  # linear constraints for loadings
  lam2 == 1.1*lam1
  lam3 == 0.9*lam1 + (-.1)*lam0
  lam8 == lam0
  lam9 == lam0
    "
mod4 <- TAM::tamaan( tammodel , resp=dat , control=list(maxiter=5) )
summary(mod4)

#*********************************************************************
#*** Model 5: Latent class analysis with three classes

tammodel <- "
ANALYSIS:
  TYPE=LCA;
  NCLASSES(3);   # 3 classes
  NSTARTS(5,20); # 5 random starts with 20 iterations
LAVAAN MODEL:
  F =~ A1__C4
    "
mod5 <- TAM::tamaan( tammodel , resp=dat , control=list(maxiter=100)  )
summary(mod5)

#*********************************************************************
#*** Model 6: Ordered latent class analysis with three classes

tammodel <- "
ANALYSIS:
  TYPE=OLCA;
  NCLASSES(3);    # 3 classes
  NSTARTS(20,40); # 20 random starts with 40 iterations
LAVAAN MODEL:
  F =~ A1__C4
    "
mod6 <- TAM::tamaan( tammodel , dat )
summary(mod6)

#*********************************************************************
#*** Model 7: Unidimensional located latent class model with three classes

tammodel <- "
ANALYSIS:
  TYPE=LOCLCA;
  NCLASSES(3)
  NSTARTS(10,40)
LAVAAN MODEL:
  F =~ A1__C4
  B2 | 0*t1
    "
mod7 <- TAM::tamaan( tammodel , resp=dat)
summary(mod7)

#*********************************************************************
#*** Model 8: Two-dimensional located latent class analysis with some
#             priors and equality constraints among thresholds

tammodel <- "
ANALYSIS:
  TYPE=LOCLCA;
  NCLASSES(4);
  NSTARTS(10,20);
LAVAAN MODEL:
  AB =~ A1__B4
  C =~ C1__C4
  A1 | a1diff*t1
  B2 | 0*t1
  C2 | 0*t1
  B1 | a1diff*t1
MODEL PRIOR:
  # prior distributions for cluster locations
  DO2(1,4,1,1,2,1)
    Cl\%1_Dim\%2 ~ N(0,2);
  DOEND
    "
# estimate model
mod8 <- TAM::tamaan( tammodel , resp=dat )
summary(mod8)

#*********************************************************************
#*** Model 9: Two-dimensional model with constraints on parameters

tammodel <- "
LAVAAN MODEL:
  FA =~ A1+b*A2+A3+d*A4
  FB =~ B1+b*B2+B3+d*B4
  FA ~~ 1*FA
  FA ~~ FB
  FB ~~ 1*FB
  A1 | c*t1
  B1 | c*t1
  A2 | .7*t1
    "
# estimate model
mod9 <- TAM::tamaan( tammodel , resp=dat , control=list(maxiter=30) )
summary(mod9)

#############################################################################
# EXAMPLE 2: Examples polytomous data | data.Students
#############################################################################

library(CDM)
data( data.Students , package="CDM")
dat <- data.Students[,3:13]
  ##   > colnames(dat)
  ##    [1] "act1" "act2" "act3" "act4" "act5" "sc1"  "sc2"  "sc3"  "sc4"  "mj1"  "mj2"

#*********************************************************************
#*** Model 1: Two-dimensional generalized partial credit model

tammodel <- "
LAVAAN MODEL:
  FA =~ act1__act5
  FS =~ sc1__sc4
  FA ~~ 1*FA
  FS ~~ 1*FS
  FA ~~ FS
    "
# estimate model
mod1 <- TAM::tamaan( tammodel , dat , control=list(maxiter=10)  )
summary(mod1)

#*********************************************************************
#*** Model 2: Two-dimensional model, some constraints

tammodel <- "
LAVAAN MODEL:
  FA =~ a1__a4*act1__act4 + 0.89*act5
  FS =~ 1*sc1 + sc2__sc4
  FA ~~ FA
  FS ~~ FS
  FA ~~ FS
  # some equality constraints
  act1 + act3 | a13_t1 * t1
  act1 + act3 | a13_t2 * t2
    "
# only create design matrices with tamaanify
mod2 <- TAM::tamaanify( tammodel , dat  )
mod2$lavpartable
# estimate model (only few iterations as a test)
mod2 <- TAM::tamaan( tammodel , dat , control=list(maxiter=10)  )
summary(mod2)

#*********************************************************************
#*** Model 3: Two-dimensional model, some more linear constraints

tammodel <- "
LAVAAN MODEL:
  FA =~ a1__a5*act1__act5
  FS =~ b1__b4*sc1__sc4
  FA ~~ 1*FA
  FA ~~ FS
  FS ~~ 1*FS
  act1 + act3 | a13_t1 * t1
  act1 + act3 | a13_t2 * t2
MODEL CONSTRAINT:
  a1 == q0
  a2 == q0
  a3 == q0    + q1
  a4 ==    q2
  a5 ==    q2 + q1
    "
# estimate
mod3 <- TAM::tamaan( tammodel , dat , control=list(maxiter=300 )  )
summary(mod3)

#*********************************************************************
#*** Model 4: Latent class analysis with three latent classes

tammodel <- "
ANALYSIS:
  TYPE=LCA;
  NCLASSES(3);    # 3 classes
  NSTARTS(10,30); # 10 random starts with 30 iterations
LAVAAN MODEL:
  F =~ act1__act5
    "
# estimate model
mod4 <- TAM::tamaan( tammodel , resp=dat)
summary(mod4)

#*********************************************************************
#*** Model 5: Partial credit model with "PCM2" parametrization

# select data
dat1 <- dat[, paste0("act",1:5) ]
# specify tamaan model
tammodel <- "
  LAVAAN MODEL:
    F =~ act1__act5
    F ~~ F
    # use DO statement as shortages
    DO(1,5,1)
      act\% | b\%_1 * t1
      act\% | b\%_2 * t2
    DOEND
  MODEL CONSTRAINT:
    DO(1,5,1)
      b\%_1 == delta\% + tau\%_1
      b\%_2 == 2*delta\%
    DOEND
  ITEM TYPE:
    ALL(PCM)
  "
# estimate model
mod5 <- TAM::tamaan( tammodel , dat1 )
summary(mod5)
# compare with PCM2 parametrization in tam.mml
mod5b <- TAM::tam.mml( dat1 , irtmodel="PCM2" )
summary(mod5b)

#*********************************************************************
#*** Model 6: Rating scale model

# select data
dat1 <- dat[, paste0("sc",1:4) ]
psych::describe(dat1)

# specify tamaan model
tammodel <- "
  LAVAAN MODEL:
    F =~ sc1__sc4
    F ~~ F
    # use DO statement as shortages
    DO(1,4,1)
      sc\% | b\%_1 * t1
      sc\% | b\%_2 * t2
      sc\% | b\%_3 * t3
    DOEND
  MODEL CONSTRAINT:
    DO(1,4,1)
      b\%_1 == delta\% + step1
      b\%_2 == 2*delta\% + step1 + step2
      b\%_3 == 3*delta\%
    DOEND
  ITEM TYPE:
    ALL(PCM)
  "
# estimate model
mod6 <- TAM::tamaan( tammodel , dat1 )
summary(mod6)
# compare with RSM in tam.mml
mod6b <- TAM::tam.mml( dat1 , irtmodel="RSM" )
summary(mod6b)

#*********************************************************************
#*** Model 7: Partial credit model with Fourier basis for
#             item intercepts (Thissen, Cai & Bock, 2010)
# see ?tamaanify manual

# define tamaan model
tammodel <- "
LAVAAN MODEL:
   mj =~ mj1__mj4
   mj ~~ 1*mj
ITEM TYPE:
  mj1(PCM,2)
  mj2(PCM,3)
  mj3(PCM)
  mj4(PCM,1)
   "
# estimate model
mod7 <- TAM::tamaan( tammodel , dat )
summary(mod7)
# -> This function can also be applied for the generalized partial credit
#    model (GPCM).

#############################################################################
# EXAMPLE 3: Rasch model and mixture Rasch model (Geiser & Eid, 2010)
#############################################################################

data(data.geiser, package="TAM")
dat <- data.geiser

#*********************************************************************
#*** Model 1: Rasch model
tammodel <- "
LAVAAN MODEL:
  F =~ mrt1__mrt6
  F ~~ F
ITEM TYPE:
  ALL(Rasch);
    "
mod1 <- TAM::tamaan( tammodel , resp=dat  )
summary(mod1)

#*********************************************************************
#*** Model 2: Mixed Rasch model with two classes
tammodel <- "
ANALYSIS:
  TYPE=MIXTURE ;
  NCLASSES(2);
  NSTARTS(20,25);
LAVAAN MODEL:
  F =~ mrt1__mrt6
  F ~~ F
ITEM TYPE:
  ALL(Rasch);
    "
mod2 <- TAM::tamaan( tammodel , resp=dat  )
summary(mod2)

# plot item parameters
ipars <- mod2$itempartable_MIXTURE[ 1:6 , ]
plot( 1:6 , ipars[,3] , type="o" , ylim=c(-3,2) , pch=16 ,
        xlab="Item" , ylab="Item difficulty")
lines( 1:6 , ipars[,4] , type="l", col=2 , lty=2)
points( 1:6 , ipars[,4] ,  col=2 , pch=2)

# extract individual posterior distribution
post2 <- IRT.posterior(mod2)
str(post2)
# num [1:519, 1:30] 0.000105 0.000105 0.000105 0.000105 0.000105 ...
# - attr(*, "theta")= num [1:30, 1:30] 1 0 0 0 0 0 0 0 0 0 ...
# - attr(*, "prob.theta")= num [1:30, 1] 1.21e-05 2.20e-04 2.29e-03 1.37e-02 4.68e-02 ...
# - attr(*, "G")= num 1

# There are 2 classes and 15 theta grid points for each class
# The loadings of the theta grid on items are as follows
mod2$E[1,2,,"mrt1_F_load_Cl1"]
mod2$E[1,2,,"mrt1_F_load_Cl2"]

# compute individual posterior probability for class 1 (first 15 columns)
round( rowSums( post2[ , 1:15] ) , 3 )
# columns 16 to 30 refer to class 2

}
}


% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{Model specification}
\keyword{TAM language}% __ONLY ONE__ keyword per line
