library("lubridate")

context("sun_times")

test_that("sunrise_time", {

  expect_equal(
    tz(sunrise_time()), "UTC"
  )
  expect_equal(
    tz(sunrise_time(geocode = data.frame(lon = 24.93838, lat = 60.16986,
                                         address = "Helsinki, Finland"))), "UTC"
  )
  expect_equal(
    tz(sunrise_time(ymd("2016-04-17", tz = "UTC"),
                    geocode = data.frame(lon = 24.93838, lat = 60.16986,
                                         address = "Helsinki, Finland"),
                    twilight = "none")), "UTC"
  )
  expect_equal(
    tz(sunrise_time(ymd("2016-04-17", tz = "UTC"), tz = "UTC",
                    geocode = data.frame(lon = 24.93838, lat = 60.16986,
                                         address = "Helsinki, Finland"),
                    twilight = "none")), "UTC"
  )
  expect_equal(
    tz(sunrise_time(ymd_hms("2016-04-17 12:00:20", tz = "UTC"), tz = "UTC",
                    geocode = data.frame(lon = 24.93838, lat = 60.16986,
                                         address = "Helsinki, Finland"),
                    twilight = "none")), "UTC"
  )
  expect_equal(
    tz(sunrise_time(ymd_hms("2016-04-17 12:00:20", tz = "UTC"), tz = "EET",
                    geocode = data.frame(lon = 24.93838, lat = 60.16986,
                                         address = "Helsinki, Finland"),
                    twilight = "none")), "EET"
  )
  expect_equal(
    as.duration(
      sunrise_time(ymd("2016-04-17", tz = "UTC"), tz = "EET",
                   geocode = data.frame(lon = 24.93838, lat = 60.16986,
                                        address = "Helsinki, Finland"),
                   twilight = "none") %--%
        sunrise_time(ymd("2016-04-17", tz = "UTC"), tz = "UTC",
                     geocode = data.frame(lon = 24.93838, lat = 60.16986,
                                          address = "Helsinki, Finland"),
                     twilight = "none")), as.duration(seconds(0))
  )
  expect_lt(
    abs(as.numeric(sunrise_time(ymd("2016-04-17", tz = "UTC"), tz = "UTC",
                                geocode = data.frame(lon = 24.93838, lat = 60.16986,
                                                     address = "Helsinki, Finland"),
                                twilight = "none") -
                     ymd_hms("2016-04-17 03:02:32", tz = "UTC"))), 1
  )
  expect_lt(
    abs(as.numeric(sunrise_time(ymd("2016-04-17", tz = "UTC"), tz = "UTC",
                                geocode = data.frame(lon = 25.46508, lat = 65.01209,
                                                     address = "Oulu, Finland"),
                                twilight = "none") -
                     ymd_hms("2016-04-17 02:41:40", tz = "UTC"))), 1
  )
  expect_lt(
    abs(as.numeric(sunrise_time(ymd("2016-04-17", tz = "UTC"), tz = "UTC",
                                geocode = data.frame(lon = 27.02853, lat = 69.90905,
                                                     adress = "Utsjoki, Finland"),
                                twilight = "none") -
                     ymd_hms("2016-04-17 02:06:34", tz = "UTC"))), 1
  )
  expect_lt(
    abs(as.numeric(sunrise_time(ymd("2016-04-17", tz = "UTC"), tz = "UTC",
                                geocode = data.frame(lon = -68.30295, lat = -54.80191,
                                                     address = "Ushuaia, Argentina"),
                                twilight = "none") -
                     ymd_hms("2016-04-17 11:34:59", tz = "UTC"))), 1
  )
  expect_lt(
    as.numeric(sunrise_time(ymd("2016-04-17", tz = "UTC"), tz = "UTC",
                            geocode = data.frame(lon = 23.67027, lat = 77.5536),
                            twilight = "none") -
                 ymd_hms("2016-04-17 00:28:16", tz = "UTC")), 1
  )
  expect_lt(
    as.numeric(sunrise_time(ymd("2016-04-21", tz = "UTC"), tz = "UTC",
                            geocode = data.frame(lon = 23.67027, lat = 77.5536),
                            twilight = "none") -
                 ymd_hms("2016-04-20 23:18:52", tz = "UTC")), 1
  )
  expect_true(
    is.na(sunrise_time(ymd("2016-04-23", tz = "UTC"), tz = "UTC",
                       geocode = data.frame(lon = 23.67027, lat = 77.5536),
                       twilight = "none"))
  )
})

test_that("noon_time", {

  expect_equal(
    tz(noon_time()), "UTC"
  )
  expect_equal(
    tz(noon_time(geocode = data.frame(lon = 24.93838, lat = 60.16986,
                                      address = "Helsinki, Finland"))), "UTC"
  )
  expect_equal(
    tz(noon_time(ymd("2016-04-17", tz = "UTC"),
                    geocode = data.frame(lon = 24.93838, lat = 60.16986,
                                         address = "Helsinki, Finland"),
                    twilight = "none")), "UTC"
  )
  expect_equal(
    tz(noon_time(ymd("2016-04-17", tz = "UTC"), tz = "UTC",
                    geocode = data.frame(lon = 24.93838, lat = 60.16986,
                                         address = "Helsinki, Finland"),
                    twilight = "none")), "UTC"
  )
  expect_equal(
    tz(noon_time(ymd_hms("2016-04-17 12:00:20", tz = "UTC"), tz = "UTC",
                    geocode = data.frame(lon = 24.93838, lat = 60.16986,
                                         address = "Helsinki, Finland"),
                    twilight = "none")), "UTC"
  )
  expect_equal(
    tz(noon_time(ymd_hms("2016-04-17 12:00:20", tz = "UTC"), tz = "EET",
                    geocode = data.frame(lon = 24.93838, lat = 60.16986,
                                         address = "Helsinki, Finland"),
                    twilight = "none")), "EET"
  )
  expect_equal(
    as.duration(
      noon_time(ymd("2016-04-17", tz = "UTC"), tz = "EET",
                   geocode = data.frame(lon = 24.93838, lat = 60.16986,
                                        address = "Helsinki, Finland"),
                   twilight = "none") %--%
        noon_time(ymd("2016-04-17", tz = "UTC"), tz = "UTC",
                     geocode = data.frame(lon = 24.93838, lat = 60.16986,
                                          address = "Helsinki, Finland"),
                     twilight = "none")), as.duration(seconds(0))
  )
  expect_lt(
    abs(as.numeric(noon_time(ymd("2016-04-17", tz = "UTC"), tz = "UTC",
                                geocode = data.frame(lon = 24.93838, lat = 60.16986,
                                                     address = "Helsinki, Finland"),
                                twilight = "none") -
                     ymd_hms("2016-04-17 10:19:42", tz = "UTC"))), 1
  )
  expect_lt(
    abs(as.numeric(noon_time(ymd("2016-04-17", tz = "UTC"), tz = "UTC",
                                geocode = data.frame(lon = 25.46508, lat = 65.01209,
                                                     address = "Oulu, Finland"),
                                twilight = "none") -
                     ymd_hms("2016-04-17 10::17:36", tz = "UTC"))), 1
  )
  expect_lt(
    abs(as.numeric(noon_time(ymd("2016-04-17", tz = "UTC"), tz = "UTC",
                                geocode = data.frame(lon = 27.02853, lat = 69.90905,
                                                     adress = "Utsjoki, Finland"),
                                twilight = "none") -
                     ymd_hms("2016-04-17 10:11:20", tz = "UTC"))), 1
  )
  expect_lt(
    abs(as.numeric(noon_time(ymd("2016-04-17", tz = "UTC"), tz = "UTC",
                                geocode = data.frame(lon = -68.30295, lat = -54.80191,
                                                     address = "Ushuaia, Argentina"),
                                twilight = "none") -
                     ymd_hms("2016-04-17 16:32:40", tz = "UTC"))), 1
  )
  expect_lt(
    as.numeric(noon_time(ymd("2016-04-17", tz = "UTC"), tz = "UTC",
                            geocode = data.frame(lon = 23.67027, lat = 77.5536),
                            twilight = "none") -
                 ymd_hms("2016-04-17 10:24:46", tz = "UTC")), 1
  )
  expect_lt(
    as.numeric(noon_time(ymd("2016-04-21", tz = "UTC"), tz = "UTC",
                            geocode = data.frame(lon = 23.67027, lat = 77.5536),
                            twilight = "none") -
                 ymd_hms("2016-04-20 10:23:56", tz = "UTC")), seconds(1.5)
               )
})

test_that("sunset_time", {
  expect_equal(
    tz(sunset_time()), "UTC"
  )
  expect_equal(
    tz(sunset_time(geocode = data.frame(lon = 24.93838, lat = 60.16986,
                                      address = "Helsinki, Finland"))), "UTC"
  )
  expect_equal(
    tz(sunset_time(ymd("2016-04-17", tz = "UTC"),
                    geocode = data.frame(lon = 24.93838, lat = 60.16986,
                                         address = "Helsinki, Finland"),
                    twilight = "none")), "UTC"
  )
  expect_equal(
    tz(sunset_time(ymd("2016-04-17", tz = "UTC"), tz = "UTC",
                    geocode = data.frame(lon = 24.93838, lat = 60.16986,
                                         address = "Helsinki, Finland"),
                    twilight = "none")), "UTC"
  )
  expect_equal(
    tz(sunset_time(ymd_hms("2016-04-17 12:00:20", tz = "UTC"), tz = "UTC",
                    geocode = data.frame(lon = 24.93838, lat = 60.16986,
                                         address = "Helsinki, Finland"),
                    twilight = "none")), "UTC"
  )
  expect_equal(
    tz(sunset_time(ymd_hms("2016-04-17 12:00:20", tz = "UTC"), tz = "EET",
                    geocode = data.frame(lon = 24.93838, lat = 60.16986,
                                         address = "Helsinki, Finland"),
                    twilight = "none")), "EET"
  )
  expect_equal(
    as.duration(
      sunset_time(ymd("2016-04-17", tz = "UTC"), tz = "EET",
                   geocode = data.frame(lon = 24.93838, lat = 60.16986,
                                        address = "Helsinki, Finland"),
                   twilight = "none") %--%
        sunset_time(ymd("2016-04-17", tz = "UTC"), tz = "UTC",
                     geocode = data.frame(lon = 24.93838, lat = 60.16986,
                                          address = "Helsinki, Finland"),
                     twilight = "none")), as.duration(seconds(0))
  )
  expect_lt(
    as.numeric(sunset_time(ymd("2016-04-17", tz = "UTC"), tz = "UTC",
                           geocode = data.frame(lon = 24.93838, lat = 60.16986),
                           #                            geocode = geocode("Helsinki, Finland"),
                           twilight = "none") -
                 ymd_hms("2016-04-17 17:36:52", tz = "UTC")), 1
  )
  expect_lt(
    as.numeric(sunset_time(ymd("2016-04-17", tz = "UTC"), tz = "UTC",
                           geocode = data.frame(lon = 25.46508, lat = 65.01209),
                           #                           geocode = geocode("Oulu, Finland"),
                           twilight = "none") -
                 ymd_hms("2016-04-17 17:53:31", tz = "UTC")), 1
  )
  expect_lt(
    as.numeric(sunset_time(ymd("2016-04-17", tz = "UTC"), tz = "UTC",
                           geocode = data.frame(lon = 27.02853, lat = 69.90905),
                           #                          geocode = geocode("Utsjoki, Finland"),
                           twilight = "none") -
                 ymd_hms("2016-04-17 18:16:06", tz = "UTC")), 1
  )
})

test_that("sunrise_time_vectorized", {
  expect_equal(
    length(sunrise_time(ymd("2016-04-17", tz = "UTC") + days(0:5),
                        geocode = data.frame(lon = 24.93838, lat = 60.16986),
                        #                       geocode = geocode("Helsinki, Finland"),
                        twilight = "none")), 6)
  expect_equal(
    sunrise_time(ymd("2016-04-17", tz = "UTC") + days(0:5), tz = "UTC",
                 geocode = data.frame(lon = 24.93838, lat = 60.16986),
                 #                       geocode = geocode("Helsinki, Finland"),
                 twilight = "none")[1],
    sunrise_time(ymd("2016-04-17", tz = "UTC"),
                 geocode = data.frame(lon = 24.93838, lat = 60.16986), tz = "UTC",
                 #                       geocode = geocode("Helsinki, Finland"),
                 twilight = "none"))
  expect_equal(
    length(sunrise_time(ymd("2016-04-20", tz = "UTC") + days(0:5),
                        geocode = data.frame(lon = 23.67027, lat = 77.5536),
                        #                           geocode = geocode("Ushuaia, Argentina"),
                        twilight = "none")), 6)
  expect_equal(
    sunrise_time(ymd("2016-04-17", tz = "UTC") + days(0:5), tz = "UTC",
                 geocode = data.frame(lon = 23.67027, lat = 77.5536),
                 #                           geocode = geocode("Ushuaia, Argentina"),
                 twilight = "none")[1],
    sunrise_time(ymd("2016-04-17", tz = "UTC"), tz = "UTC",
                 geocode = data.frame(lon = 23.67027, lat = 77.5536),
                 #                           geocode = geocode("Ushuaia, Argentina"),
                 twilight = "none"))
  expect_equal(
    sunrise_time(ymd("2016-04-20", tz = "UTC") + days(0:5), tz = "UTC",
                 geocode = data.frame(lon = 23.67027, lat = 77.5536),
                 #                           geocode = geocode("Ushuaia, Argentina"),
                 twilight = "none")[3],
    sunrise_time(ymd("2016-04-22", tz = "UTC"), tz = "UTC",
                 geocode = data.frame(lon = 23.67027, lat = 77.5536),
                 #                           geocode = geocode("Ushuaia, Argentina"),
                 twilight = "none"))
})

test_that("sunset_time_vectorized", {
  expect_equal(
    length(sunset_time(ymd("2016-04-17", tz = "UTC") + days(0:5), tz = "UTC",
                       geocode = data.frame(lon = 24.93838, lat = 60.16986),
                       #                     geocode = geocode("Helsinki, Finland"),
                       twilight = "none")), 6)
})

test_that("daylength", {

  expect_equal(day_length(now(tzone = "UTC")), day_length(today(tzone = "UTC"))) # is conversion o.k.?
  expect_equal(day_length(ymd("2014-12-21"),
                          geocode = data.frame(lat = 85, lon = 0)),
               0)
  expect_equal(day_length(ymd("2014-06-21"),
                          geocode = data.frame(lat = 85, lon = 0)),
               24)
  expect_equal(night_length(ymd("2014-12-21"),
                            geocode = data.frame(lat = 85, lon = 0)),
               24)
  expect_equal(night_length(ymd("2014-06-21"),
                            geocode = data.frame(lat = 85, lon = 0)),
               0)
  expect_equal(round(day_length(ymd("2014-12-21"),
                                geocode = data.frame(lat = 0, lon = 0)), 3),
               12.121)
  expect_equal(round(day_length(ymd("2014-06-21"),
                                geocode = data.frame(lat = 0, lon = 0)), 3),
               12.121)
  expect_equal(round(night_length(ymd("2014-12-21"),
                                  geocode = data.frame(lat = 0, lon = 0)), 3),
               11.879)
  expect_equal(round(night_length(ymd("2014-06-21"),
                                  geocode = data.frame(lat = 0, lon = 0)), 3),
               11.879)

  expect_equal(round(day_length(ymd("2014-03-20"),
                                geocode = data.frame(lat = 45, lon = 0)), 3),
               12.147)
  expect_equal(round(day_length(ymd("2014-09-22"),
                                geocode = data.frame(lat = 45, lon = 0)), 3),
               12.188)
  expect_equal(round(night_length(ymd("2014-03-20"),
                                  geocode = data.frame(lat = 45, lon = 0)), 3),
               11.853)
  expect_equal(round(night_length(ymd("2014-09-22"),
                                  geocode = data.frame(lat = 45, lon = 0)), 3),
               11.812)

  expect_gt(round(day_length(ymd("2014-03-21"),
                             geocode = data.frame(lat = 45, lon = 0),
                             twilight = "civil"), 2), 12)
  expect_gt(round(day_length(ymd("2014-09-21"),
                             geocode = data.frame(lat = 45, lon = 0),
                             twilight = "civil"), 2), 12)
  expect_lt(round(night_length(ymd("2014-03-21"),
                               geocode = data.frame(lat = 45, lon = 0),
                               twilight = "civil"), 2), 12)
  expect_lt(round(night_length(ymd("2014-09-21"),
                               geocode = data.frame(lat = 45, lon = 0),
                               twilight = "civil"), 2), 12)

  expect_gt(round(day_length(ymd("2014-03-21"),
                             geocode = data.frame(lat = 45, lon = 0),
                             twilight = "nautical"), 2), 12)
  expect_gt(round(day_length(ymd("2014-09-21"),
                             geocode = data.frame(lat = 45, lon = 0),
                             twilight = "nautical"), 2), 12)
  expect_lt(round(night_length(ymd("2014-03-21"),
                               geocode = data.frame(lat = 45, lon = 0),
                               twilight = "nautical"), 2), 12)
  expect_lt(round(night_length(ymd("2014-09-21"),
                               geocode = data.frame(lat = 45, lon = 0),
                               twilight = "nautical"), 2), 12)

  expect_gt(round(day_length(ymd("2014-03-21"),
                             geocode = data.frame(lat = 45, lon = 0),
                             twilight = "astronomical"), 2), 12)
  expect_gt(round(day_length(ymd("2014-09-21"),
                             geocode = data.frame(lat = 45, lon = 0),
                             twilight = "astronomical"), 2), 12)
  expect_lt(round(night_length(ymd("2014-03-21"),
                               geocode = data.frame(lat = 45, lon = 0),
                               twilight = "astronomical"), 2), 12)
  expect_lt(round(night_length(ymd("2014-09-21"),
                               geocode = data.frame(lat = 45, lon = 0),
                               twilight = "astronomical"), 2), 12)

  expect_gt(round(day_length(ymd("2014-03-21"),
                             geocode = data.frame(lat = 45, lon = 0),
                             twilight = -1), 2), 12)
  expect_gt(round(day_length(ymd("2014-09-21"),
                             geocode = data.frame(lat = 45, lon = 0),
                             twilight = -1), 2), 12)
  expect_lt(round(night_length(ymd("2014-03-21"),
                               geocode = data.frame(lat = 45, lon = 0),
                               twilight = -1), 2), 12)
  expect_lt(round(night_length(ymd("2014-09-21"),
                               geocode = data.frame(lat = 45, lon = 0),
                               twilight = -1), 2), 12)

  expect_lt(round(day_length(ymd("2014-03-21"),
                             geocode = data.frame(lat = 45, lon = 0),
                             twilight = +1), 2), 12)
  expect_lt(round(day_length(ymd("2014-09-21"),
                             geocode = data.frame(lat = 45, lon = 0),
                             twilight = +1), 2), 12)
  expect_gt(round(night_length(ymd("2014-03-21"),
                               geocode = data.frame(lat = 45, lon = 0),
                               twilight = +1), 2), 12)
  expect_gt(round(night_length(ymd("2014-09-21"),
                               geocode = data.frame(lat = 45, lon = 0),
                               twilight = +1), 2), 12)

  expect_equal(night_length(ymd("2014-09-21"),
                            geocode = data.frame(lat = 45, lon = 0),
                            twilight = "none"),
               night_length(ymd("2014-09-21"),
                            geocode = data.frame(lat = 45, lon = 0),
                            twilight = 0))
  expect_equal(night_length(ymd("2014-09-21"),
                            geocode = data.frame(lat = 45, lon = 0),
                            twilight = "civil"),
               night_length(ymd("2014-09-21"),
                            geocode = data.frame(lat = 45, lon = 0),
                            twilight = -6))
  expect_equal(night_length(ymd("2014-09-21"),
                            geocode = data.frame(lat = 45, lon = 0),
                            twilight = "civil"),
               night_length(ymd("2014-09-21"),
                            geocode = data.frame(lat = 45, lon = 0),
                            twilight = c(-6, -6)))

  testthat::expect_true(is_daytime(ymd_hm("2014-03-21 12:00"),
                                   geocode = data.frame(lat = 45, lon = 0)))
  testthat::expect_false(is_daytime(ymd_hm("2014-03-21 23:59"),
                                    geocode = data.frame(lat = 45, lon = 0)))

  expect_warning(day_length(ymd("2014-03-21"),
                          geocode = data.frame(lat = 45, lon = 0),
                          twilight = rep("none", 2)))
  expect_warning(day_length(ymd("2014-03-21"),
                            geocode = data.frame(lat = 45, lon = 0),
                            twilight = "bad"))
  expect_warning(day_length(ymd("2014-03-21"),
                          geocode = data.frame(lat = 45, lon = 0),
                          twilight = +91))
  expect_warning(day_length(ymd("2014-03-21"),
                          geocode = data.frame(lat = 45, lon = 0),
                          twilight = -91))
  expect_warning(day_length(ymd("2014-03-21"),
                            geocode = data.frame(lat = 45, lon = 0),
                            twilight = NA))
  expect_warning(day_length(ymd("2014-03-21"),
                            geocode = data.frame(lat = 45, lon = 0),
                            twilight = NULL))
  expect_error(day_length(ymd("2014-03-21"),
                            geocode = data.frame(lat = 45, lon = 0),
                            twilight = rep(0, 3)))
  expect_warning(day_length(ymd("2014-03-21"),
                          geocode = data.frame(lat = 45, lon = 0),
                          twilight = numeric()))
  expect_warning(day_length(ymd("2014-03-21"),
                          geocode = data.frame(lat = 45, lon = 0),
                          twilight = character()))

})

