#' Create an edge table file for Cytoscape
#' 
#' The returned data frame can be saved as a .csv file. Then, in Cytoscape use 
#' File -> Import -> Network -> File. Select the .csv file containing the data
#' frame generated by this function. There will be a pop-up window. The source, 
#' interaction, and target columns should automatically be identified. Click OK.
#' @param g A 'network_plot' object. See \code{\link{plot_network}}.
#' @return A data frame containing an edge table that can be saved as a .csv file 
#' to be used in Cytoscape. 
#' @export
#' @examples 
#' nw <- random_network(10)
#' g <- plot(nw)
#' nw_plot_cytoscape <- create_cytoscape_file(g)
#' \donttest{
#' # Save the edge table in a .csv file to be used in cytoscape.
#' write.table(nw_plot_cytoscape, file.path(tempdir(), "file_name.csv"), 
#'             sep = ",", row.names = FALSE, col.names = TRUE, quote = FALSE)
#' }
create_cytoscape_file <- function(g) {
  
  edges <- do.call(rbind, 
                   lapply(lapply(attr(igraph::E(g$graph), "vnames"), 
                                 strsplit, split = "\\|"), 
                          function(x) as.numeric(unlist(x))))
  weights <- g$edge.width
  
  #Determine number of rows needed for .csv file.
  n_adj <- nrow(edges)
  
  #Initialize table.
  edge_table <- data.frame(source = edges[, 1],
                           interaction = rep("gg", n_adj),
                           target = edges[, 2], 
                           scores = weights)
  
  return(edge_table)
}
