% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SVEMnet.R
\name{SVEMnet}
\alias{SVEMnet}
\title{Fit an SVEMnet Model (with optional relaxed elastic net)}
\usage{
SVEMnet(
  formula,
  data,
  nBoot = 200,
  glmnet_alpha = c(0.5, 1),
  weight_scheme = c("SVEM", "FRW_plain", "Identity"),
  objective = c("auto", "wAIC", "wBIC", "wSSE"),
  auto_ratio_cutoff = 1.3,
  relaxed = TRUE,
  response = NULL,
  unseen = c("warn_na", "error"),
  family = c("gaussian", "binomial"),
  ...
)
}
\arguments{
\item{formula}{A formula specifying the model to be fitted, OR a bigexp_spec
created by \code{bigexp_terms()}.}

\item{data}{A data frame containing the variables in the model.}

\item{nBoot}{Number of bootstrap iterations (default 200).}

\item{glmnet_alpha}{Elastic Net mixing parameter(s). May be a vector with
entries in the range between 0 and 1, inclusive, where alpha = 1 is Lasso
and alpha = 0 is Ridge. Defaults to \code{c(0.5, 1)}.}

\item{weight_scheme}{Character; weighting scheme used to generate bootstrap
training and validation weights (default \code{"SVEM"}).
One of:
\itemize{
\item \code{"SVEM"}: Self-Validated Ensemble Model weights. For each
bootstrap, independent \eqn{U_i \sim \mathrm{Uniform}(0,1)} are drawn
and converted to anti-correlated FRW copies
\eqn{w_i^{\mathrm{train}} = -\log U_i} and
\eqn{w_i^{\mathrm{valid}} = -\log(1-U_i)}, each rescaled to have mean 1.
This is the default and implements the SVEM scheme of Lemkus et al.
\item \code{"FRW_plain"}: Fractional–random–weight (FRW) regression
without self-validation. A single FRW weight vector
\eqn{w_i = -\log U_i} (rescaled to mean 1) is used for both training
and validation. This reproduces the fractional-random-weight bootstrap
regression of Xu et al. (2020) and related work, with one weighted fit
and no self-validation split.
\item \code{"Identity"}: Uses unit weights for both training and
validation (no resampling). This is primarily useful with
\code{nBoot = 1} when you want a single glmnet fit whose penalty
is chosen via the selected information criterion (wAIC/wBIC/wSSE),
while still using SVEMnet’s formula expansion and diagnostics.
}}

\item{objective}{Objective used to pick lambda on each bootstrap path
(default \code{"auto"}). One of \code{"auto"}, \code{"wAIC"}, \code{"wBIC"}, or \code{"wSSE"}.}

\item{auto_ratio_cutoff}{Single cutoff for the automatic rule when
\code{objective = "auto"} (default 1.3). Let \code{r = n_X / p_X}, where \code{n_X} is the
number of training rows and \code{p_X} is the number of predictors in the model
matrix after dropping the intercept column. If \code{r >= auto_ratio_cutoff},
SVEMnet uses wAIC; otherwise it uses wBIC.}

\item{relaxed}{Logical, TRUE or FALSE (default TRUE). When TRUE (for
\code{family = "gaussian"}), use glmnet's relaxed elastic net path and
select both lambda and relaxed gamma on each bootstrap. When FALSE,
fit the standard glmnet path. Note: if \code{relaxed = TRUE} and
\code{glmnet_alpha} includes 0 (ridge), alpha = 0 is dropped. For
\code{family = "binomial"}, relaxed fits are currently disabled for
stability: SVEMnet behaves as if \code{relaxed = FALSE}, and a warning
is issued if \code{relaxed = TRUE} is requested.}

\item{response}{Optional character. When \code{formula} is a \code{bigexp_spec}, this names
the response column to use on the LHS; defaults to the response stored in the spec.}

\item{unseen}{How to treat unseen factor levels when \code{formula} is a \code{bigexp_spec}:
\code{"warn_na"} (default; convert to NA with a warning) or \code{"error"} (stop).}

\item{family}{Character scalar specifying the model family.
One of \code{"gaussian"} (default) or \code{"binomial"}.
SVEMnet currently assumes the canonical identity link for Gaussian
and the canonical logit link for binomial. For \code{"binomial"},
the response must be numeric 0/1, logical, or a factor with exactly
two levels (the second level is treated as 1).}

\item{...}{Additional args passed to \code{glmnet()} (e.g., \code{penalty.factor},
\code{lower.limits}, \code{upper.limits}, \code{offset}, \code{standardize.response}, etc.).
Any user-supplied \code{weights} are ignored so SVEM can supply its own bootstrap weights.
Any user-supplied \code{standardize} is ignored; SVEMnet always uses \code{standardize = TRUE}.}
}
\value{
An object of class \code{svem_model} with elements:
\itemize{
\item \code{parms}: averaged coefficients (including intercept).
\item \code{parms_debiased}: averaged coefficients adjusted by the calibration fit.
\item \code{debias_fit}: \code{lm(y ~ y_pred)} calibration model used for debiasing (or \code{NULL}).
\item \code{coef_matrix}: per-bootstrap coefficient matrix.
\item \code{nBoot}, \code{glmnet_alpha}, \code{best_alphas}, \code{best_lambdas}, \code{weight_scheme}, \code{relaxed}.
\item \code{best_relax_gammas}: per-bootstrap relaxed gamma chosen (NA if \code{relaxed = FALSE}).
\item \code{objective_input}, \code{objective_used}, \code{objective} (same as \code{objective_used}),
\code{auto_used}, \code{auto_decision}, \code{auto_rule}.
\item \code{dropped_alpha0_for_relaxed}: whether alpha = 0 was removed because \code{relaxed = TRUE}.
\item \code{schema}: list(\code{feature_names}, \code{terms_str}, \code{xlevels}, \code{contrasts}, \code{terms_hash}) for safe predict.
\item \code{sampling_schema}: list(
\code{predictor_vars}, \code{var_classes},
\code{num_ranges} = rbind(min=..., max=...) for numeric raw predictors,
\code{factor_levels} = list(...) for factor/character raw predictors).
\item \code{diagnostics}: list with \code{k_summary} (median and IQR of selected size),
\code{fallback_rate}, \code{n_eff_summary}, \code{alpha_freq}, \code{relax_gamma_freq}.
\item \code{actual_y}, \code{training_X}, \code{y_pred}, \code{y_pred_debiased}, \code{nobs}, \code{nparm},
\code{formula}, \code{terms}, \code{xlevels}, \code{contrasts}.
\item \code{used_bigexp_spec}: logical flag indicating whether a \code{bigexp_spec} was used.
}
}
\description{
Wrapper for glmnet (Friedman et al. 2010) to fit an ensemble of Elastic Net
models using the Self-Validated Ensemble Model method (SVEM; Lemkus et al. 2021),
with an option to use glmnet's built-in relaxed elastic net (Meinshausen 2007).
Supports searching over multiple alpha values in the Elastic Net penalty.
}
\details{
You can pass either:
\itemize{
\item a standard model formula, e.g. y ~ X1 + X2 + X3 + I(X1^2) + (X1 + X2 + X3)^2
\item a bigexp_spec created by bigexp_terms(), in which case SVEMnet will prepare
the data deterministically (locked types/levels) and, if requested, swap
the response to fit multiple independent responses over the same expansion.
}

In many applications, \code{SVEMnet()} is used as part of a closed-loop
optimization workflow:
models are fit on current experimental data, whole-model tests (WMT) are
optionally used to reweight response goals, and then
\code{svem_optimize_random()} proposes both optimal and exploration
candidates for the next experimental round.
See the \code{lipid_screen} help page for a worked example.

SVEM applies fractional bootstrap weights to training data and anti-correlated
weights for validation when \code{weight_scheme = "SVEM"}. For each bootstrap, glmnet
paths are fit for each alpha in \code{glmnet_alpha}, and the lambda (and, if \code{relaxed = TRUE},
relaxed gamma) minimizing a weighted validation criterion is selected.

\strong{Weighting schemes.}
With \code{weight_scheme = "SVEM"} (the default), SVEMnet uses the
fractional-random-weight (FRW) construction with an explicit
self-validation split: for each bootstrap replicate and observation
\eqn{i}, a shared \eqn{U_i \sim \mathrm{Uniform}(0,1)} is drawn and
converted into anti-correlated train/validation copies
\eqn{w_i^{\mathrm{train}} = -\log U_i} and
\eqn{w_i^{\mathrm{valid}} = -\log(1-U_i)}, each rescaled so that their
mean is 1. This keeps all rows in every fit while inducing a stable
out-of-bag–style validation set for selecting \eqn{\lambda} (and, if used,
the relaxed \eqn{\gamma}).

With \code{weight_scheme = "FRW_plain"}, SVEMnet instead uses a single FRW
weight vector \eqn{w_i = -\log U_i} for \emph{both} training and validation
(one weighted fit, no self-validation split). It is included mainly for
historical comparison and method-teaching; for small-sample prediction we
recommend the default \code{"SVEM"} scheme.

Finally, \code{weight_scheme = "Identity"} sets both training and
validation weights to 1. In combination with \code{nBoot = 1}, this
effectively wraps a single \code{glmnet} fit and chooses \eqn{\lambda}
(and, for Gaussian models, the relaxed \eqn{\gamma}) by the chosen
information criterion (wAIC or wBIC), without any bootstrap variation.
This can be useful when you want classical AIC/BIC selection on top of
a deterministic expansion, but do not want or need ensembling.

Predictors are always standardized internally via
\code{glmnet::glmnet(..., standardize = TRUE)}.

When \code{relaxed = TRUE} and \code{family = "gaussian"},
\code{coef(fit, s = lambda, gamma = g)} is used to obtain the coefficient
path at each relaxed gamma in the internal grid (by default
\code{c(0.2, 0.6, 1)}). Metrics are computed from validation-weighted
errors and model size is taken as the number of nonzero coefficients
including the intercept (support size), keeping selection consistent
between standard and relaxed paths. For \code{family = "binomial"},
relaxed fits are currently disabled for numerical stability, so only
the standard glmnet path is used even if \code{relaxed = TRUE}.

Automatic objective rule (\code{"auto"}): This function uses a single ratio
cutoff, \code{auto_ratio_cutoff}, applied to \code{r = n_X / p_X}, where
\code{p_X} is computed from the model matrix with the intercept column
removed. If \code{r >= auto_ratio_cutoff} the selection criterion is wAIC;
otherwise it is wBIC.

Implementation notes for safety:
\itemize{
\item The training terms object is stored with environment set to \code{baseenv()} to avoid
accidental lookups in the calling environment.
\item A compact schema (feature names, xlevels, contrasts) is stored to let \code{predict()}
reconstruct the design matrix deterministically.
\item A lightweight sampling schema (numeric ranges and factor levels for raw predictors)
is cached to enable random-table generation without needing the original data.
}

For \code{family = "gaussian"}, the loss used in validation is a weighted SSE,
and wAIC/wBIC are computed from a Gaussian log-likelihood proxy.

For \code{family = "binomial"}, the validation loss is the weighted binomial
deviance, and wAIC/wBIC are computed as \code{deviance + 2 * k} or
\code{deviance + log(n_eff) * k}, where \code{k} is the number of active
parameters (1 for the intercept plus the number of nonzero parameters) and
\code{n_eff} is the effective validation size. The response must be numeric
0/1 or a two-level factor; internally it is converted to 0/1.
}
\section{Acknowledgments}{

OpenAI's GPT models (o1-preview and GPT-5 Thinking via ChatGPT) were
used to assist with coding and roxygen documentation; all
content was reviewed and finalized by the author.
}

\examples{
set.seed(42)

n  <- 30
X1 <- rnorm(n)
X2 <- rnorm(n)
X3 <- rnorm(n)
eps <- rnorm(n, sd = 0.5)
y <- 1 + 2*X1 - 1.5*X2 + 0.5*X3 + 1.2*(X1*X2) + 0.8*(X1^2) + eps
dat <- data.frame(y, X1, X2, X3)

# Minimal hand-written expansion
mod_relax <- SVEMnet(
  y ~ (X1 + X2 + X3)^2 + I(X1^2) + I(X2^2),
  data          = dat,
  glmnet_alpha  = c(1, 0.5),
  nBoot         = 75,
  objective     = "auto",
  weight_scheme = "SVEM",
  relaxed       = FALSE
)

pred_in_raw <- predict(mod_relax, dat, debias = FALSE)
pred_in_db  <- predict(mod_relax, dat, debias = TRUE)

\donttest{
# ---------------------------------------------------------------------------
# Big expansion (full factorial + polynomial surface + partial-cubic crosses)
# Build once, reuse for one or more responses
# ---------------------------------------------------------------------------
spec <- bigexp_terms(
  y ~ X1 + X2 + X3,
  data            = dat,
  factorial_order = 3,  # up to 3-way interactions among X1, X2, X3
  polynomial_order = 3, # include I(X^3) for each continuous predictor
  include_pc_3way = FALSE
)

# Fit using the spec (auto-prepares data)
fit_y <- SVEMnet(
  spec, dat,
  glmnet_alpha  = c(1, 0.5),
  nBoot         = 50,
  objective     = "auto",
  weight_scheme = "SVEM",
  relaxed       = FALSE
)

# A second, independent response over the same expansion
set.seed(99)
dat$y2 <- 0.5 + 1.4*X1 - 0.6*X2 + 0.2*X3 + rnorm(n, 0, 0.4)
fit_y2 <- SVEMnet(
  spec, dat, response = "y2",
  glmnet_alpha  = c(1, 0.5),
  nBoot         = 50,
  objective     = "auto",
  weight_scheme = "SVEM",
  relaxed       = FALSE
)

p1  <- predict(fit_y,  dat)
p2  <- predict(fit_y2, dat, debias = TRUE)

# Show that a new batch expands identically under the same spec
newdat <- data.frame(
  y  = y,
  X1 = X1 + rnorm(n, 0, 0.05),
  X2 = X2 + rnorm(n, 0, 0.05),
  X3 = X3 + rnorm(n, 0, 0.05)
)
prep_new <- bigexp_prepare(spec, newdat)
stopifnot(identical(
  colnames(model.matrix(spec$formula, bigexp_prepare(spec, dat)$data)),
  colnames(model.matrix(spec$formula, prep_new$data))
))
preds_new <- predict(fit_y, prep_new$data)


#' ## ---- Binomial example ------------------------------------------------
set.seed(2)
n  <- 120
X1 <- rnorm(n); X2 <- rnorm(n); X3 <- rnorm(n)
eta <- -0.3 + 1.1*X1 - 0.8*X2 + 0.5*X1*X3
p   <- plogis(eta)
yb  <- rbinom(n, 1, p)
db  <- data.frame(yb = yb, X1 = X1, X2 = X2, X3 = X3)

fit_b <- SVEMnet(
  yb ~ (X1 + X2 + X3)^2, db,
  nBoot = 50, glmnet_alpha = c(1, 0.5), relaxed = FALSE, family = "binomial"
)

## Probabilities, link, and classes
p_resp <- predict(fit_b, db, type = "response")
p_link <- predict(fit_b, db, type = "link")
y_hat  <- predict(fit_b, db, type = "class")      # 0/1 labels (no SE/interval)

## Mean-aggregation with uncertainty on probability scale
out_b <- predict(fit_b, db, type = "response",
                 se.fit = TRUE, interval = TRUE, level = 0.9)
str(out_b)
}

}
\references{
Gotwalt, C., & Ramsey, P. (2018). Model Validation Strategies for Designed Experiments Using Bootstrapping Techniques With Applications to Biopharmaceuticals. \emph{JMP Discovery Conference}. \url{https://community.jmp.com/t5/Abstracts/Model-Validation-Strategies-for-Designed-Experiments-Using/ev-p/849873/redirect_from_archived_page/true}

Karl, A. T. (2024). A randomized permutation whole-model test heuristic for Self-Validated Ensemble Models (SVEM). \emph{Chemometrics and Intelligent Laboratory Systems}, \emph{249}, 105122. \doi{10.1016/j.chemolab.2024.105122}

Karl, A., Wisnowski, J., & Rushing, H. (2022). JMP Pro 17 Remedies for Practical Struggles with Mixture Experiments. JMP Discovery Conference. \doi{10.13140/RG.2.2.34598.40003/1}

Lemkus, T., Gotwalt, C., Ramsey, P., & Weese, M. L. (2021). Self-Validated Ensemble Models for Design of Experiments. \emph{Chemometrics and Intelligent Laboratory Systems}, 219, 104439. \doi{10.1016/j.chemolab.2021.104439}

Xu, L., Gotwalt, C., Hong, Y., King, C. B., & Meeker, W. Q. (2020). Applications of the Fractional-Random-Weight Bootstrap. \emph{The American Statistician}, 74(4), 345–358. \doi{10.1080/00031305.2020.1731599}

Ramsey, P., Gaudard, M., & Levin, W. (2021). Accelerating Innovation with Space Filling Mixture Designs, Neural Networks and SVEM. \emph{JMP Discovery Conference}. \url{https://community.jmp.com/t5/Abstracts/Accelerating-Innovation-with-Space-Filling-Mixture-Designs/ev-p/756841}

Ramsey, P., & Gotwalt, C. (2018). Model Validation Strategies for Designed Experiments Using Bootstrapping Techniques With Applications to Biopharmaceuticals. \emph{JMP Discovery Conference - Europe}. \url{https://community.jmp.com/t5/Abstracts/Model-Validation-Strategies-for-Designed-Experiments-Using/ev-p/849647/redirect_from_archived_page/true}

Ramsey, P., Levin, W., Lemkus, T., & Gotwalt, C. (2021). SVEM: A Paradigm Shift in Design and Analysis of Experiments. \emph{JMP Discovery Conference - Europe}. \url{https://community.jmp.com/t5/Abstracts/SVEM-A-Paradigm-Shift-in-Design-and-Analysis-of-Experiments-2021/ev-p/756634}

Ramsey, P., & McNeill, P. (2023). CMC, SVEM, Neural Networks, DOE, and Complexity: It's All About Prediction. \emph{JMP Discovery Conference}.

Friedman, J. H., Hastie, T., and Tibshirani, R. (2010).
Regularization Paths for Generalized Linear Models via Coordinate Descent.
Journal of Statistical Software, 33(1), 1-22.

Meinshausen, N. (2007).
Relaxed Lasso. Computational Statistics & Data Analysis, 52(1), 374-393.
}
