% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/svem_random_table_multi.R
\name{svem_random_table_multi}
\alias{svem_random_table_multi}
\title{Generate a Random Prediction Table from Multiple SVEMnet Models (no refit)}
\usage{
svem_random_table_multi(
  objects,
  n = 1000,
  mixture_groups = NULL,
  debias = FALSE,
  range_tol = 1e-08,
  numeric_sampler = c("random", "maximin", "uniform")
)
}
\arguments{
\item{objects}{A list of fitted \code{svem_model} objects returned by \code{SVEMnet()}.
Each object must contain \code{$sampling_schema} produced by the updated \code{SVEMnet()}.
A single model is also accepted and treated as a length-one list.}

\item{n}{Number of random points to generate. Default is \code{1000}.}

\item{mixture_groups}{Optional list of mixture constraint groups. Each group is a list
with elements \code{vars}, \code{lower}, \code{upper}, \code{total}.}

\item{debias}{Logical; if \code{TRUE}, apply each model's calibration during prediction
when available. Default is \code{FALSE}.}

\item{range_tol}{Numeric tolerance for comparing numeric ranges across models.
Default is \code{1e-8}.}

\item{numeric_sampler}{Sampler for non-mixture numeric predictors:
\code{"random"} (default), \code{"maximin"}, or \code{"uniform"}.
If \code{"random"} is selected and the \pkg{lhs} package is available,
\code{lhs::randomLHS} is used; otherwise plain \code{runif}.}
}
\value{
A list with three data frames:
\itemize{
\item \code{data}: the sampled predictor settings, one row per random point.
\item \code{pred}: one column per response, aligned to \code{data} rows.
\item \code{all}: \code{cbind(data, pred)} for convenience.
}
Each prediction column is named by the model's response (left-hand side). If a response
name would collide with a predictor name, \code{".pred"} is appended.
}
\description{
Samples the original predictor factor space cached in fitted \code{svem_model}
objects and computes predictions from each model at the same random points.
Intended for multiple responses built over the same factor space and a
deterministic factor expansion.
}
\details{
All models must share an identical predictor schema:
\itemize{
\item The same \code{predictor_vars} in the same order
\item The same \code{var_classes} for each predictor
\item Identical factor \code{levels} and level order
\item Numeric \code{ranges} that match within \code{range_tol}
}
The function stops with an informative error message if any of these checks fail.
}
\section{Sampling strategy}{

Non-mixture numeric variables are sampled using a selectable method:
\itemize{
\item \code{"random"}: random Latin hypercube when \pkg{lhs} is available, else uniform.
\item \code{"maximin"}: maximin Latin hypercube (space-filling; slower).
\item \code{"uniform"}: independent uniform draws within numeric ranges (fastest).
}
Mixture variables (if any) are sampled jointly within each specified group using
a truncated Dirichlet so that elementwise bounds and the total sum are satisfied.
Categorical variables are sampled from cached factor levels. The same random
predictor table is fed to each model so response columns are directly comparable.
}

\section{Notes on mixtures}{

Each mixture group should list only numeric-like variables. Bounds are interpreted
on the original scale of those variables. If \code{total} equals the sum of lower
bounds, the sampler returns the lower-bound corner for that group.
}

\examples{
\donttest{
set.seed(1)
n <- 60
X1 <- runif(n); X2 <- runif(n)
A <- runif(n); B <- runif(n); C <- pmax(0, 1 - A - B)
F <- factor(sample(c("lo","hi"), n, TRUE))
y1 <- 1 + 2*X1 - X2 + 3*A + 1.5*B + 0.5*C + (F=="hi") + rnorm(n, 0, 0.3)
y2 <- 0.5 + 0.8*X1 + 0.4*X2 + rnorm(n, 0, 0.2)
d  <- data.frame(y1, y2, X1, X2, A, B, C, F)

fit1 <- SVEMnet(y1 ~ X1 + X2 + A + B + C + F, d, nBoot = 40)
fit2 <- SVEMnet(y2 ~ X1 + X2 + A + B + C + F, d, nBoot = 40)

# Mixture constraint for A, B, C that sum to 1
mix <- list(list(vars=c("A","B","C"),
                 lower=c(0,0,0), upper=c(1,1,1), total=1))

# Fast random sampler
tab_fast <- svem_random_table_multi(
  objects        = list(fit1, fit2), n = 2000, mixture_groups = mix,
  debias         = FALSE, numeric_sampler = "random"
)
head(tab_fast$all)

# Uniform sampler (fastest)
tab_uni <- svem_random_table_multi(
  objects        = list(fit1, fit2), n = 2000,
  debias         = FALSE, numeric_sampler = "uniform"
)
head(tab_uni$all)
}

}
\seealso{
\code{SVEMnet}, \code{predict.svem_model}
}
