% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/svem_random_table.R
\name{svem_random_table}
\alias{svem_random_table}
\title{Generate a Random Prediction Table for a Fitted SVEMnet Model}
\usage{
svem_random_table(
  formula,
  data,
  n = 1000,
  mixture_groups = NULL,
  nBoot = 200,
  glmnet_alpha = c(1),
  weight_scheme = c("SVEM"),
  objective = c("wAIC", "wSSE"),
  debias = FALSE,
  ...
)
}
\arguments{
\item{formula}{A formula specifying the fitted model.  This should be the
same formula used when fitting the SVEMnet model.}

\item{data}{A data frame containing the variables in the model.}

\item{n}{Number of random points to generate (default: 1000).}

\item{mixture_groups}{Optional list describing mixture factor groups.  Each
element should be a list with components `vars` (character vector of
mixture variable names), `lower` (numeric vector of lower bounds),
`upper` (numeric vector of upper bounds) and `total` (scalar sum).
See `svem_significance_test_with_mixture()` for details.  Defaults
to `NULL` (no mixtures).}

\item{nBoot}{Number of bootstrap iterations to use when fitting the
SVEMnet model (default: 200).}

\item{glmnet_alpha}{Elastic net mixing parameter(s) passed to `SVEMnet`
(default: `c(1)`).}

\item{weight_scheme}{Weighting scheme for SVEM (default: "SVEM").}

\item{objective}{Objective function for SVEM ("wAIC" or "wSSE";
default: "wAIC").}

\item{debias}{Logical; if `TRUE`, the debiasing coefficients of the
fitted model are applied when predicting (default: `FALSE`).}

\item{...}{Additional arguments passed to `SVEMnet()` and then to
`glmnet()`.}
}
\value{
A data frame containing the sampled predictor values and the
  corresponding predicted responses.  The response column is named
  according to the left-hand side of `formula`.
}
\description{
This utility function generates a random sample of points from the
predictor space and computes the corresponding predicted responses from
a fitted SVEMnet model.  It can be used to explore the fitted
response surface in a way analogous to JMP's "Output Random Table"
feature.  The function recognizes mixture factor groups and draws
Dirichlet-distributed compositions within the specified bounds so
that mixture variables sum to a user-supplied total.  Continuous
non-mixture variables are sampled uniformly across their observed
ranges using a maximin Latin hypercube design, and categorical
variables are sampled from their observed levels.  No random noise is
added to the predicted responses.
}
\details{
This function first fits an SVEMnet model using the supplied
parameters.  It then generates a random grid of points in the
predictor space, honouring mixture constraints if `mixture_groups`
is provided.  Predictions are computed from the fitted model on these
points.  No random noise is added; the predictions come directly from
the model.  If you wish to explore the uncertainty of predictions,
consider adding noise separately or using the standard error output
from `predict.svem_model()`.
}
\examples{
\donttest{
set.seed(42)
n <- 40

# Helper to generate training data mixtures with bounds
sample_trunc_dirichlet <- function(n, lower, upper, total) {
  k <- length(lower)
  min_sum <- sum(lower); max_sum <- sum(upper)
  stopifnot(total >= min_sum, total <= max_sum)
  avail <- total - min_sum
  out <- matrix(NA_real_, n, k)
  i <- 1
  while (i <= n) {
    g <- rgamma(k, 1, 1)
    w <- g / sum(g)
    x <- lower + avail * w
    if (all(x <= upper + 1e-12)) {
      out[i, ] <- x; i <- i + 1
    }
  }
  out
}

# Three mixture factors (A, B, C) with distinct bounds; sum to total = 1
lower <- c(0.10, 0.20, 0.05)
upper <- c(0.60, 0.70, 0.50)
total <- 1.0
ABC   <- sample_trunc_dirichlet(n, lower, upper, total)
A <- ABC[, 1]; B <- ABC[, 2]; C <- ABC[, 3]

# Additional predictors
X <- runif(n)
F <- factor(sample(letters[1:3], n, replace = TRUE))

# Response
y <- 1 + 2*A + 3*B + 1.5*C + 0.5*X +
     ifelse(F == "a", 0, ifelse(F == "b", 1, -1)) +
     rnorm(n, sd = 0.3)

dat <- data.frame(y = y, A = A, B = B, C = C, X = X, F = F)

# Mixture specification for the random table generator
mix_spec <- list(
  list(
    vars  = c("A", "B", "C"),
    lower = c(0.10, 0.20, 0.05),
    upper = c(0.60, 0.70, 0.50),
    total = 1.0
  )
)

# Fit SVEMnet and generate 50 random points
rand_tab <- svem_random_table(
  y ~ A + B + C + X + F,
  data           = dat,
  n              = 50,
  mixture_groups = mix_spec,
  nBoot          = 50,
  glmnet_alpha   = c(1),
  weight_scheme  = "SVEM",
  objective      = "wAIC",
  debias         = FALSE
)

# Check mixture validity in the generated table
stopifnot(all(abs((rand_tab$A + rand_tab$B + rand_tab$C) - 1) < 1e-8))
summary(rand_tab[c("A","B","C")])
head(rand_tab)
}
}
\seealso{
`SVEMnet`, `predict.svem_model`, `svem_significance_test_with_mixture`.
}
