% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/semLearn.R
\name{resizeGraph}
\alias{resizeGraph}
\title{Interactome-assisted graph re-seizing}
\usage{
resizeGraph(g = list(), gnet, d = 2, v = TRUE, verbose = FALSE, ...)
}
\arguments{
\item{g}{A list of two graphs as igraph objects, g=list(graph1, graph2).}

\item{gnet}{External directed network as an igraph object. The reference
network should have weighted edges, corresponding to their interaction
p-values, as an edge attribute \code{E(gnet)$pv}. Then, connections in
\code{graph2} will be checked by known connections from the reference network,
intercepted by the minimum-weighted shortest path found among the equivalent
ones by the Dijkstra algorithm, as implemented in the \pkg{igraph} function
\code{all_shortest_paths()}.}

\item{d}{An integer value indicating the maximum geodesic distance between
two nodes in the interactome to consider the inferred interaction between
the same two nodes in \code{graph2} as validated, otherwise the edges are
removed. For instance, if \code{d = 2}, two interacting nodes must either
share a direct interaction or being connected through at most one mediator
in the reference interactome (in general, at most \code{d - 1} mediators are
allowed). Typical \code{d} values include \code{2} (at most one mediator), or
\code{mean_distance(gnet)} (i.e., the average shortest path length for
the reference network). Setting d = 0, is equivalent to \code{gnet = NULL}.}

\item{v}{A logical value. If TRUE (default) new nodes and edges on the
validated shortest path in the reference interactome will be added in the
re-sized graph.}

\item{verbose}{A logical value. If FALSE (default), the processed graphs
will not be plotted to screen, saving execution time (for large graphs)}

\item{...}{Currently ignored.}
}
\value{
"Ug", the re-sized graph, the graph union of the causal graph \code{graph1}
and the re-sized graph \code{graph2}
}
\description{
An input directed graph is re-sized, removing edges
or adding edges/nodes. This function takes three input graphs: the
first is the input causal model (i.e., a directed graph), and the
second can be either a directed or undirected graph, providing a set
of connections to be checked against a directed reference network
(i.e., the third input) and imported to the first graph.
}
\details{
Typically, the first graph is an estimated causal graph (DAG),
and the second graph is the output of either \code{\link[SEMgraph]{SEMdag}}
or \code{\link[SEMgraph]{SEMbap}}.
In the former we use the new inferred causal structure stored in the
\code{dag.new} object. In the latter, we use the new inferred covariance
structure stored in the \code{guu} object. Both directed (causal) edges
inferred by \code{SEMdag()} and covariances (i.e., bidirected edges)
added by \code{SEMbap()}, highlight emergent hidden topological
proprieties, absent in the input graph. Estimated directed edges between
nodes X and Y are interpreted as either direct links or direct paths
mediated by hidden connector nodes. Covariances between any two bow-free
nodes X and Y may hide causal relationships, not explicitly represented
in the current model. Conversely, directed edges could be redundant or artifact,
specific to the observed data and could be deleted.
Function \code{resizeGraph()} leverage on these concepts to extend/reduce a
causal model, importing new connectors or deleting estimated edges, if they are
present or absent in a given reference network. The whole process may lead to
the discovery of new paths of information flow, and cut edges not corroborate
by a validated network. Since added nodes can already be present in the causal
graph, network resize may create cross-connections between old and new paths
and their possible closure into circuits.
}
\examples{

\donttest{

# Extract the "Protein processing in endoplasmic reticulum" pathway:

g <- kegg.pathways[["Protein processing in endoplasmic reticulum"]]
G <- properties(g)[[1]]; summary(G)

# Extend a graph using new inferred DAG edges (dag+dag.new):

library(huge)
als.npn <- huge.npn(alsData$exprs)

dag <- SEMdag(graph = G, data = als.npn, beta = 0.1)
gplot(dag$dag)
ext <- resizeGraph(g=list(dag$dag, dag$dag.new), gnet = kegg, d = 2)
gplot(ext) 

# Extend a graph using the inferred bow-free path diagram (dag+guu):

bap <- SEMbap(graph = G, data = als.npn, gnet = kegg, alpha = 0.01)
ext <- resizeGraph(g=list(bap$dag, bap$guu), gnet = kegg, d = 1, v = FALSE)
gplot(ext)

# Create a directed graph from correlation matrix, using
# i) empty graph as causal graph, and ii) KEGG as reference:

v <- which(colnames(als.npn) \%in\% V(G)$name)
selectedData <- als.npn[, v]
G0 <- make_empty_graph(n = ncol(selectedData))
V(G0)$name <- colnames(selectedData)

G1 <- corr2graph(R = cor(selectedData), n = nrow(selectedData),
                 type = "tmfg")
ext <- resizeGraph(g=list(G0, G1), gnet = kegg, d = 2, v = FALSE)

#Graphs
old.par <- par(no.readonly = TRUE)
par(mfrow=c(1,2), mar=rep(1,4))
plot(G1, layout = layout.circle)
plot(ext, layout = layout.circle)
par(old.par)

}

}
\references{
Palluzzi F, Grassi M (2021). SEMgraph: An R Package for Causal Network
Analysis of High-Throughput Data with Structural Equation Models.
<arXiv:2103.08332>
}
\author{
Mario Grassi \email{mario.grassi@unipv.it}
}
