genShinyApp.template <-
function(appDir = "shinyExample", verbose = TRUE)
{
   if(.Platform$OS.type=="windows")
      appDir <- gsub("\\\\", "/", shortPathName(appDir))  # safe pathname

   if(!file.exists(appDir))
         dir.create(appDir, recursive = TRUE)

   pkpd = "
   C2 = centr/V2;
   C3 = peri/V3;
   d/dt(depot) =-KA*depot;
   d/dt(centr) = KA*depot - CL*C2 - Q*C2 + Q*C3;
   d/dt(peri)  =                    Q*C2 - Q*C3;
   d/dt(eff)  = Kin - Kout*(1-C2/(EC50+C2))*eff; 
   "

   if(verbose){
      cat("\nGenerating an example (template) for a dosing regimen shiny app\n\n")
      cat("Using the following PK/PD model:")
      cat(pkpd, "\n")

      cat("Translating the PK/PD ODE model into C, compiling, etc.\n\n")
   }

   mod1 <- RxODE(pkpd, modName = "mod1", wd = appDir)

   params <- 
      c(KA=.294, CL=18.6, V2=40.2, Q=10.5, V3=297.0,
        Kin=1.0, Kout=1.0, EC50=200.0)

   inits <- c(depot=0, centr=0, pari=0, eff=1)      

   if(verbose){
      cat("\nParameters and their values:\n")
      print(params)
      cat("\nInitial values in each compartment:\n")
      print(inits)
      cat("\n")
   }

   # save the model, parameters, init values, etc. in the
   # file rx_shiny_data.rda to be loaded by the server.R

   fn <- file.path(appDir, "rx_shiny_data.rda")
   stiff <- TRUE
   atol <- 1e08
   rtol <- 1e06
   save(mod1, params, inits, stiff, atol, rtol, file = fn)

   # write the shiny server.R and ui.R files 
   write.template.server(appDir)
   write.template.ui(appDir)

   if(verbose){
      cat("Shiny files (ui.R, server.R) plus R data saved.\n")
   }

   cat("\nTo launch the Shiny app, type the following two R commands:\n\n")
   cat("\tlibrary(shiny)\n")
   cat(sprintf('\trunApp("%s")\n\n', appDir))
}

write.template.server <-
function(appDir)
{
   # create a shiny server interface server.R script in the appDir

   server <- file.path(appDir, "server.R")

   server.code <- 
      sprintf(appDir, fmt = '
      #
      # Dosing regimen template generated by RxODE::genShinyApp.template()
      #

      # Server inputs: Dose, dosing regimen, dosing frequency,
      # dosing cycle definition, number of dosing cycles

      library(shiny)
      library(RxODE)

      # read objects from "rx_shiny_data.rda" in the  AppDir folder,
      # objects include, mod1, params, inits, stiff, atol, rtol.
      load(file.path("rx_shiny_data.rda"))

      # Define server logic 
      shinyServer(function(input, output) {

        get.cp <- reactive({    
          ds <- input$Dose 
          reg <- switch(input$regimen, "QD"=1, "BID"=2) 
          cyc <- switch(input$cycle, 
              "continous"=c(7,0), 
              "1wkon 1wkoff"=c(7,7),
              "2wkon 1wkoff"=c(14,7),
              "3wkon 1wkoff"=c(21,7)
          ) 
          cyc <- rep(1:0, cyc) 
          ncyc <- input$ncyc
          lcyc <- length(cyc)
          
          ev <- eventTable()
          for (i in 1:ncyc) ev$add.dosing(
              dose=ds, 
              nbr.doses=sum(cyc)*reg, 
              dosing.interval=24/reg,
              start.time=(i-1)*lcyc*24
          )
          ev$get.EventTable()
          ev$add.sampling(0:(ncyc*lcyc*24))

          mod1$solve(params, ev, inits, stiff=stiff, atol=atol, rtol=rtol)
        })


        output$summary <- renderPrint({
          x <- get.cp()
          print(x[1:4,])
        })
        
        output$CpPlot <- renderPlot({
          x <- get.cp()
          cmp <- input$compartment
          plot(x[,c("time", cmp)], xlab = "Time", ylab = "Drug amount", 
               main = cmp, type = "l")
        })
      })
   ')
   writeLines(server.code, con = server)
}

write.template.ui <- 
function(appDir)
{
   # create a shiny user interface ui.R script in the appDir

   ui <- file.path(appDir, "ui.R")

   ui.code <- 
      sprintf(appDir, fmt = '
      #
      # Dosing regimen template automatically generated by 
      # RxODE::genShinyApp.template()
      #

      library(RxODE)
      library(shiny)

      # Define UI for dataset viewer application
      shinyUI(pageWithSidebar(
        
        # Application title.
        headerPanel("Regimen simulator"),
        
        # Sidebar with controls to select a dataset and specify the number
        # of observations to view. The helpText function is also used to 
        # include clarifying text. Most notably, the inclusion of a 
        # submitButton defers the rendering of output until the user 
        # explicitly clicks the button (rather than doing it immediately
        # when inputs change). This is useful if the computations required
        # to render output are inordinately time-consuming.
        sidebarPanel(
          sliderInput("Dose", "Dose:", 
                        min=100, max=1000, value=400, step=50),
                        
          selectInput("regimen", "Regimen:", 
                      choices = c("QD", "BID"),
                      selected="QD"),

          selectInput("cycle", "Cycle:", 
                      choices = c(
                              "continous", 
                              "1wkon 1wkoff", 
                              "2wkon 1wkoff", 
                              "3wkon 1wkoff"),
                      selected="1wkon 1wkoff"),

          sliderInput("ncyc", "# cycles:", 
                        min=1, max=5, value=2, step=1),

          radioButtons("compartment", "Select PK compartment for plotting:", 
                       choices=c("depot", "centr", "peri"),
                       selected = "centr")
        ),
        
        # Show a summary of the dataset and an HTML table with the requested
        # number of observations. Note the use of the h4 function to provide
        # an additional header above each output section.
        mainPanel(
          h4("State variables"),
          verbatimTextOutput("summary"),
          h4("Concentration time course"),
          plotOutput("CpPlot")
        )
      ))
   ')

   writeLines(ui.code, con = ui)
}
