% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Application.R
\docType{data}
\name{Application}
\alias{Application}
\title{Creates application - RestRserve usage starts from here}
\format{\link[R6:R6Class]{R6::R6Class} object.}
\description{
Creates Application object.
Application provides an interface for building high-performance
REST API by registering R functions as handlers http requests.
}
\section{Construction}{

Constructor for \code{Application}.\preformatted{Application$new(middleware = list(), content_type = "text/plain", ...)
}
\itemize{
\item \strong{\code{middleware}} :: \code{list} of \link{Middleware}\cr
List of middlewares.
\item \strong{\code{content_type}} :: \code{character(1)}\cr
Default response body content (media) type. \code{"text/plain"} by default.
\item \strong{\code{...}} \cr
Not used at the moment
}
}

\section{Fields}{

\itemize{
\item \strong{\code{logger}} :: \link{Logger}\cr
Logger object which records events during request processing.
Alternatively user can use loggers from lgr package as a
drop-in replacement - \code{Logger} methods and loggers created by \code{lgr} share function signatures.
\item \strong{\code{content_type}} :: \code{character(1)}\cr
Default response body content type.
\item \strong{\code{HTTPError}} :: \code{HTTPError}\cr
Class which raises HTTP errors. Global \link{HTTPError} is used by default. In theory
user can replace it with his own class (see \code{RestRserve:::HTTPErrorFactory}). However we believe
in the majority of the cases using \link{HTTPError} will be enough.
\item \strong{\code{endpoints}} :: \code{named list()}\cr
Prints all the registered routes with allowed methods.
}
}

\section{Methods}{

\itemize{
\item \strong{\code{add_route}}\code{(path, method, FUN, match = c("exact", "partial", "regex"), ...)}\cr
\code{character(1)}, \code{character(1)}, \code{character(1)} -> \code{invisible(self)} - \link{Application} \cr
Adds endpoint and register user-supplied R function as a handler.

Allowed methods at the moment: GET, HEAD, POST, PUT, DELETE, OPTIONS, PATCH.

\code{match} parameter defines how route will be processed.
\itemize{
\item \code{exact} - match route as is. Returns 404 if route is not matched.
\item \code{partial} - match route as prefix. Returns 404 if prefix are not matched.
\item \code{regex} - match route as template. Returns 404 if template pattern not matched.
}

User function \code{FUN} \strong{must} take two arguments: first is \code{request}
(\link{Request}) and second is \code{response} (\link{Response}).

The goal of the user function is to \strong{modify} \code{response} or throw
exception (call \code{\link[=raise]{raise()}} or \code{\link[=stop]{stop()}}).

Both \code{response} and \code{request} objects modified in-place and internally
passed further to RestRserve execution pipeline.
\item \strong{\code{add_get}}\code{(path, FUN, match = c("exact", "partial", "regex"), ..., add_head = TRUE)}\cr
\code{character(1)}, \code{character(1)}, \code{character(1)}, \code{any}, \code{logical(1)} -> \code{invisible(self)} - \link{Application} \cr
Shorthand to \code{add_route} with \code{GET} method. With \code{add_head = TRUE} HEAD method
handlers will be added (with \code{add_head()}).
\item \strong{\code{add_post}}\code{(path, FUN, match = c("exact", "partial", "regex"), ...)}\cr
\code{character(1)}, \code{character(1)}, \code{character(1)}, \code{any} -> \code{invisible(self)} - \link{Application} \cr
Shorthand to \code{add_route} with \code{POST} method.
\item \strong{\code{add_static}}\code{(path, file_path, content_type = NULL, ...)}\cr
\code{character(1)}, \code{character(1)}, \code{character(1)}, \code{any} -> \code{invisible(self)} - \link{Application} \cr
Adds GET method to serve file or directory at \code{file_path}.

If \code{content_type = NULL} then MIME code \code{content_type}  will be inferred
automatically (from file extension).

If it will be impossible to guess about file type then \code{content_type} will
be set to \code{"application/octet-stream"}.
\item \strong{\code{append_middleware}}\code{(mw)}\cr
\link{Middleware} -> \code{invisible(self)} - \link{Application} \cr
Appends middleware to handlers pipeline.
\item \strong{\code{process_request}}\code{(request)}\cr
\link{Request} -> \link{Response}\cr
Process incoming request and generate \link{Response} object.
Useful for tests your handlers before deploy application.
\item \strong{\code{add_openapi}}\code{(path = "/openapi.yaml", file_path = "openapi.yaml")}
\code{character(1)}, \code{named list()}, \code{character(1)} -> \code{invisible(self)} - \link{Application} \cr
Adds endpoint to serve \href{https://www.openapis.org/}{OpenAPI} description of
available methods.
\item \strong{\code{add_swagger_ui}}\code{(path = "/swagger", path_openapi = "/openapi.yaml", use_cdn = TRUE, path_swagger_assets = "/__swagger__/", file_path = "swagger-ui.html")}\cr
\code{character(1)}, \code{character(1)}, \code{logical(1)}, \code{character(1)}, \code{character(1)} ->
\code{invisible(self)} - \link{Application} \cr
Adds endpoint to show \href{https://swagger.io/tools/swagger-ui/}{Swagger UI}.
}
}

\examples{
# init logger
app_logger = Logger$new()
# set log level for the middleware
app_logger$set_log_level("debug")
# set logger name
app_logger$set_name("MW Logger")
# init middleware to logging
mw = Middleware$new(
  process_request = function(rq, rs) {
    app_logger$info(sprintf("Incomming request (id \%s): \%s", rq$id, rq$path))
  },
  process_response = function(rq, rs) {
    app_logger$info(sprintf("Outgoing response (id \%s): \%s", rq$id, rs$status))
  },
  id = "awesome-app-logger"
)

# init application
app = Application$new(middleware = list(mw))

# set internal log level
app$logger$set_log_level("error")

# define simply request handler
status_handler = function(rq, rs) {
  rs$set_body("OK")
  rs$set_content_type("text/plain")
  rs$set_status_code(200L)
}
# add route
app$add_get("/status", status_handler, "exact")

# add static file handler
desc_file = system.file("DESCRIPTION", package = "RestRserve")
# add route
app$add_static("/desc", desc_file, "text/plain")

# define say message handler
say_handler = function(rq, rs) {
  who = rq$parameters_path[["user"]]
  msg = rq$parameters_query[["message"]]
  if (is.null(msg)) msg <- "Hello"
  rs$set_body(paste(who, "say", dQuote(msg)))
  rs$set_content_type("text/plain")
  rs$set_status_code(200L)
}
# add route
app$add_get("/say/{user}", say_handler, "regex")

# print application info
app

# test app
# simulate requests
not_found_rq = Request$new(path = "/no")
status_rq = Request$new(path = "/status")
desc_rq = Request$new(path = "/desc")
say_rq = Request$new(path = "/say/anonym", parameters_query = list("message" = "Hola"))
# process prepared requests
app$process_request(not_found_rq)
app$process_request(status_rq)
app$process_request(desc_rq)
app$process_request(say_rq)

# run app
backend = BackendRserve$new()
\donttest{
backend$start(app, 8080)
}

}
\seealso{
\link{HTTPError} \link{Middleware}
\link{Request} \link{Response}
}
\keyword{datasets}
