\name{Rdsm-package}
\alias{Rdsm-package}
\alias{Rdsm}
\docType{package}

\title{
Adds a threaded parallel programming paradigm to R.
}

\description{

This package provides a parallel shared-memory programming paradigm for
R, very similar to threaded programming in C/C++, This enabling the
programmer to write simpler, clearer code.  Even more importantly, it
allows placing very large matrices in secondary storage, while treating
them as being in shared memory.  Furthermore, in some applications this
package produces significantly faster code, compared to versions
written for other parallel R libraries.

}

\details{
\tabular{ll}{
Package: \tab Rdsm\cr
Type: \tab Package\cr
Version: \tab 2.0\cr
Date: \tab 2013-08-18\cr
License: \tab GPL (>= 2)\cr
}

List of functions:

   \preformatted{
   initialization, run at manager:  

      mgrinit():  initialize system
      mgrmakevar():  create a shared variable 
      mgrmakelock():  create a lock
      mgrmakebarr():  create a barrier
      shmcls():  create a Snow cluster

   called by applications:  

      barr():  barrier call
      rdsmlock():  lock operation (via realrdsmlock())
      rdsmunlock():  unlock operation (via realrdsmunlock())

   application utilities:  

      getidxs():  partition a set of indices for work assignment
   }

Built-in variables in the threads, at the worker nodes:

   \preformatted{
   myinfo$nwrkrs:  total number of threads
   myinfo$id:  this thread's ID number
   }


To run, set up a \pkg{snow} cluste (via the \pkg{parallel} package)r;
we'll refer to the R process from which this is done as the manager.
Create the application's shared variables from the manager, using
\code{mgrmakevar()}.  Launch the threads, again from the manager, by the
\pkg{snow} call \code{clusterEvalQ()}.  One typically codes so that the
results are in shared variables.  See examples below, and more in the
\code{examples/} directory in this distribution.

While an \pkg{Rdsm} application is running, other R processes can access
the shared variables.  See the file ExternalAccess.txt in the doc/
directory.

\pkg{Rdsm} uses the \pkg{bigmemory} library to store its shared
variables.  Though the latter can work on a (physical) cluster of
several machines sharing a file system, \pkg{Rdsm} does not run on such
systems at this time.

Further documentation in the doc/ directory.

}

\author{
Norm Matloff <matloff@cs.ucdavis.edu>
}

\keyword{
parallel computation
shared memory 
threads
}

\seealso{
   \code{\link{mgrinit}}, 
   \code{\link{mgrmakevar}}, 
   \code{\link{mgrmakelock}},
   \code{\link{barr}},
   \code{\link{rdsmlock}},
   \code{\link{rdsmunlock}}
}

\examples{
library(parallel)
c2 <- shmcls(2)  # form 2-thread Snow cluster
mgrinit(c2)  # initialize Rdsm
mgrmakevar(c2,"m",2,2)  # make a 2x2 shared matrix
m[,] <- 3  # 2x2 matrix of all 3s
# example of shared memory:
# at each thread, set id to Rdsm built-in ID variable for that thread
clusterEvalQ(c2,id <- myinfo$id)
clusterEvalQ(c2,m[1,id] <- id^2)  # assignment executed by each thread
m[,]  # top row of m should now be (1,4)

# matrix multiplication; the product u %*% v is computed, product
# placed in w

# note again:  mmul() call will be executed by each thread

mmul <- function(u,v,w) {
   require(parallel)
   # decide which rows of u this thread will work on
   myidxs <- splitIndices(nrow(u),myinfo$nwrkrs)[[myinfo$id]]
   # multiply this thread's part of u with v, placing the product in the
   # corresponding part of w
   w[myidxs,] <- u[myidxs,] \%*\% v[,]
   invisible(0)  
}

# create test matrices
mgrmakevar(c2,"a",6,2)
mgrmakevar(c2,"b",2,6)
mgrmakevar(c2,"c",6,6)
# give them values
a[,] <- 1:12
b[,] <- 1  # all 1s
clusterExport(c2,"mmul")  # send mmul() to the threads
clusterEvalQ(c2,mmul(a,b,c)) # run the threads
c[,]  # check results
}


