# Generated by using Rcpp::compileAttributes() -> do not edit by hand
# Generator token: 10BE3573-1514-4C36-9D1C-5A225CD40393

#' @title Perform Non-negative Matrix Factorization
#' @description
#' Regularly, Non-negative Matrix Factorization (NMF) is factorizes input
#' matrix \eqn{X} into low rank matrices \eqn{W} and \eqn{H}, so that
#' \eqn{X \approx WH}. The objective function can be stated as
#' \eqn{\arg\min_{W\ge0,H\ge0}||X-WH||_F^2}. In practice, \eqn{X} is usually
#' regarded as a matrix of \eqn{m} features by \eqn{n} sample points. And the
#' result matrix \eqn{W} should have the dimensionality of \eqn{m \times k} and
#' H with \eqn{n \times k} (transposed).
#' This function wraps the algorithms implemented in PLANC library to solve
#' NMF problems. Algorithms includes Alternating Non-negative Least Squares
#' with Block Principal Pivoting (ANLS-BPP), Alternating Direction Method of
#' Multipliers (ADMM), Hierarchical Alternating Least Squares (HALS), and
#' Multiplicative Update (MU).
#' @param x Input matrix for factorization. Can be either dense or sparse.
#' @param k Integer. Factor matrix rank.
#' @param niter Integer. Maximum number of NMF interations.
#' @param algo Algorithm to perform the factorization, choose from "anlsbpp",
#' "admm", "hals" or "mu". See detailed sections.
#' @param nCores The number of parallel tasks that will be spawned. Only applies to anlsbpp.
#' Default \code{2}
#' @param Winit Initial left-hand factor matrix, must be of size m x k.
#' @param Hinit Initial right-hand factor matrix, must be of size n x k.
#' @returns A list with the following elements:
#' \itemize{
#'  \item{\code{W} - the result left-hand factor matrix}
#'  \item{\code{H} - the result right hand matrix.}
#'  \item{\code{objErr} - the objective error of the factorization.}
#' }
#' @references
#' Ramakrishnan Kannan and et al., A High-Performance Parallel Algorithm for
#' Nonnegative Matrix Factorization, PPoPP '16, 2016, 10.1145/2851141.2851152
nmf <- function(x, k, niter = 30L, algo = "anlsbpp", nCores = 2L, Winit = NULL, Hinit = NULL) {
    .Call(`_RcppPlanc_nmf`, x, k, niter, algo, nCores, Winit, Hinit)
}

#' Perform Symmetric Non-negative Matrix Factorization
#'
#' Symmetric input matrix \eqn{X} of size \eqn{n \times n} is required. Two
#' approaches are provided. Alternating Non-negative Least Squares Block
#' Principal Pivoting algorithm (ANLSBPP) with symmetric regularization, where
#' the objective function is set to be \eqn{\arg\min_{H\ge0,W\ge0}||X-WH||_F^2+
#' \lambda||W-H||_F^2}, can be run with \code{algo = "anlsbpp"}.
#' Gaussian-Newton algorithm, where the objective function is set to be
#' \eqn{\arg\min_{H\ge0}||X-H^\mathsf{T}H||_F^2}, can be run with \code{algo =
#' "gnsym"}. In the objectives, \eqn{W} is of size \eqn{n \times k} and \eqn{H}
#' is of size \eqn{k \times n}. The returned results will all be
#' \eqn{n \times k}.
#'
#' @param x Input matrix for factorization. Must be symmetric. Can be either
#' dense or sparse.
#' @param k Integer. Factor matrix rank.
#' @param niter Integer. Maximum number of symNMF interations.
#' Default \code{30}
#' @param lambda Symmetric regularization parameter. Must be
#' non-negative. Default \code{0.0} uses the square of the maximum value in
#' \code{x}.
#' @param algo Algorithm to perform the factorization, choose from "gnsym" or
#' "anlsbpp". Default \code{"gnsym"}
#' @param nCores The number of parallel tasks that will be spawned. Only applies to anlsbpp.
#' Default \code{2}
#' @param Hinit Initial right-hand factor matrix, must be of size n x k.
#' Default \code{NULL}.
#' @returns A list with the following elements:
#' \itemize{
#'  \item{\code{W} - the result left-hand factor matrix, non-empty when using
#'  \code{"anlsbpp"}}
#'  \item{\code{H} - the result right hand matrix.}
#'  \item{\code{objErr} - the objective error of the factorization.}
#' }
#' @references
#' Srinivas Eswar and et al., Distributed-Memory Parallel Symmetric Nonnegative
#' Matrix Factorization, SC '20, 2020, 10.5555/3433701.3433799
symNMF <- function(x, k, niter = 30L, lambda = 0.0, algo = "gnsym", nCores = 2L, Hinit = NULL) {
    .Call(`_RcppPlanc_symNMF`, x, k, niter, lambda, algo, nCores, Hinit)
}

.bppinmf_h5dense <- function(filenames, dataPath, k, nCores, lambda, niter, verbose = TRUE, Hinit = NULL, Vinit = NULL, Winit = NULL) {
    .Call(`_RcppPlanc_bppinmf_h5`, filenames, dataPath, k, nCores, lambda, niter, verbose, Hinit, Vinit, Winit)
}

.bppinmf_h5sparse <- function(filenames, valuePath, rowindPath, colptrPath, nrow, ncol, k, nCores, lambda, niter, verbose = TRUE, Hinit = NULL, Vinit = NULL, Winit = NULL) {
    .Call(`_RcppPlanc_bppinmf_h5sp`, filenames, valuePath, rowindPath, colptrPath, nrow, ncol, k, nCores, lambda, niter, verbose, Hinit, Vinit, Winit)
}

.bppinmf <- function(objectList, k, nCores, lambda = 5, niter = 30L, verbose = TRUE, Hinit = NULL, Vinit = NULL, Winit = NULL) {
    .Call(`_RcppPlanc_bppinmf`, objectList, k, nCores, lambda, niter, verbose, Hinit, Vinit, Winit)
}

.onlineINMF_h5dense <- function(filenames, dataPaths, k, nCores, lambda, maxEpoch = 5L, minibatchSize = 5000L, maxHALSIter = 1L, permuteChunkSize = 1000L, verbose = TRUE) {
    .Call(`_RcppPlanc_onlineINMF_h5dense`, filenames, dataPaths, k, nCores, lambda, maxEpoch, minibatchSize, maxHALSIter, permuteChunkSize, verbose)
}

.onlineINMF_h5sparse <- function(filenames, valuePaths, rowindPaths, colptrPaths, nrows, ncols, k, nCores, lambda, maxEpoch = 5L, minibatchSize = 5000L, maxHALSIter = 1L, permuteChunkSize = 1000L, verbose = TRUE) {
    .Call(`_RcppPlanc_onlineINMF_h5sparse`, filenames, valuePaths, rowindPaths, colptrPaths, nrows, ncols, k, nCores, lambda, maxEpoch, minibatchSize, maxHALSIter, permuteChunkSize, verbose)
}

.onlineINMF <- function(objectList, k, nCores, lambda, maxEpoch = 5L, minibatchSize = 5000L, maxHALSIter = 1L, permuteChunkSize = 1000L, verbose = TRUE) {
    .Call(`_RcppPlanc_onlineINMF`, objectList, k, nCores, lambda, maxEpoch, minibatchSize, maxHALSIter, permuteChunkSize, verbose)
}

.onlineINMF_withInitial <- function(objectList, Hinit, Vinit, Winit, Ainit, Binit, objectListNew, k, nCores, lambda, maxEpoch = 5L, minibatchSize = 5000L, maxHALSIter = 1L, permuteChunkSize = 1000L, verbose = TRUE) {
    .Call(`_RcppPlanc_onlineINMF_withInitial`, objectList, Hinit, Vinit, Winit, Ainit, Binit, objectListNew, k, nCores, lambda, maxEpoch, minibatchSize, maxHALSIter, permuteChunkSize, verbose)
}

.onlineINMF_project <- function(objectList, Winit, objectListNew, k, nCores, lambda) {
    .Call(`_RcppPlanc_onlineINMF_project`, objectList, Winit, objectListNew, k, nCores, lambda)
}

.onlineINMF_h5dense_withInitial <- function(filenames, dataPaths, filenamesNew, dataPathsNew, Hinit, Vinit, Winit, Ainit, Binit, k, nCores, lambda, maxEpoch = 5L, minibatchSize = 5000L, maxHALSIter = 1L, permuteChunkSize = 1000L, verbose = TRUE) {
    .Call(`_RcppPlanc_onlineINMF_h5dense_withInitial`, filenames, dataPaths, filenamesNew, dataPathsNew, Hinit, Vinit, Winit, Ainit, Binit, k, nCores, lambda, maxEpoch, minibatchSize, maxHALSIter, permuteChunkSize, verbose)
}

.onlineINMF_project_h5dense <- function(filenames, dataPaths, filenamesNew, dataPathsNew, Winit, k, nCores, lambda) {
    .Call(`_RcppPlanc_onlineINMF_project_h5dense`, filenames, dataPaths, filenamesNew, dataPathsNew, Winit, k, nCores, lambda)
}

.onlineINMF_project_h5sparse <- function(filenames, valuePaths, rowindPaths, colptrPaths, nrows, ncols, filenamesNew, valuePathsNew, rowindPathsNew, colptrPathsNew, nrowsNew, ncolsNew, Winit, k, nCores, lambda) {
    .Call(`_RcppPlanc_onlineINMF_project_h5sparse`, filenames, valuePaths, rowindPaths, colptrPaths, nrows, ncols, filenamesNew, valuePathsNew, rowindPathsNew, colptrPathsNew, nrowsNew, ncolsNew, Winit, k, nCores, lambda)
}

.onlineINMF_h5sparse_withInitial <- function(filenames, valuePaths, rowindPaths, colptrPaths, nrows, ncols, filenamesNew, valuePathsNew, rowindPathsNew, colptrPathsNew, nrowsNew, ncolsNew, Hinit, Vinit, Winit, Ainit, Binit, k, nCores, lambda, maxEpoch = 5L, minibatchSize = 5000L, maxHALSIter = 1L, permuteChunkSize = 1000L, verbose = TRUE) {
    .Call(`_RcppPlanc_onlineINMF_h5sparse_withInitial`, filenames, valuePaths, rowindPaths, colptrPaths, nrows, ncols, filenamesNew, valuePathsNew, rowindPathsNew, colptrPathsNew, nrowsNew, ncolsNew, Hinit, Vinit, Winit, Ainit, Binit, k, nCores, lambda, maxEpoch, minibatchSize, maxHALSIter, permuteChunkSize, verbose)
}

.uinmf_rcpp <- function(objectList, unsharedList, whichUnshared, k, nCores, lambda, niter, verbose) {
    .Call(`_RcppPlanc_uinmf_rcpp`, objectList, unsharedList, whichUnshared, k, nCores, lambda, niter, verbose)
}

.uinmf_h5dense <- function(filenames, dataPaths, unsharedFilenames, unsharedDataPaths, whichUnshared, k, nCores, lambda, niter, verbose) {
    .Call(`_RcppPlanc_uinmf_h5dense`, filenames, dataPaths, unsharedFilenames, unsharedDataPaths, whichUnshared, k, nCores, lambda, niter, verbose)
}

.uinmf_h5sparse <- function(filenames, rowindPaths, colptrPaths, valuePaths, nrows, ncols, unsharedFilenames, unsharedRowindPaths, unsharedColptrPaths, unsharedValuePaths, unsharedNrows, unsharedNcols, whichUnshared, k, nCores, lambda, niter, verbose) {
    .Call(`_RcppPlanc_uinmf_h5sparse`, filenames, rowindPaths, colptrPaths, valuePaths, nrows, ncols, unsharedFilenames, unsharedRowindPaths, unsharedColptrPaths, unsharedValuePaths, unsharedNrows, unsharedNcols, whichUnshared, k, nCores, lambda, niter, verbose)
}

.openblaspthreadoff <- function(libloc) {
    invisible(.Call(`_RcppPlanc_openblas_pthread_off`, libloc))
}

.openblaspthreadon <- function(libloc) {
    invisible(.Call(`_RcppPlanc_openblas_pthread_on`, libloc))
}

#' Block Principal Pivoted Non-Negative Least Squares
#'
#' Use the BPP algorithm to get the nonnegative least squares solution. Regular
#' NNLS problem is described as optimizing \eqn{\min_{x\ge0}||CX - B||_F^2}
#' where \eqn{C} and \eqn{B} are given and \eqn{X} is to be solved.
#' \code{bppnnls} takes \eqn{C} and \eqn{B} as input. \code{bppnnls_prod} takes
#' \eqn{C^\mathsf{T}C} and \eqn{C^\mathsf{T}B} as
#' input to directly go for the intermediate step of BPP algorithm. This can be
#' useful when the dimensionality of \eqn{C} and \eqn{B} is large while
#' pre-calculating \eqn{C^\mathsf{T}C} and \eqn{C^\mathsf{T}B} is cheap.
#'
#' @param C Input dense \eqn{C} matrix
#' @param B Input \eqn{B} matrix of either dense or sparse form
#' @param nCores The number of parallel tasks that will be spawned.
#' Default \code{2}
#' @returns The calculated solution matrix in dense form.
#' @rdname bppnnls
#' @examples
#' set.seed(1)
#' C <- matrix(rnorm(250), nrow = 25)
#' B <- matrix(rnorm(375), nrow = 25)
#' res1 <- bppnnls(C, B)
#' dim(res1)
#' res2 <- bppnnls_prod(t(C) %*% C, t(C) %*% B)
#' all.equal(res1, res2)
bppnnls <- function(C, B, nCores = 2L) {
    .Call(`_RcppPlanc_bppnnls`, C, B, nCores)
}

#' @param CtC The \eqn{C^\mathsf{T}C} matrix, see description.
#' @param CtB The \eqn{C^\mathsf{T}B} matrix, see description.
#' @param nCores The number of parallel tasks that will be spawned.
#' Default \code{2}
#' @rdname bppnnls
bppnnls_prod <- function(CtC, CtB, nCores = 2L) {
    .Call(`_RcppPlanc_bppnnls_prod`, CtC, CtB, nCores)
}

