% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EstSAEMspatial_USER.R
\name{SAEM.sclm}
\alias{SAEM.sclm}
\title{Censored Spatial Model Estimation via SAEM Algorithm}
\usage{
SAEM.sclm(y, x, cens, LI, LS, coords, init.phi, init.nugget,
  type = "exponential", kappa = 0, lower = c(0.01, 0.01), upper = c(30,
  30), MaxIter = 300, M = 20, pc = 0.25, error = 1e-05,
  show.SE = TRUE)
}
\arguments{
\item{y}{vector of responses.}

\item{x}{design matrix.}

\item{cens}{vector of censoring indicators. For each observation: \code{1} if censored/missing
and \code{0} otherwise.}

\item{LI}{lower limit of detection. For each observation: if non-censored \code{=y},
if left-censored/missing \code{=-Inf}, or \code{=LOD} if right/interval-censored.}

\item{LS}{upper limit of detection. For each observation: if non-censored \code{=y},
if right-censored/missing \code{=Inf}, or \code{=LOD} if left/interval-censored.}

\item{coords}{2D spatial coordinates.}

\item{init.phi}{initial value for the spatial scaling parameter.}

\item{init.nugget}{initial value for the nugget effect parameter.}

\item{type}{type of spatial correlation function: '\code{exponential}', '\code{gaussian}',
'\code{matern}', and '\code{pow.exp}' for exponential, gaussian, matern, and power exponential, respectively.}

\item{kappa}{parameter for all spatial correlation functions. See \code{\link{CovMat}}.}

\item{lower, upper}{vectors of lower and upper bounds for the optimization method. If unspecified, the default is
\code{c(0.01,0.01)} for lower and \code{c(30,30)} for upper.}

\item{MaxIter}{maximum number of iterations of the SAEM algorithm. By default \code{=300}.}

\item{M}{number of Monte Carlo samples for stochastic approximation. By default \code{=20}.}

\item{pc}{percentage of iterations of the SAEM algorithm with no-memory. By default \code{=0.25}.}

\item{error}{maximum convergence error. By default \code{=1e-5}.}

\item{show.SE}{\code{TRUE} or \code{FALSE}. It indicates if the standard errors should be estimated. By default \code{=TRUE}.}
}
\value{
The function returns an object of class \code{sclm} which is a list given by:

\item{Theta}{estimated parameters in all iterations, \eqn{\theta = (\beta, \sigma^2, \phi, \tau^2)}.}
\item{theta}{final estimation of \eqn{\theta = (\beta, \sigma^2, \phi, \tau^2)}.}
\item{beta}{estimated \eqn{\beta}.}
\item{sigma2}{estimated \eqn{\sigma^2}.}
\item{phi}{estimated \eqn{\phi}.}
\item{tau2}{estimated \eqn{\tau^2}.}
\item{EY}{stochastic approximation of the first moment for the truncated normal distribution.}
\item{EYY}{stochastic approximation of the second moment for the truncated normal distribution.}
\item{SE}{vector of standard errors of \eqn{\theta = (\beta, \sigma^2, \phi, \tau^2)}.}
\item{InfMat}{observed information matrix.}
\item{loglik}{log-likelihood for the SAEM method.}
\item{AIC}{Akaike information criterion.}
\item{BIC}{Bayesian information criterion.}
\item{Iterations}{number of iterations needed to converge.}
\item{ptime}{processing time.}
\item{range}{the effective range.}
}
\description{
This function returns the maximum likelihood (ML) estimates of the unknown parameters in Gaussian spatial models
with censored/missing responses via the SAEM algorithm. It supports left, right,
interval, or missing values in the dependent variable. It also computes the observed information
matrix using the method developed by \insertCite{louis1982finding;textual}{RcppCensSpatial}.
}
\details{
The spatial Gaussian model is given by

\eqn{Y = X\beta + \xi},

where \eqn{Y} is the \eqn{n x 1} vector of response, \eqn{X} is the \eqn{n x q} design matrix,
\eqn{\beta} is the \eqn{q x 1} vector of regression coefficients to be estimated, and \eqn{\xi}
is the error term which is normally distributed with zero-mean and covariance matrix
\eqn{\Sigma=\sigma^2 R(\phi) + \tau^2 I_n}. We assume that \eqn{\Sigma} is non-singular and
\eqn{X} has full rank \insertCite{diggle2007springer}{RcppCensSpatial}.

The estimation process was performed via the SAEM \insertCite{delyon1999convergence}{RcppCensSpatial} algorithm.
The spatial SAEM algorithm was previously proposed by \insertCite{lachos2017influence;textual}{RcppCensSpatial}
and \insertCite{ordonez2018geostatistical;textual}{RcppCensSpatial} and is available in package \code{CensSpatial}.
The difference between this package to \code{CensSpatial} is that the random observations are sampled
through the slice sampling algorithm available in package \code{relliptical} and the optimization procedure
by the \code{roptim} package.

This model is also a particular case of the Spatio-temporal model defined by \insertCite{valeriano2021likelihood;textual}{RcppCensSpatial},
when the number of temporal observations is equal to one. The computing codes of the Spatio-temporal
SAEM algorithm are available in the package \code{StempCens}.
}
\note{
The SAEM final estimates correspond to the estimates obtained at the last iteration of the
algorithm.

To fit a regression model for non-censored data, just set \code{cens} as a vector of zeros.

Functions \code{print}, \code{summary}, and \code{plot} work for objects of class \code{sclm}.
}
\examples{
# Simulated example: 10\% of right-censored observations
n = 50   # Test with another values for n
set.seed(1000)
coords = round(matrix(runif(2*n,0,15),n,2),5)
x = cbind(rbinom(n,1,0.50), rnorm(n), rnorm(n))
data = rCensSp(c(1,4,-2),2,3,0.50,x,coords,"right",0.10,0,"matern",2)

fit = SAEM.sclm(y=data$yobs, x=data[,7:9], cens=data$cens, LI=data$LI,
             LS=data$LS, coords=data[,5:6], init.phi=2, init.nugget=1,
             type="matern", kappa=2, MaxIter = 20, error=1e-4)
summary(fit)

\donttest{
# Simulated example: censored and missing observations
n = 200
set.seed(123)
coords = round(matrix(runif(2*n,0,20),n,2),5)
x = cbind(1, rnorm(n), rexp(n))
data = rCensSp(c(1,4,-1),2,4,0.50,x,coords,"left",0.10,0,"exponential",0)
data$yobs[c(10,20)] = NA;   data$cens[c(10,20)] = 1
data$LI[c(10,20)] = -Inf;   data$LS[c(10,20)] = Inf

fit2 = SAEM.sclm(y=data$yobs, x=data[,7:9], cens=data$cens, LI=data$LI,
              LS=data$LS, coords=data[,5:6], init.phi=2, init.nugget=1,
              type="exponential", MaxIter = 300, error=1e-4)
fit2$theta  # Estimates
fit2$SE     # Standard error
fit2$InfMat # Information matrix
plot(fit2)}
}
\references{
\insertAllCited
}
\seealso{
\code{\link{EM.sclm}}, \code{\link{MCEM.sclm}}, \code{\link{predict.sclm}}
}
\author{
Katherine L. Valeriano, Alejandro Ordonez, Christian E. Galarza and Larissa A. Matos.
}
