/* 
 *  IMS Open Corpus Workbench (CWB)
 *  Copyright (C) 1993-2006 by IMS, University of Stuttgart
 *  Copyright (C) 2007-     by the respective contributers (see file AUTHORS)
 * 
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2, or (at your option) any later
 *  version.
 * 
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
 *  Public License for more details (in the file "COPYING", or available via
 *  WWW at http://www.gnu.org/copyleft/gpl.html).
 */

#ifndef _cqp_corpmanag_h_
#define _cqp_corpmanag_h_


#include "../cl/corpus.h"
#include "../cl/bitfields.h"
#include "cqp.h"
#include "context_descriptor.h"



/**
 * The FieldType object represents the fields (or 'anchor points') of a subcorpus (inc query result).
 *
 * NoField is always the last field (so it can be used to determine range of field type codes)
 * (the mnemonic is NoField = "no field" = "number of fields")
 */
typedef enum _field_type {
  MatchField = 0, MatchEndField, TargetField, KeywordField, NoField
} FieldType;

/* type definitions copied from datastructs.h, Mon Apr 11 13:00:59 1994 (oli) */

/* definition of the corpus list */



/**
 * Underlying enumeration for CorpusType.
 *
 * Gives an identifier for each "type" of corpus CQP can deal with.
 */
enum corpus_type { UNDEF,    /**< undefined status.                        */
                   SYSTEM,   /**< system corpus, ie registered corpus      */
                   SUB,      /**< subcorpus which was generated by a query.*/
                   TEMP,     /**< temporary subcorpus, deleted after query */
                   ALL
                 };

/**
 * The CorpusType object.
 */
typedef enum corpus_type CorpusType;


/**
 * Creates an enumeration for lowercase vs. uppercase mode.
 */
enum case_mode {LOWER, UPPER};

/**
 * The Range object represents a range of corpus positions.
 */
typedef struct _Range
{
  int        start;     /**< start position of the corpus interval  */
  int        end;       /**< end position of the corpus interval    */
} Range;

/**
 * The CorpusList object records information on a corpus that CQP recognises.
 */
typedef struct cl {
  char            *name;         /**< corpus name                                */
  char            *mother_name;  /**< name of the original corpus.               */
  int              mother_size;  /**< size (nr tokens) of mother                 */
  char            *registry;     /**< registry directory of the original corpus. */
  char            *abs_fn;       /**< absolute file name                         */
  enum corpus_type type;         /**< type of the corpus.                        */

  char            *local_dir;    /**< for unloaded subcorpora, this is the directory
                                      where the subcorpus is stored on disk; required
                                      for delayed loading (in ensure_corpus_size()) */

  char            *query_corpus; /**< name of the corpus the query was run on    */
  char            *query_text;   /**< the query text proper                      */

  Boolean          saved;        /**< is the corpus saved (=stored on disk)?     */
  Boolean          loaded;       /**< is the corpus loaded?                      */
  Boolean          needs_update; /**< True iff saved & loaded & contents changed */

  Corpus          *corpus;       /**< associated corpus data structure.          */

  Range           *range;        /**< an array of corpus intervals.              */
  int              size;         /**< number of intervals.                       */
  int             *sortidx;      /**< sorting index for intervals                */
  int             *targets;      /**< list of targets                            */
  int             *keywords;     /**< one keyword, for each concordance line     */

  ContextDescriptor *cd;         /**< additional attributes to print -- only
                                      for ``SYSTEM'' corpora                     */

  struct cl       *next;         /**< Next corpus on CQP's linked list.          */
} CorpusList;

/**
 * Global pointer to the "current" corpus.
 */
CorpusList *current_corpus;

/**
 * Global pointer to the head of CQP's linked list of corpora.
 */
CorpusList *corpuslist;

/* ---------------------------------------------------------------------- */

/* this should usually be provided by a FIELD or FIELDLABEL token recognised by flex,
   but we _may_ need it for the CQi server */
FieldType field_name_to_type(char *name);
char *field_type_to_name(FieldType field);
int NrFieldValues(CorpusList *cl, FieldType ft);

/* ---------------------------------------------------------------------- */

/*
 * functions associated with the corpus list, 
 * copied from symtab.[ch] 
 * Mon Apr 11 13:40:54 1994 (oli)
 */


void init_corpuslist(void);

void free_corpuslist(void);

/**
 * finds the pointer to the corpus with name s (strcmp), no
 * case conversion is done. NULL is returned when the corpus
 * is not yet present. if "type" is UNDEF, it returns the first 
 * corpus with matching name, otherwise the returned corpus 
 * has the type "type"
 */
CorpusList *findcorpus(char *s, enum corpus_type type, int try_recursive_search);

void dropcorpus(CorpusList *cl);

/* duplicate_corpus creates a copy of an existing corpus
 * and casts its type to SUB. The new corpus is given the
 * name "new_name". If a subcorpus of that name is already
 * present, NULL is retured if force_overwrite is False. if
 * force_overwrite is True, the old corpus is discarded.
 */
CorpusList *duplicate_corpus(CorpusList *cl,
                             char *new_name,
                             Boolean force_overwrite);

/* make_temp_corpus makes a copy of the corpus in *cl
 * into a corpus of type "TEMP" with name "new_name".
 * If a temporary corpus with that name already exists,
 * it is overwritten.
 */
CorpusList *make_temp_corpus(CorpusList *cl,
                             char *new_name);

/* assign_temp_to_sub assigns the temporary corpus in *tmp
 * to a "real" subcorpus with name "subname". If such a 
 * subcorpus already exists, it is overwritten. The temporary
 * corpus is deleted afterwards. The return value is the new
 * subcorpus (which may be equal to tmp, but not necessarily
 */
CorpusList *assign_temp_to_sub(CorpusList *tmp, char *subname);


/* drop_temp_corpora clears the list of corpora from all
 * temporary stuff.
 */
void drop_temp_corpora(void);


/* access_corpus makes sure that the data for corpus in
 * "cl" is loaded and accessible.
 */
Boolean access_corpus(CorpusList *cl);

/* change_corpus sets the current corpus to the corpus with
 * name "name", first searching SUB corpora, then searching
 * SYSTEM corpora.
 */
Boolean change_corpus(char *name, Boolean silent);

void check_available_corpora(enum corpus_type ct);

/* check whether {corpusname} is a syntactically valid as a query result name */
Boolean valid_subcorpus_name(char *corpusname);

/* check whether {corpusname} is fully qualified (with name of mother corpus); does not imply syntatic validity */
Boolean is_qualified(char *corpusname);

char *split_subcorpus_name(char *corpusname, char *mother_name);

Boolean save_subcorpus(CorpusList *cl, char *fname);

void save_unsaved_subcorpora();

/* Iterate through list of corpora */

CorpusList *FirstCorpusFromList();
CorpusList *NextCorpusFromList(CorpusList *cl);


int set_current_corpus(CorpusList *cp, int force);

int set_current_corpus_name(char *name, int force);

Boolean touch_corpus(CorpusList *cp);

/* IO/Functions */

void show_corpora_files(enum corpus_type type);

#endif
