#' Download MODIS images from a search list
#'
#' \code{modDownSearch} downloads the images from a list of uniform resource 
#' locator (URL) generated by the \code{\link{modSearch}} function from NASA’s
#' `EartData' plataform. The images are saved as GTiff files in the 
#' \code{AppRoot} directory.
#'
#' \code{modDownSearch} is able to download MODIS Terra and Aqua products.
#' These products are published in the 
#' \href{https://earthdata.nasa.gov}{`EarthData' Platform}.
#' The platform is supported by the Earth Observing System Data and Information
#' System (EODIS) and managed NASA’s Earth Science Data Systems (ESDS).
#' \code{modDownSearch} requires credentianls from an `EarthData' account to
#' access the NASA’s web data service, which can be obtained 
#' \href{https://urs.earthdata.nasa.gov/users/new}{here}.
#'
#' @param searchres the output from the \code{\link{modSearch}} function.
#' @param username NASA’s `EarthData' username.
#' @param password NASA’s `EarthData' password.
#' @param AppRoot the directory where the images will be saved.
#' @param overwrite logical argument. If \code{TRUE}, overwrites the existing
#' images with the same name.
#' @param ... argument for nested functions:
#'
#' @examples
#' \dontrun{
#' # load a spatial polygon object of Navarre
#' data(ex.navarre)
#' mList <- modSearch(product = "MYD13A2",
#'                    startDate = as.Date("01-01-2011", "%d-%m-%Y"),
#'                    endDate = as.Date("31-12-2013", "%d-%m-%Y"),
#'                    collection = 6,
#'                    extent = ex.navarre)
#' head(mList)
#' # download the first image in mList
#' src <- paste0(tempdir(),"/Path_for_downloading_folder")
#' print(src)
#' src.mod <- file.path(src,"Modis","MYD13A2")
#' src.mod.hdf <- file.path(src.mod,"hdf")
#' modDownSearch(mList[1], 
#'               username = "user", 
#'               password = "pass",
#'               AppRoot = src.mod.hdf)
#' # download all images in mList
#' modDownSearch(mList, 
#'               username = "user", 
#'               password = "pass",
#'               AppRoot = src.mod.hdf)
#' }
modDownSearch<-function(searchres, AppRoot, username = NULL,password = NULL,overwrite=FALSE,...){
  arg<-list(...)
  if(is.null(username)|is.null(password)){
    stop("Username and/or password not defined!")
  }
  AppRoot<-pathWinLx(AppRoot)
  dir.create(AppRoot,showWarnings = FALSE,recursive = TRUE)
  message(paste0("Downloading the images in: ",AppRoot))
  for(l in searchres){
    if(file.exists(paste0(AppRoot,"/",basename(l)))&&overwrite){
      file.remove(paste0(AppRoot,"/",basename(l)))
    }

    if(!file.exists(paste0(AppRoot,"/",basename(l)))){
      c.handle = new_handle()
      handle_setopt(c.handle,
                    referer=paste0("https://",domain(l),"/"),
                    useragent = getRGISToolsOpt("USERAGENT"),
                    followlocation = TRUE ,
                    autoreferer = TRUE ,
                    username=username,
                    password=password)
      message(paste0("Downloading ",basename(l)," image."))
      curl_download(l, destfile=paste0(AppRoot,"/",basename(l)),handle = c.handle)
    }else if(overwrite){
      c.handle = new_handle()
      handle_setopt(c.handle,
                    referer=paste0("https://",domain(l),"/"),
                    useragent = getRGISToolsOpt("USERAGENT"),
                    followlocation = TRUE ,
                    autoreferer = TRUE ,
                    username=username,
                    password=password)
      file.remove(paste0(AppRoot,"/",basename(l)))
      message(paste0("Downloading ",basename(l)," image."))
      curl_download(l, destfile=paste0(AppRoot,"/",basename(l)),handle = c.handle)
    }else{
      message("File already exists.")
    }
  }
  message(paste0("The images have been downloaded and saved on HDD. \nFile path: ",AppRoot))
}

