% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MCMC.R
\name{MCMC_OAAT}
\alias{MCMC_OAAT}
\title{Adaptive One-At-A-Time Metropolis sampler}
\usage{
MCMC_OAAT(
  logPdf,
  x0,
  s0 = 0.05 * (abs(x0) + 0.1),
  nTheta = length(x0),
  nAdapt = 50,
  nCycles = 20,
  minMoveRate = 0.2,
  maxMoveRate = 0.5,
  downMult = 0.9,
  upMult = 1.1,
  ...
)
}
\arguments{
\item{logPdf}{function, evaluating the log-density of the distribution to sample from (up to a proportionality constant).
logPdf can return either a single numeric value, interpreted as the target log-pdf,
or a list containing components named 'logPosterior', 'logLikelihood' and 'logPrior'.}

\item{x0}{numeric vector, starting point.}

\item{s0}{numeric vector, starting jump standard deviations.}

\item{nTheta}{integer>0, size of the "theta" part of x0, i.e. components that represent the
model parameters rather than structural errors parameters (gamma).
This is used to speed-up the sampler by avoiding running the model for gamma components.
nTheta=length(x0) (default) implies no attempt at speeding up.}

\item{nAdapt}{integer > 1, number of iterations before adapting the jump standard deviations.}

\item{nCycles}{integer > 1, number of adaption cycles. Total number of iterations is hence equal to nAdapt*nCycles.
nCycles=1 leads to a non-adaptive one-at-a-time Metropolis sampler.}

\item{minMoveRate}{numeric in (0;1), lower bound for the desired move rate interval.}

\item{maxMoveRate}{numeric in (0;1), upper bound for the desired move rate interval.}

\item{downMult}{numeric in (0;1), multiplication factor used to decrease the jump standard deviation when move rate is too low.}

\item{upMult}{numeric (>1, avoid 1/downMult) multiplication factor used to increase the jump standard deviations when move rate is too high.}

\item{...}{other arguments passed to function logPdf}
}
\value{
A list with the following components:
    \item{samples}{data frame, MCMC simulations.}
    \item{components}{data frame, corresponding values of the log-posterior, the log-prior and the log-likelihood.}
    \item{sjump}{numeric vector, the adapted jump standard deviations.}
}
\description{
An adaptive Metropolis sampler that updates the parameter vector one component at a time
using a 1-dimensional jump. This allows easily adapting the jump standard deviation for each
component in order to comply with a specified move rate interval.
}
\examples{
# Define a 2-dimensional target log-pdf
logPdf <- function(x){
    p1=log(0.6*dnorm(x[1],0,1)+0.4*dnorm(x[1],2,0.5))
    p2=log(dlnorm(x[2],0,1))
    return(p1+p2)
}
# Sample from it
mcmc=MCMC_OAAT(logPdf,c(1,1))
plot(mcmc$samples)
}
