#' @title Vector Autoregressive Model
#' @description Vector autoregressive model. This is a regression method used to
#' establish the temporal relationship between the original time series and
#' the modes generated by quantum walks.
#' @usage qwdap.var(in_data, data_range, plotting)
#' @param in_data a 'QWMS' object, which includes the target series and the
#' selected modes which can be obtained from modes selection.
#' @param data_range the range of the train samples.
#' @param plotting whether to plot.
#'
#' @return a 'QWMODEL' object which includes the information of regression analysis.
#' @import MTS
#' @importFrom graphics lines legend
#' @export qwdap.var
#'
#' @examples
#' data("traffic.n1")
#' res.var <- qwdap.var(traffic.n1,c(1,500))
#' 
qwdap.var<-function(in_data, data_range, plotting = FALSE){
  if(class(in_data)!='QWMS'){
    stop("The 'in_data' is not a 'QWMS' object.")
  }
  if(!is.vector(data_range)||!is.numeric(data_range)||length(data_range)<2){
    stop("The parameter 'data_range' is error.")
  }
  # pre combine
  co_data <- cbind(in_data$real, in_data$ctqw)
  co_data <- subset(co_data, select = c(colnames(in_data$real), in_data$variate))
  indexes<-VARXorder(co_data[data_range[1]:data_range[2],1],
                     co_data[data_range[1]:data_range[2],-1],10)
  if(indexes$aicor[1]==0){
    indexes$aicor[1]=indexes$bicor[1]
  }
  ## 使用VARX
  res <- MTS::VARX(co_data[data_range[1]:data_range[2],1],
                   indexes$aicor[1],co_data[data_range[1]:data_range[2],-1],
                   indexes$aicor[2],output = F)
  if(plotting){
    plot(x=c(data_range[1]:data_range[2]),
         y=co_data[data_range[1]:data_range[2],1],
         type="l",col=1,xlab="index",ylab="value",lwd=1)
    cir<-c((data_range[2]-length(res$residuals[,1])):(data_range[2]-1))
    lines(x=cir,y=res$residuals[,1]+co_data[cir,1],type="l",col=2,lwd=1)
    legend("topleft", c("Actual series","Fitted series"), col = c(1,2),
           lwd = c(1), bg = "grey95", box.col = NA,
           cex = 0.8, inset = c(0.02, 0.03), ncol = 1)
  }
  res<-list(real = in_data$real, ctqw = co_data[,-1], index = in_data$index,
            method = "VAR",model=res)
  res<-structure(res,class="QWMODEL")
  return(res)
}
