#' EM Algorithm for QTL MIM
#'
#' Expectation-maximization algorithm for QTL multiple interval mapping.
#'
#' @param D.matrix matrix. The design matrix of QTL effects which is a
#' g*p matrix, where g is the number of possible QTL genotypes, and p
#' is the number of effects considered in the MIM model. The design
#' matrix can be easily generated by the function D.make().
#' @param cp.matrix matrix. The conditional probability matrix which
#' is an n*g matrix, where n is the number of individuals, and g is
#' the number of possible genotypes of QTLs. The conditional
#' probability matrix can be easily generated by the function Q.make().
#' @param y vector. An vector with n elements that contains the phenotype
#' values of individuals.
#' @param E.vector0 vevtor. The initial value for QTL effects. The
#' number of elements corresponds to the column dimension of the
#' design matrix. If E.vector0=NULL, the initial value will be 0
#' for all effects.
#' @param X matrix. The design matrix of the fixed factors except
#' QTL effects. It is an n*k matrix, where n is the number of
#' individuals, and k is the number of fixed factors. If X=NULL,
#' the matrix will be an n*1 matrix that all elements are 1.
#' @param beta0 vector. The initial value for effects of the fixed
#' factors. The number of elements corresponds the column dimension
#' of the fixed factor design matrix.  If beta0=NULL, the initial
#' value will be the average of y.
#' @param variance0 numeric. The initial value for variance. If
#' variance0=NULL, the initial value will be the variance of
#' phenotype values.
#' @param crit numeric. The convergence criterion of EM algorithm.
#' The E and M steps will be iterated until a convergence criterion
#' is satisfied.
#' @param stop numeric. The stopping criterion of EM algorithm.
#' The E and M steps will be stop when the iteration number reaches the
#' stop criterion, and it will treat the algorithm as failing to converge.
#' @param conv logical. If being False, it will ignore the inability to
#' converge, and output the last result in the process of EM algorithm
#' before the stopping criterion.
#' @param console logical. To decide whether the process of algorithm
#' will be shown in the R console or not.
#'
#' @return
#' \item{E.vector}{The QTL effects calculated by EM algorithm.}
#' \item{beta}{The effects of the fixed factors calculated by EM
#' algorithm.}
#' \item{variance}{The error variance calculated by EM algorithm.}
#' \item{PI.matrix}{The posterior probabilities matrix after the
#' process of EM algorithm.}
#' \item{log.likelihood}{The log likelihood value of this model.}
#' \item{LRT}{The LRT statistic of this model.}
#' \item{R2}{The coefficient of determination of this model. This
#' can be used as an estimate of heritability.}
#' \item{y.hat}{The fitted values of trait values calculated by
#' the estimated values from the EM algorithm.}
#' \item{iteration.number}{The iteration number of EM algorithm.}
#'
#' @export
#'
#' @references
#'
#' KAO, C.-H. and Z.-B. ZENG 1997 General formulas for obtaining the maximum
#' likelihood estimates and the asymptotic variance-covariance matrix in QTL
#' mapping when using the EM algorithm. Biometrics 53, 653-665.
#'
#' KAO, C.-H., Z.-B. ZENG and R. D. TEASDALE 1999 Multiple interval mapping
#' for Quantitative Trait Loci. Genetics 152: 1203-1216.
#'
#' @seealso
#' \code{\link[QTLEMM]{D.make}}
#' \code{\link[QTLEMM]{Q.make}}
#' \code{\link[QTLEMM]{EM.MIM2}}
#'
#' @examples
#' # load the example data
#' load(system.file("extdata", "exampledata.RDATA", package = "QTLEMM"))
#'
#' # run and result
#' D.matrix <- D.make(3, type = "RI", aa = c(1, 3, 2, 3), dd = c(1, 2, 1, 3), ad = c(1, 2, 2, 3))
#' cp.matrix <- Q.make(QTL, marker, geno, type = "RI", ng = 2)$cp.matrix
#' result <- EM.MIM(D.matrix, cp.matrix, y)
#' result$E.vector
EM.MIM <- function(D.matrix, cp.matrix, y, E.vector0 = NULL, X = NULL, beta0 = NULL,
                   variance0 = NULL, crit = 10^-5, stop = 1000, conv = TRUE, console = TRUE){

  if(is.null(D.matrix) | is.null(cp.matrix) | is.null(y)){
    stop("Input data is missing, please cheak and fix", call. = FALSE)
  }

  datatry <- try(y%*%cp.matrix%*%D.matrix, silent=TRUE)
  if(class(datatry)[1] == "try-error" | NA %in% D.matrix | NA %in% cp.matrix){
    stop("Input data error, please check your input data.", call. = FALSE)
  }

  Y <- y
  ind <- length(Y)
  g <- nrow(D.matrix)
  eff <- ncol(D.matrix)

  Y[is.na(Y)] <- mean(Y,na.rm = TRUE)

  E.vector <- E.vector0
  beta <- beta0
  variance <- variance0

  if(is.null(E.vector)){E.vector <- rep(0, eff)}
  if(is.null(X)){
    X <- matrix(1, ind, 1)
  } else if (is.vector(X)){
    X <- matrix(X, length(X), 1)
  }
  if(is.null(beta)){
    beta <- matrix(rep(mean(Y), ncol(X)), ncol(X), 1)
  } else if (is.numeric(beta)){
    beta <- matrix(rep(beta, ncol(X)), ncol(X), 1)
  }
  if(is.null(variance)){variance <- stats::var(Y)}
  if(!console[1] %in% c(0,1) | length(console) > 1){console <- TRUE}
  if(!conv[1] %in% c(0,1) | length(conv) > 1){conv <- TRUE}

  datatry <- try(D.matrix%*%E.vector, silent=TRUE)
  if(class(datatry)[1] == "try-error" | NA %in% E.vector){
    stop("Parameter E.vector0 error, please check and fix.", call. = FALSE)
  }

  datatry <- try(Y%*%X%*%beta, silent=TRUE)
  if(class(datatry)[1] == "try-error" | NA %in% X | NA %in% beta){
    stop("Parameter X or bata0 error, please check and fix.", call. = FALSE)
  }

  if(!is.numeric(variance) | length(variance) > 1 | min(variance) < 0){
    stop("Parameter variance0 error, please input a positive number.", call. = FALSE)
  }
  sigma <- sqrt(variance)

  if(!is.numeric(crit) | length(crit) > 1 | min(crit) < 0){
    stop("Parameter crit error, please input a positive number.", call. = FALSE)
  }

  Delta <- 1
  number <- 0

  if(length(colnames(D.matrix)) == ncol(D.matrix)){
    effectname <- colnames(D.matrix)
  }

  if(console){
    cat("number", "var", effectname, "\n", sep = "\t")
  }
  while (max(abs(Delta)) > crit & number < stop) {
    muji.matrix <- t(D.matrix%*%E.vector%*%matrix(1, 1, ind))+X%*%beta%*%matrix(1, 1, g)

    PI.matrix <- matrix(0, ind, g)
    for(j in 1:ind){
      P0 <- c()
      for(i in 1:g){
        P0[i] <- cp.matrix[j, i]*stats::dnorm(Y[j], muji.matrix[j, i], sigma)
      }
      if(sum(P0) != 0){P0 <- P0/sum(P0)
      } else {P0 <- rep(1/g, g)}
      PI.matrix[j, ] <- P0
    }

    r.vector=c()
    for(i in 1:eff){
      r01 <- t(Y-X%*%beta)%*%PI.matrix%*%D.matrix[, i]
      r02 <- matrix(1, 1, ind)%*%PI.matrix%*%(D.matrix[, i]*D.matrix[, i])
      r.vector[i] <- r01/r02
    }

    M.matrix <- matrix(0, eff, eff)
    for(i in 1:eff){
      for(j in 1:eff){
        if(i != j){
          M01 <- matrix(1, 1, ind)%*%PI.matrix%*%(D.matrix[, i]*D.matrix[, j])
          M02 <- matrix(1, 1, ind)%*%PI.matrix%*%(D.matrix[, i]*D.matrix[, i])
          M.matrix[i, j]=M01/M02
        }
      }
    }

    E.t <- r.vector-M.matrix%*%E.vector

    beta.t <- solve(t(X)%*%X)%*%t(X)%*%(Y-PI.matrix%*%D.matrix%*%E.t)

    V.matrix <- matrix(0, eff, eff)
    for(i in 1:eff){
      for(j in 1:eff){
        V.matrix[i, j] <- matrix(1, 1, ind)%*%PI.matrix%*%(D.matrix[, i]*D.matrix[, j])
      }
    }

    sigma.t <- sqrt((t(Y-X%*%beta.t)%*%(Y-X%*%beta.t)-t(Y-X%*%beta.t)%*%PI.matrix%*%D.matrix%*%E.t*2+t(E.t)%*%V.matrix%*%E.t)/ind)

    Delta <- E.t-E.vector
    if(NaN %in% Delta){
      break()
    }
    number <- number+1
    if(console){
      Ep <- round(E.t, 3)
      sp <- round(sigma.t^2, 3)
      cat(number, sp, Ep, "\n", sep = "\t")
    }

    E.vector <- E.t
    beta <- beta.t
    sigma <- sigma.t
  }

  PI.matrix <- matrix(0, ind, g)
  for(j in 1:ind){
    P0 <- c()
    for(i in 1:g){
      P0[i] <- cp.matrix[j, i]*stats::dnorm(Y[j], muji.matrix[j, i], sigma)
    }
    P0 <- P0/sum(P0)
    PI.matrix[j, ] <- P0
  }
  E.vector <- c(E.vector)
  names(E.vector) <- effectname
  colnames(PI.matrix) <- colnames(cp.matrix)

  variance <- sigma^2

  L0 <- c()
  L1 <- c()
  for(k in 1:nrow(cp.matrix)){
    L00 <- c()
    L01 <- c()
    for(m in 1:nrow(D.matrix)){
      L00[m] <- cp.matrix[k, m]*stats::dnorm(Y[k], mean(X%*%beta), sigma)
      L01[m] <- cp.matrix[k, m]*stats::dnorm(Y[k], mean(X%*%beta)+D.matrix[m, ]%*%E.vector, sigma)
    }
    L0[k] <- sum(L00)
    L1[k] <- sum(L01)
  }
  like0 <- sum(log(L0))
  like1 <- sum(log(L1))
  LRT <- 2*(like1-like0)
  y.hat <- PI.matrix%*%D.matrix%*%E.vector+X%*%beta
  r2 <- stats::var(y.hat)/stats::var(y)

  if(number == stop){
    if(conv){
      E.vector <- rep(0, length(E.vector))
      beta <- 0
      variance <- 0
      PI.matrix <- matrix(0, nrow(PI.matrix), ncol(PI.matrix))
      like1 <- -Inf
      LRT <- 0
      r2 <- 0
    }
    warning("EM algorithm fails to converge, please check the input data or adjust the convergence criterion and stopping criterion.")
  }

  result <- list(E.vector = E.vector, beta = as.numeric(beta), variance = as.numeric(variance),
                 PI.matrix = PI.matrix, log.likelihood = like1, LRT = LRT, R2 = r2, y.hat = y.hat, iteration.number = number)
  return(result)
}
