% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PAsso.R
\name{PAsso}
\alias{PAsso}
\title{Partial association analysis between ordinal responses after adjusting for a set of covariates}
\usage{
PAsso(
  responses,
  adjustments,
  data,
  uni.model = c("probit", "logit", "acat"),
  models = NULL,
  method = c("kendall", "pearson", "wolfsigma"),
  resids.type = c("surrogate", "sign", "general", "deviance"),
  jitter = c("latent", "uniform"),
  jitter.uniform.scale = c("probability", "response"),
  fitted.models = NULL,
  n_draws = 20,
  association = "partial",
  ...
)
}
\arguments{
\item{responses}{A string vector that specifies response variables. It requires to be equal
or greater than two variables in the data frame.}

\item{adjustments}{A string vector specifies covariates/confounders that need to
be adjusted.}

\item{data}{A data.frame including responses and adjustments.}

\item{uni.model}{A character string specifying the universal model setting for all
responses. Default \code{"logit"} refers to cumulative logit model. \code{"probit"}
refers to cumulative probit model. \code{"acat"} fits an adjacent categories regression model.}

\item{models}{A string vector contains default link functions of fitting models with
respect to each response variable. If \code{"models"} is missing or has any one of the model
unspecified, \code{"uni.model"} is used to specify same models for all responses automatically.
But, this argument has higher priority than the \code{"uni.model"} as long as the length of
\code{"models"} equals to the number of \code{"responses"}.}

\item{method}{A string argument to specify correlation coefficient method.
Three choices \code{c("kendall", "pearson", "wolfsigma")}. The default is
\code{"kendall"}}

\item{resids.type}{A character string specifying which type of residuals to generate
Current options are \code{"latent"} and \code{"uniform"}. Default is \code{"latent"}.
\describe{
  \item{\code{surrogate}}{surrogate residuals (Liu and Zhang, 2017);}
  \item{\code{sign}}{sign-based residuals (Li and Shepherd, 2010, 2012);}
  \item{\code{general}}{generalized residuals (Franses and Paap, 2001);}
  \item{\code{deviance}}{deviance residuals (-2*loglik).}
}

Although \code{"sign"}, \code{"general"}, and \code{"deviance"} are provided in
this package, these residuals are problematic for partial association analysis
between ordinal response (more discussions see Liu, Dungang, Li, Shaobo, Yu, Yan,
and Moustaki, Irini.(2020))}

\item{jitter}{A character string specifying how to generate surrogate residuals.
Current options are \code{"latent"} and \code{"uniform"}. Default is \code{"latent"}.
\describe{
  \item{\code{latent}}{surrogate residuals.}
  \item{\code{uniform}}{sign-based residuals.}
}}

\item{jitter.uniform.scale}{A character string specifying the scale on which to perform
the jittering whenever \code{jitter = "uniform"}. More details: \code{PAsso::residuals}.}

\item{fitted.models}{A list contains all the models (S3 objects) you want to
assess for the partial association between ordinal responses after adjusting
for a set of covariates covariates. All of these models should be applied to the
same dataset, having same covariates, same sample size etc. The models in this
list can be an object of class \code{\link[ordinal]{clm}},
\code{\link[stats]{glm}}, \code{\link[rms]{lrm}}, \code{\link[rms]{orm}},
\code{\link[MASS]{polr}}, \code{\link[VGAM]{vglm}}.}

\item{n_draws}{A number to specify draws of surrogate residuls
such that the partial correlation coefficients are calculated repeatedly. The final
correlation coefficients are the average of all partial correlation coefficients.
It is the \code{"nsim"} argument in \code{"residuals()"} function.}

\item{association}{An default argument to specify the partial association. Leave this
further development of package such that other association analyses can be embedded.}

\item{...}{Additional optional arguments.}
}
\value{
An object of class \code{"PAsso"} is a list containing at least the following
components. It contains the partial correlation matrix and multiple repeats if
\code{n_draws} > 1. This object has "arguments"
attribute saved as c(association, method, resids.type), "responses" attribute, and
"adjustments" attribute.
The list contains:
\describe{
  \item{\code{corr}}{The estimated correlation matrix(average of \code{rep_MatCorr})
  of partial association after adjusting confounders;}
  \item{\code{rep_corr}}{The replications of estimated correlation matrix;}
  \item{\code{rep_SRs}}{The replications of surrogate residuals if partial association is applied;}
  \item{\code{fitted.models}}{The list stores all fitted.models;}
  \item{\code{data}}{The data.frame of original dataset;}
  \item{\code{mods_n}}{The sample size of each fitted model;}
  \item{\code{cor_func}}{The correlation function after assign different method;}
  \item{\code{Marg_corr}}{The marginal association matrix.}
}
}
\description{
This function is mainly designed for conducting the partial association analysis.
It provides two ways of using:

1. A user-friendly way: only need "responses", "adjustments", and
"data". All the rest of the argument will be setted as default (see Arguments for details of
default).

2. An advanced way: user can input a list of fitted models by "fitted.models", then the
"responses" and "adjustments" are not necessary. Supported class of cumulative link models in
\code{\link[ordinal]{clm}}, \code{\link[stats]{glm}}, \code{\link[rms]{lrm}},
\code{\link[rms]{orm}}, \code{\link[MASS]{polr}}, \code{\link[VGAM]{vglm}}, .

It generates an object that has partial association matrix, marginal association,
and some attributes: "arguments" saves c(association, method, resids.type). "responses"
contains the names of response variables. The attribute "adjustments" contains the names
of covariates. The "summary" function of "PAsso" class of object provides marginal association '
matrix for comparison purpose.
}
\examples{

###########################################################
# User-friendly way of using
###########################################################
library(MASS)

# Import ANES2016 data in "PAsso"
data(ANES2016)

# User-friendly way of using: Parial association analysis
PAsso_1 <- PAsso(responses = c("PreVote.num", "PID"),
                adjustments = c("income.num", "age", "edu.year"),
                data = ANES2016,
                method = c("kendall"))

print(PAsso_1, digits = 4)
summary(PAsso_1, digits = 4)

###########################################################
# Advanced way of using
###########################################################

fit.vote<- glm(PreVote.num ~ income.num+ age + edu.year, data = ANES2016,
               family = binomial(link = "probit"))
fit.PID<- polr(as.factor(PID) ~ income.num+age+edu.year, data = ANES2016,
               method="probit", Hess = TRUE)

PAsso_adv1 <- PAsso(fitted.models=list(fit.vote, fit.PID),
                    method = c("kendall"),
                    resids.type = "surrogate")

print(PAsso_adv1, digits = 4)
summary(PAsso_adv1, digits = 4)

}
\references{
Liu, Dungang, Li, Shaobo, Yu, Yan, and Moustaki, Irini. Assessing partial association between
ordinal variables: quantification, visualization, and hypothesis testing, \emph{Journal of the
American Statistical Association}, in press. \doi{10.1080/01621459.2020.1796394}

Liu, Dungang and Zhang, Heping. Residuals and Diagnostics for Ordinal
Regression Models: A Surrogate Approach. \emph{Journal of the American Statistical Association}.
\doi{10.1080/01621459.2017.1292915}

Li, Chun, and Bryan E. Shepherd. "Test of association between two ordinal variables while
adjusting for covariates." \emph{Journal of the American Statistical Association} 105, no.
490 (2010): 612-620. \doi{10.1198/jasa.2010.tm09386}

Li, Chun, and Bryan E. Shepherd. "A new residual for ordinal outcomes." \emph{Biometrika}
99, no. 2 (2012): 473-480. \doi{10.1093/biomet/asr073}

Franses, Philip Hans, and Richard Paap. Quantitative models in marketing research.
Cambridge University Press, 2001. \doi{10.1017/CBO9780511753794}
}
