% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/full_threshold.r
\name{FT}
\alias{FT}
\alias{FT.start}
\alias{FT.step}
\alias{FT.stop}
\alias{FT.final}
\title{Full Threshold}
\usage{
FT(est = 25, instRange = c(0, 40), verbose = FALSE, makeStim, ...)

FT.start(est = 25, instRange = c(0, 40), makeStim, ...)

FT.step(state, nextStim = NULL)

FT.stop(state)

FT.final(state)
}
\arguments{
\item{est}{Starting estimate in dB}

\item{instRange}{Dynamic range of the instrument c(min,max) in dB}

\item{verbose}{True if you want each presentation printed}

\item{makeStim}{A function that takes a dB value and numPresentations
and returns an OPI datatype ready for passing to opiPresent}

\item{...}{Extra parameters to pass to the opiPresent function}

\item{state}{Current state of the FT returned by \code{FT.start} and
\code{FT.step}}

\item{nextStim}{A valid object for \code{opiPresent} to use as its
\code{nextStim}}
}
\value{
\subsection{Single location}{
Returns a list containing
\itemize{
\item{npres}{ Total number of presentations}
\item{respSeq}{ Response sequence stored as a list of (seen,dB) pairs}
\item{first}{ First staircase estimate in dB}
\item{final}{ Final threshold estimate in dB}
}
}
\subsection{Multilple locations}{
\code{FT.start} returns a list that can be passed to \code{FT.step},
\code{FT.stop}, and \code{FT.final}. It represents the state of a FT
at a single location at a point in time and contains the following.
\itemize{
\item{name:}{ \code{FT}}
\item{}{ A copy of all of the parameters supplied to FT.start:
\code{startingEstimate=est}, \code{minStimulus=instRange[1]},
\code{maxStimulus=instRange[2]}, \code{makeStim}, and \code{opiParams=list(...)}.}
\item{currentLevel:}{ The next stimulus to present.}
\item{lastSeen:}{ The last seen stimulus.}
\item{lastResponse:}{ The last response given.}
\item{firstStairResult:}{ The result of the first staircase (initially \code{NA}).}
\item{secondStairResult:}{ The result of the first staircase (initially \code{NA},
and could remain \code{NA}).}
\item{finished:}{ \code{TRUE} if staircae has finished (2 reversals, or max/min
seen/not-seen twice).}
\item{numberOfReversals:}{ Number of reversals so far.}
\item{currSeenLimit:}{ Number of times \code{maxStimulus} has been seen.}
\item{currNotSeenLimit:}{ Number of times \code{minStimulus} not seen.}
\item{numPresentations:}{ Number of presentations so far.}
\item{stimuli:}{ Vector of stimuli shown at each call to \code{FT.step}.}
\item{responses:}{ Vector of responses received (1 seen, 0 not) receieved at each
call to \code{FT.step}.}
\item{responseTimes:}{ Vector of response times receieved at each call to
\code{FT.step}.}
}
}
\code{FT.step} returns a list containing
\itemize{
\item{state:}{ The new state after presenting a stimuli and getting a response.}
\item{resp:}{ The return from the \code{opiPresent} call that was made.}
}
\code{FT.stop} returns \code{TRUE} if the first staircase has had 2 reversals, or
\code{maxStimulus} is seen twice or \code{minStimulus} is not seen twice and the
final estimate is within 4 dB of the starting stimulus. Returns \code{TRUE} if
the second staircase has had 2 reversals, or \code{maxStimulus} is seen twice or
\code{minStimulus} is not seen twice

\code{FT.final} returns the final estimate of threshold based on state, which is
the last seen in the second staircase, if it ran, or the first staircase otherwise

\code{FT.final.details} returns a list containing
\itemize{
\item{final:}{ The final threshold.}
\item{first:}{ The threshold determined by the first staircase (might be
different from final).}
\item{stopReason:}{ Either \code{Reversals}, \code{Max}, or \code{Min} which
are the three ways in which FT can terminate.}
\item{np:}{ Number of presentation for the whole procedure (indcluding both
staircases if run).}
}
}
\description{
FT begins with a 4-2dB staircase beginning at level
\code{est}. If the final estimate (last seen) is more than 4dB away
from \code{est}, a second 4-2 staircase is completed beginning at the
estimate returned from the first
}
\details{
This is an implementation of a 4-2 1-up 1-down staircase as
implemented in the first Humphrey Field Analyzer. The initial staircase
starts at \code{est} and proceeds in steps of 4 dB until the first
reversal, and 2dB until the next reversal. The last seen stimulus is
taken as the threshold value. If, after the first staircase, the
threshold is more than 4 dB away from the starting point, then a second
staircase is initiated with a starting point equal to the threshold
found with the first staircase.

Note this function will repeatedly call \code{opiPresent} for a stimulus
until \code{opiPresent} returns \code{NULL} (ie no error occured)

If more than one FT is to be interleaved (for example, testing multiple
locations), then the \code{FT.start}, \code{FT.step}, \code{FT.stop}
and \code{FT.final} calls can maintain the state of the FT after each
presentation, and should be used. If only a single FT is required, then
the simpler \code{FT} can be used. See examples below
}
\examples{
# Stimulus is Size III white-on-white as in the HFA
makeStim <- function(db, n) { 
  s <- list(x=9, y=9, level=dbTocd(db), size=0.43, color="white",
            duration=200, responseWindow=1500)
  class(s) <- "opiStaticStimulus"
  return(s)
}
chooseOpi("SimHenson")
if (!is.null(opiInitialize(type="C", cap=6)))
  stop("opiInitialize failed")

result <- FT(makeStim=makeStim, tt=30, fpr=0.15, fnr=0.01)
if (!is.null(opiClose()))
  warning("opiClose() failed")

##############################################
# This section is for multiple FTs
##############################################
makeStimHelper <- function(db,n, x, y) {  # returns a function of (db,n)
  ff <- function(db, n) db+n
  body(ff) <- substitute({
    s <- list(x=x, y=y, level=dbTocd(db), size=0.43, color="white",
              duration=200, responseWindow=1500)
    class(s) <- "opiStaticStimulus"
    return(s)
  }, list(x=x,y=y))
  return(ff)
}

# List of (x, y, true threshold) triples
locations <- list(c(9,9,30), c(-9,-9,32), c(9,-9,31), c(-9,9,33))
# Setup starting states for each location
states <- lapply(locations, function(loc) {
  FT.start(makeStim=makeStimHelper(db,n,loc[1],loc[2]),
           tt=loc[3], fpr=0.03, fnr=0.01)})

# Loop through until all states are "stop"
while(!all(st <- unlist(lapply(states, FT.stop)))) {
  i <- which(!st)                         # choose a random, 
  i <- i[runif(1, min=1, max=length(i))]  # unstopped state
  r <- FT.step(states[[i]])               # step it
  states[[i]] <- r$state                  # update the states
}

finals <- lapply(states, FT.final)    # get final estimates of threshold
for(i in 1:length(locations)) {
  cat(sprintf("Location (\%+2d,\%+2d) ",locations[[i]][1], locations[[i]][2]))
  cat(sprintf("has threshold \%4.2f\n", finals[[i]]))
}

if(!is.null(opiClose()))
  warning("opiClose() failed")
}
\references{
A. Turpin, P.H. Artes and A.M. McKendrick. "The Open Perimetry
Interface: An enabling tool for clinical visual psychophysics", Journal
of Vision 12(11) 2012.

H. Bebie, F. Fankhauser and J. Spahr. "Static perimetry: strategies",
Acta Ophthalmology 54 1976.

C.A. Johnson, B.C. Chauhan, and L.R. Shapiro. "Properties of staircase
procedures for estimating thresholds in automated perimetry",
Investagative Ophthalmology and Vision Science 33 1993.
}
\seealso{
\code{\link{dbTocd}}, \code{\link{opiPresent}}, \code{\link{fourTwo.start}}
}
