% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Simulations.R
\name{Spot.Price.Forecast}
\alias{Spot.Price.Forecast}
\title{Forecast N-Factor Model Spot Prices}
\usage{
Spot.Price.Forecast(X.0, parameters, t, Percentiles = NULL)
}
\arguments{
\item{X.0}{Initial values of the state vector.}

\item{parameters}{A named vector of parameter values of a specified N-factor model. Function \code{NFCP.Parameters} is recommended.}

\item{t}{a vector of discrete time points to forecast}

\item{Percentiles}{Optional. A vector of percentiles to include probabilistic forecasting intervals.}
}
\value{
\code{Spot.Price.Forecast} returns a vector of expected future spot prices under a given N-factor model at specified discrete future time points. When \code{percentiles} are specified, the function returns a matrix with the corresponding confidence bands in each column of the matrix.
}
\description{
Analytically forecast expected spot prices following the "true" process of a given n-factor stochastic model
}
\details{
Future expected spot prices under the N-factor model can be forecasted through the analytic expression of expected future prices under the "true" N-factor process.

Given that the log of the spot price is equal to the sum of the state variables (equation 1), the spot price is log-normally distributed with the expected prices given by:

\loadmathjax
\mjdeqn{E[S_t] = exp(E[ln(S_t)] + \frac{1}{2}Var[ln(S_t)])}{exp(E[ln(S[t])] + 1/2 Var[ln(S[t])])}
Where:
\mjdeqn{E[ln(S_t)] = \sum_{i=1}^Ne^{-(\kappa_it)}x_i(0) + \mu t}{E[ln(S[t])] = sum_{i=1}^N (e^(-(kappa[i] t)) x[i,0] + mu * t)}

Where \mjeqn{\kappa_i = 0}{kappa[i] = 0} when \code{GBM=T} and \mjeqn{\mu = 0}{mu = 0} when \code{GBM = F}

\mjdeqn{Var[ln(S_t)] =  \sigma_1^2t + \sum_{i.j\neq1}\sigma_i\sigma_j\rho_{i,j}\frac{1-e^{-(\kappa_i+\kappa_j)t}}{\kappa_i+\kappa_j}}{
Var[ln(S[t])] = sigma[1]^2 * t + sum_{i.j != 1} (sigma[i] sigma[j] rho[i,j] (1 - e^(-(kappa[i] + kappa[j])t)) / (kappa[i] + kappa[j]) )}

and thus:

\mjdeqn{E[S_t] = exp(\sum_{i=1}^N e^{-\kappa_it}x_i(0) + (\mu + \frac{1}{2}\sigma_1^2)t + \frac{1}{2}\sum_{i.j\neq1} \sigma_i\sigma_j\rho_{i,j}\frac{1-e^{-(\kappa_i+\kappa_j)t}}{\kappa_i+\kappa_j})}{
E[S[t]] = exp( sum_{i=1}^N e^(-kappa[i] t) x[i,0] + (mu + 1/2 sigma[1]^2)t + 1/2 (sum_{i.j != 1}( sigma[i] sigma[j] rho[i,j] (1 - e^(-(kappa[i] + kappa[j])t)) / (kappa[i] + kappa[j]))) )}

Under the assumption that the first factor follows a Brownian Motion, in the long-run expected spot prices grow over time at a constant rate of \mjeqn{\mu + \frac{1}{2}\sigma_1^2}{mu + 1/2 sigma[1]} as the \mjeqn{e^{-\kappa_it}}{e^(-kappa[i] * t)} and \mjeqn{e^{-(\kappa_i + \kappa_j)t}}{e^(-(kappa[i] + kappa[j]))} terms approach zero.

An important consideration when forecasting spot prices using parameters estimated through maximum likelihood estimation is that the parameter estimation process takes the assumption of risk-neutrality and thus the true process growth rate \mjeqn{\mu}{mu} is not estimated with a high level of precision. This can be shown from the higher standard error for \mjeqn{\mu}{mu} than other estimated parameters, such as the risk-neutral growth rate \mjeqn{\mu^*}{mu^*}. See Schwartz and Smith (2000) for more details.
}
\examples{
# Forecast the Schwartz and Smith (2000) two-factor oil model:

##Step 1 - Run the Kalman filter for the two-factor oil model:
Schwartz.Smith.Oil <- NFCP.Kalman.filter(SS.Oil$Two.Factor,
                                      names(SS.Oil$Two.Factor),
                                      log(SS.Oil$Stitched.Futures),
                                      SS.Oil$dt,
                                      SS.Oil$Stitched.TTM,
                                      verbose = TRUE)

##Step 2 - Probabilistic forecast of n-factor stochastic differential equation (SDE):
E.Spot <- Spot.Price.Forecast(X.0 = Schwartz.Smith.Oil$X.t,
                                 parameters = SS.Oil$Two.Factor,
                                 t = seq(0,9,1/12),
                                 Percentiles = c(0.1, 0.9))
}
\references{
Schwartz, E. S., and J. E. Smith, (2000). Short-Term Variations and Long-Term Dynamics in Commodity Prices. \emph{Manage. Sci.}, 46, 893-911.

Cortazar, G., and L. Naranjo, (2006). An N-factor Gaussian model of oil futures prices. \emph{Journal of Futures Markets: Futures, Options, and Other Derivative Products}, 26(3), 243-268.
}
