% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/KDEEM.R
\name{kdeem}
\alias{kdeem}
\title{Kernel Density-based EM-type algorithm for Semiparametric Mixture Regression
with Unspecified Error Distributions}
\usage{
kdeem(x, y, C = 2, ini = NULL, maxiter = 200)
}
\arguments{
\item{x}{an n by p data matrix where n is the number of observations and p is the
number of explanatory variables (including the intercept).}

\item{y}{an n-dimensional vector of response variable.}

\item{C}{number of mixture components. Default is 2.}

\item{ini}{initial values for the parameters. Default is NULL, which obtains the initial values
using the \code{\link{kdeem.lse}} function.
If specified, it can be a list with the form of \code{list(beta, prop, tau, pi, h)}, where
\code{beta} is a p by C matrix for regression coefficients of C components,
\code{prop} is an n by C matrix for probabilities of each observation belonging to each component,
caculated based on the initial \code{beta} and \code{h}, \code{tau} is a vector of C precision parameters
(inverse of standard deviation), \code{pi} is a vector of C mixing proportions, and
\code{h} is the bandwidth for kernel estimation.}

\item{maxiter}{maximum number of iterations for the algorithm. Default is 200.}
}
\value{
A list containing the following elements:
\item{posterior}{posterior probabilities of each observation belonging to each component.}
\item{beta}{estimated regression coefficients.}
\item{tau}{estimated precision parameters, the inverse of standard deviation.}
\item{pi}{estimated mixing proportions.}
\item{h}{bandwidth used for the kernel estimation.}
}
\description{
`kdeem' is used for semiparametric mixture regression using a kernel density-based
expectation-maximization (EM)-type algorithm with unspecified
homogeneous or heterogenous error distributions (Ma et al., 2012).
}
\details{
It can be used for a semiparametric mixture of linear regression models with
unspecified component error distributions. The errors can be either homogeneous or heterogenous.
The model is as follows:
\deqn{f_{Y|\boldsymbol{X}}(y,\boldsymbol{x},\boldsymbol{\theta},g) = \sum_{j=1}^C\pi_j\tau_jg\{(y-\boldsymbol{x}^{\top}\boldsymbol{\beta}_j)\tau_j\}.}
Here, \eqn{\boldsymbol{\theta}=(\pi_1,...,\pi_{C-1},\boldsymbol{\beta}_1^{\top},..,\boldsymbol{\beta}_C^{\top},\tau_1,...,\tau_C)^{\top}},
\eqn{g(\cdot)} is an unspecified density function with mean 0 and variance 1, and \eqn{\tau_j} is a precision parameter.
For the calculation of \eqn{\beta} in the M-step, this function employs the universal optimizer function \code{\link{ucminf}} from the `ucminf' package.
}
\examples{
n = 300
C = 2
Dimen = 2
Beta.true.matrix = matrix(c(-3, 3, 3, -3), Dimen, C)
PI.true = c(0.5, 0.5)
x = runif(n)
X = cbind(1, x)
Group.ID = Rlab::rbern(n, prob = 0.5)
Error = rnorm(n, 0, 1)
n1 = sum(Group.ID)
n2 = n - n1
y = rep(0, n)
err = rep(0, n)

for(i in 1:n){
  if(Group.ID[i] == 1){
    err[i] = Error[i]
    y[i] = X[i, ] \%*\% Beta.true.matrix[, 1] + err[i]
  } else {
    err[i] = 0.5 * Error[i]
    y[i] = X[i, ] \%*\% Beta.true.matrix[, 2] + err[i]
  }
}
Result.kdeem.lse = kdeem.lse(x, y)
\donttest{Result.kdeem.h = kdeem.h(x, y, 2, Result.kdeem.lse, maxiter = 200)}
\donttest{Result.kdeem = kdeem(x, y, 2, Result.kdeem.lse, maxiter = 200)}
}
\references{
Ma, Y., Wang, S., Xu, L., & Yao, W. (2021). Semiparametric mixture regression
with unspecified error distributions. Test, 30, 429-444.
}
\seealso{
\code{\link{kdeem.h}}, \code{\link{kdeem.lse}}, and \code{\link[ucminf]{ucminf}} for beta calculation.
}
