% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estConnectivity.R
\name{estStrength}
\alias{estStrength}
\title{Estimate MC, migratory connectivity strength}
\usage{
estStrength(
  originDist,
  targetDist,
  originRelAbund,
  psi,
  sampleSize = NULL,
  originSites = NULL,
  targetSites = NULL,
  originNames = NULL,
  targetNames = NULL,
  nSamples = 1000,
  row0 = 0,
  verbose = 0,
  alpha = 0.05,
  approxSigTest = FALSE,
  sigConst = 0,
  maintainLegacyOutput = FALSE,
  returnAllInput = TRUE
)
}
\arguments{
\item{originDist}{Distances between the B origin sites. Symmetric B by B
matrix}

\item{targetDist}{Distances between the W target sites. Symmetric W by W
matrix}

\item{originRelAbund}{Relative abundance estimates at B origin sites. Either
a numeric vector of length B that sums to 1, or an mcmc object (such as is
produced by \code{\link{modelCountDataJAGS}}) or matrix with at least
\code{nSamples} rows. If there are more than B columns, the relevant columns
should be labeled "relN[1]" through "relN[B]"}

\item{psi}{Transition probabilities between B origin and W target sites.
Either a matrix with B rows and W columns where rows sum to 1, an array with
dimensions x, B, and W (with x samples of the transition probability matrix
from another model), an 'estPsi' object (result of calling estTransition),
or a MARK object with estimates of transition probabilities}

\item{sampleSize}{Total sample size of animals that psi will be estimated
from. Should be the number of animals released in one of the origin sites
and observed in one of the target sites (or vice-versa). Optional, but
recommended, unless psi is an estPsi object (in which case this function can
pull it from there)}

\item{originSites}{If \code{psi} is a MARK object, this must be a numeric
vector indicating which sites are origin}

\item{targetSites}{If \code{psi} is a MARK object, this must be a numeric
vector indicating which sites are target}

\item{originNames}{Optional. Vector of names for the origin sites. Mostly for
internal use}

\item{targetNames}{Optional. Vector of names for the target sites. Mostly for
internal use}

\item{nSamples}{Number of times to resample \code{psi} and/or
\code{originRelAbund}. The purpose is to estimate sampling uncertainty;
higher values here will do so with more precision}

\item{row0}{If \code{originRelAbund} is an mcmc object or array, this can be
set to 0 (default) or any greater integer to specify where to stop ignoring
samples ("burn-in")}

\item{verbose}{0 (default) to 2. 0 prints no output during run. 1 prints
a progress update and summary every 100 samples. 2 prints a
progress update and summary every sample}

\item{alpha}{Level for confidence/credible intervals provided. Default (0.05)
gives 95 percent CI}

\item{approxSigTest}{Should function compute approximate one-sided
significance tests (p-values) for MC from the resampling? Default is
FALSE}

\item{sigConst}{Value to compare MC to in significance test. Default is 0}

\item{maintainLegacyOutput}{version 0.4.0 of \code{MigConnectivity}
updated the structure of the estimates. If you have legacy code that refers
to elements within an \code{estMigConnectivity} object (results of
\code{estMC}), you can set this to TRUE to also keep the old structure.
Defaults to FALSE}

\item{returnAllInput}{if TRUE (the default) the output includes all of the
inputs. If FALSE, only the inputs currently used by another MigConnectivity
function are included in the output.}
}
\value{
\code{estStrength} returns a list with the elements:
\describe{
  \item{\code{MC}}{List containing estimates of migratory connectivity
   strength:
   \itemize{
     \item{\code{sample}} \code{nSamples} sampled values for
      MC. Provided to allow the user to compute own summary statistics.
     \item{\code{mean}} Mean of \code{MC$sample}. Main estimate of MC,
      incorporating parametric uncertainty.
     \item{\code{se}} Standard error of MC, estimated from SD of
      \code{MC$sample}.
     \item{\code{simpleCI}} Default\code{1 - alpha} confidence interval for
      MC, estimated as \code{alpha/2} and \code{1 - alpha/2} quantiles of
      \code{MC$sample}.
     \item{\code{bcCI}} Bias-corrected \code{1 - alpha} confidence interval
      for MC. May be preferable to \code{MC$simpleCI} when \code{MC$mean} is
      the best estimate of MC. \code{MC$simpleCI} is preferred when
      \code{MC$median} is a better estimator. When \code{MC$mean==MC$median},
      these should be identical.  Estimated as the
      \code{pnorm(2 * z0 + qnorm(alpha / 2))} and
      \code{pnorm(2 * z0 + qnorm(1 - alpha / 2))} quantiles of \code{MC$sample},
      where z0 is the proportion of \code{MC$sample < MC$mean}.
     \item{\code{hpdCI}} \code{1 - alpha} credible interval for MC,
      estimated using the highest posterior density (HPD) method.
     \item{\code{median}} Median of MC, alternate point estimate also
      including parametric uncertainty.
     \item{\code{point}} Simple point estimate of MC, using the point
     estimates of \code{psi} and \code{originRelAbund} (usually the mean
     values), not accounting for sampling error.
     \item{\code{simpleP}} Approximate p-value for MC, estimated as the
     proportion of bootstrap iterations where MC < \code{sigConst} (or MC >
     \code{sigConst} if \code{pointMC < sigConst}).  Note that if the
     proportion is 0, a default value of 0.5 / \code{nSamples} is provided,
     but this is best interpreted as p < 1 / \code{nSamples}.  NULL when
     \code{approxSigTest==FALSE}.
     \item{\code{bcP}} Approximate bias-corrected p-value for MC, estimated as
     \code{pnorm(qnorm(simpleP) - 2 * z0)}, where z0 is the proportion of
     \code{sampleMC < meanMC}.  May be a better approximation of the p-value
     than \code{simpleP}, but many of the same limitations apply.  NULL when
     \code{approxSigTest==FALSE}.
   }
  }
  \item{\code{input}}{List containing the inputs to \code{estStrength}.}
}
}
\description{
Resampling of uncertainty for MC (migratory connectivity strength)
from estimates of psi (transition probabilities) and/or relative abundance.
Psi estimates can come from an estMigConnectivity object, an RMark psi
matrix, MCMC samples, or other samples expressed in array form.
Abundance estimates for each origin site can be
either just point estimates (no uncertainty propagated) or MCMC samples.
Other inputs include distances between origin sites, distances between target
sites, and sample size used to estimate psi.
}
\examples{
\donttest{
  set.seed(101)
  # Uncertainty in detection (RMark estimates) with equal abundances
  # Number of resampling iterations for generating confidence intervals
  nSamplesCMR <- 100
  nSimulationsCMR <- 10
  originPos13 <- matrix(c(rep(seq(-99, -81, 2), each = 10),
                          rep(seq(49, 31, -2), 10)), 100, 2)
  targetPos13 <- matrix(c(rep(seq(-79, -61, 2), each = 10),
                          rep(seq(9, -9, -2), 10)), 100, 2)
  originPosCMR <- rowsum(originPos13, c(rep(1:2, 5, each = 5),
                                        rep(3:4, 5, each = 5))) / 25
  originPosCMR
  targetPosCMR <- rowsum(targetPos13, c(rep(1:2, 5, each = 5),
                                        rep(3:4, 5, each = 5))) / 25
  targetPosCMR

  originDist <- distFromPos(originPosCMR, 'ellipsoid')
  targetDist <- distFromPos(targetPosCMR, 'ellipsoid')
  originRelAbundTrue <- rep(0.25, 4)
  # the second intermediate psi scenario, the "low" level
  psiTrue <- samplePsis[["Low"]]
  trueMC <- calcMC(originDist, targetDist, originRelAbundTrue, psiTrue)
  trueMC

  # Storage matrix for samples
  cmrMCSample <- matrix(NA, nSamplesCMR, nSimulationsCMR)
  summaryCMR <- data.frame(Simulation = 1:nSimulationsCMR, True=trueMC,
                           mean=NA, se=NA, lcl=NA, ucl=NA)
  # Get 'RMark' psi estimates and estimate MC from each
  for (r in 1:nSimulationsCMR) {
    cat("Simulation",r,"of",nSimulationsCMR,"\n")
    # Note: getCMRexample() requires a valid internet connection and that GitHub
    # is accessible
    fm <- getCMRexample(r)
    results <- estStrength(originRelAbund = originRelAbundTrue, psi = fm,
                     originDist = originDist, targetDist = targetDist,
                     originSites = 5:8, targetSites = c(3,2,1,4),
                     nSamples = nSamplesCMR, verbose = 0,
                     sampleSize = length(grep('[2-5]', fm$data$data$ch)))
    cmrMCSample[ , r] <- results$MC$sample
    summaryCMR$mean[r] <- results$MC$mean
    summaryCMR$se[r] <- results$MC$se
    # Calculate confidence intervals using quantiles of sampled MC
    summaryCMR[r, c('lcl', 'ucl')] <- results$MC$simpleCI
  }

  summaryCMR <- transform(summaryCMR, coverage = (True>=lcl & True<=ucl))
  summaryCMR
  summary(summaryCMR)
  biasCMR <- mean(summaryCMR$mean) - trueMC
  biasCMR
  mseCMR <- mean((summaryCMR$mean - trueMC)^2)
  mseCMR
  rmseCMR <- sqrt(mseCMR)
  rmseCMR


  # Simulation of BBS data to quantify uncertainty in relative abundance

  nSamplesAbund <- 700 #1700 are stored
  nSimulationsAbund <- 10
  #\dontrun{
  #  nSamplesAbund <- 1700
  #}
  # Storage matrix for samples
  abundMCSample <- matrix(NA, nSamplesAbund, nSimulationsAbund)
  summaryAbund <- data.frame(Simulation = 1:nSimulationsAbund, True = trueMC,
                             mean = NA, se = NA, lcl = NA, ucl = NA)
  for (r in 1:nSimulationsAbund) {
    cat("Simulation",r,"of",nSimulationsAbund,"\n")
    row0 <- nrow(abundExamples[[r]]) - nSamplesAbund
    results <- estStrength(originRelAbund = abundExamples[[r]], psi = psiTrue,
                     originDist = originDist, targetDist = targetDist,
                     row0 = row0, nSamples = nSamplesAbund, verbose = 1)
    abundMCSample[ , r] <- results$MC$sample
    summaryAbund$mean[r] <- results$MC$mean
    summaryAbund$se[r] <- results$MC$se
    # Calculate confidence intervals using quantiles of sampled MC
    summaryAbund[r, c('lcl', 'ucl')] <- results$MC$simpleCI
  }

  summaryAbund <- transform(summaryAbund, coverage = (True >= lcl & True <= ucl))
  summaryAbund
  summary(summaryAbund)
  biasAbund <- mean(summaryAbund$mean) - trueMC
  biasAbund
  mseAbund <- mean((summaryAbund$mean - trueMC)^2)
  mseAbund
  rmseAbund <- sqrt(mseAbund)
  rmseAbund

  # Ovenbird example with GL and GPS data
  data(OVENdata) # Ovenbird

  nSamplesGLGPS <- 100 # Number of bootstrap iterations, set low for example

  # Estimate transition probabilities
  Combined.psi<-estTransition(isGL=OVENdata$isGL, #Light-level geolocator (T/F)
                              isTelemetry = !OVENdata$isGL,
                  geoBias = OVENdata$geo.bias, # Light-level GL location bias
                  geoVCov = OVENdata$geo.vcov, # Location covariance matrix
                  targetSites = OVENdata$targetSites, # Nonbreeding/target sites
                  originSites = OVENdata$originSites, # Breeding/origin sites
                  originPoints = OVENdata$originPoints, # Capture Locations
                  targetPoints = OVENdata$targetPoints, #Device target locations
                  verbose = 3,   # output options
                  nSamples = nSamplesGLGPS, # This is set low for example
                  resampleProjection = sf::st_crs(OVENdata$targetPoints),
                  nSim = 1000)

  # Can estimate MC from previous psi estimate
  Combo.MC1 <- estStrength(targetDist = OVENdata$targetDist, # Distance matrix
                           originDist = OVENdata$originDist, # Distance matrix
                           targetSites = OVENdata$targetSites, # Target sites
                           originSites = OVENdata$originSites, # Breeding sites
                           psi = Combined.psi,
                           originRelAbund = OVENdata$originRelAbund,
                           nSamples = nSamplesGLGPS,
                           sampleSize = nrow(OVENdata$targetPoints))
  Combo.MC1

  # Doesn't have to be an estPsi object - can simply be array of psi samples
  Combo.MC2 <- estStrength(targetDist = OVENdata$targetDist,
                           originDist = OVENdata$originDist,
                           targetSites = OVENdata$targetSites,
                           originSites = OVENdata$originSites,
                           psi = Combined.psi$psi$sample, # Array of samples
                           originRelAbund = OVENdata$originRelAbund,
                           nSamples = nSamplesGLGPS,
                           sampleSize = nrow(OVENdata$targetPoints))
  Combo.MC2
}
}
\seealso{
\code{\link{calcMC}}, \code{\link{estTransition}},
  \code{\link{estMC}}, \code{\link{estMantel}},
  \code{\link{plot.estMigConnectivity}}
}
