% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ms_outlook_folder.R
\docType{class}
\name{ms_outlook_folder}
\alias{ms_outlook_folder}
\title{Outlook mail folder}
\format{
An R6 object of class \code{ms_outlook_folder}, inheriting from \code{ms_outlook_object}, which in turn inherits from \code{ms_object}.
}
\description{
Class representing a folder in Outlook.
}
\section{Fields}{

\itemize{
\item \code{token}: The token used to authenticate with the Graph host.
\item \code{tenant}: The Azure Active Directory tenant for the email account.
\item \code{type}: always "mail folder" for an Outlook folder object.
\item \code{user_id}: the user ID of the Outlook account.
\item \code{properties}: The item properties (metadata).
}
}

\section{Methods}{

\itemize{
\item \code{new(...)}: Initialize a new object. Do not call this directly; see 'Initialization' below.
\item \code{delete(confirm=TRUE)}: Delete this folder. By default, ask for confirmation first. Note that special folders cannot be deleted.
\item \code{update(...)}: Update the item's properties (metadata) in Microsoft Graph.
\item \code{do_operation(...)}: Carry out an arbitrary operation on the item.
\item \code{sync_fields()}: Synchronise the R object with the item metadata in Microsoft Graph.
\item \code{list_emails(...)}: List the emails in this folder.
\item \code{get_email(message_id)}: Get the email with the specified ID.
\item \code{create_email(...)}: Creates a new draft email in this folder, optionally sending it as well. See 'Creating and sending emails'.
\item \code{delete_email(message_id, confim=TRUE)}: Deletes the specified email. By default, ask for confirmation first.
\item \code{list_folders(filter=NULL, n=Inf)}: List subfolders of this folder.
\item \code{get_folder(folder_name, folder_id)}: Get a subfolder, either by the name or ID.
\item \code{create_folder(folder_name)}: Create a new subfolder of this folder.
\item \code{delete_folder(folder_name, folder_id, confirm=TRUE)}: Delete a subfolder. By default, ask for confirmation first.
\item \verb{copy(dest),move(dest)}: Copies or moves this folder to another folder. All the contents of the folder will also be copied/moved. The destination should be an object of class \code{ms_outlook_folder}.
}
}

\section{Initialization}{

Creating new objects of this class should be done via the \code{get_folder}, \code{list_folders} or \code{create_folder} methods of this class or the \code{\link{ms_outlook}} class. Calling the \code{new()} method for this class only constructs the R object; it does not call the Microsoft Graph API to retrieve or create the actual folder.
}

\section{Creating and sending emails}{

Outlook allows creating new draft emails in any folder, not just the Drafts folder (although that is the default location for the Outlook app, and the \code{ms_outlook} client class). To create a new email, call the \code{create_email()} method, which has the following signature:

\if{html}{\out{<div class="sourceCode">}}\preformatted{create_email(body = "", content_type = c("text", "html"), subject = "",
             to = NULL, cc = NULL, bcc = NULL, reply_to = NULL, send_now = FALSE)
}\if{html}{\out{</div>}}
\itemize{
\item \code{body}: The body of the message. This should be a string or vector of strings, which will be pasted together with newlines as separators. You can also supply a message object as created by the blastula or emayili packages---see the examples below.
\item \code{content_type}: The format of the body, either "text" (the default) or HTML.
\item \code{subject}: The subject of the message.
\item \verb{to,cc,bcc,reply_to}: These should be lists of email addresses, in standard "user@host" format. You can also supply objects of class \code{\link[AzureGraph:az_user]{AzureGraph::az_user}} representing user accounts in Azure Active Directory.
\item \code{send_now}: Whether the email should be sent immediately, or saved as a draft. You can send a draft email later with its \code{send()} method.
}

This returns an object of class \code{\link{ms_outlook_email}}, which has methods for making further edits and attaching files.

You can also supply message objects as created by the blastula and emayili packages in the \code{body} argument. Note that blastula objects include attachments (if any), and emayili objects include attachments, recipients, and subject line; the corresponding arguments to \code{create_email()} will not be used in this case.

To reply to or forward an email, first retrieve it using \code{get_email()} or \code{list_emails()}, and then call its \code{create_reply()}, \code{create_reply_all()} or \code{create_forward()} methods.
}

\section{Listing emails}{

To list the emails in a folder, call the \code{list_emails()} method. This returns a list of objects of class \code{\link{ms_outlook_email}}, and has the following signature:

\if{html}{\out{<div class="sourceCode">}}\preformatted{list_emails(by = "received desc", search = NULL, filter = NULL, n = 100, pagesize = 10)
}\if{html}{\out{</div>}}
\itemize{
\item \code{by}: The sorting order of the message list. The possible fields are "received" (received date, the default), "from" and "subject". To sort in descending order, add a " desc". You can specify multiple sorting fields, with later fields used to break ties in earlier ones. The last sorting field is always "received desc" unless it appears earlier.
\item \code{search}: An optional string to search for. Only emails that contain the search string will be returned. See the \href{https://learn.microsoft.com/en-us/graph/query-parameters#search-parameter}{description of this parameter} for more information.
\item \verb{filter, n}: See below.
\item \code{pagesize}: The number of emails per page. You can change this to a larger number to increase throughput, at the risk of running into timeouts.
}

Currently, searching and filtering the message list is subject to some limitations. You can only specify one of \code{search} and \code{filter}; searching and filtering at the same time will not work. Ordering the results is only allowed if neither a search term nor a filtering expression is present. If searching or filtering is done, the result is always sorted by date.
}

\section{List methods generally}{

All \verb{list_*} methods have \code{filter} and \code{n} arguments to limit the number of results. The former should be an \href{https://learn.microsoft.com/en-us/graph/query-parameters#filter-parameter}{OData expression} as a string to filter the result set on. The latter should be a number setting the maximum number of (filtered) results to return. The default values are \code{filter=NULL} and \code{n=100} for listing emails, and \code{n=Inf} for listing folders. If \code{n=NULL}, the \code{ms_graph_pager} iterator object is returned instead to allow manual iteration over the results.

Support in the underlying Graph API for OData queries is patchy. Not all endpoints that return lists of objects support filtering, and if they do, they may not allow all of the defined operators. If your filtering expression results in an error, you can carry out the operation without filtering and then filter the results on the client side.
}

\examples{
\dontrun{

outl <- get_personal_outlook()

folder <- outl$get_folder("My folder")

##
## listing emails
##

# the default: 100 most recent messages
folder$list_emails()

# sorted by subject, then by most recent received date
folder$list_emails(by="subject")

# sorted by from name in descending order, then by most recent received date
folder$list_emails(by="from desc")

# searching the list
folder$list_emails(search="important information")

# retrieve a specific email:
# note the Outlook ID is NOT the same as the Internet message-id
email_id <- folder$list_emails()[[1]]$properties$id
folder$get_email(email_id)

##
## creating/sending emails
##

# a simple text email with just a body:
# you can add other properties by calling the returned object's methods
folder$create_email("Hello from R")

# HTML-formatted email with all necessary fields, sent immediately
folder$create_email("<emph>Emphatic hello</emph> from R",
    content_type="html",
    to="user@example.com",
    subject="example email",
    send_now=TRUE)

# using blastula to create a HTML email with Markdown
bl_msg <- blastula::compose_email(md(
"
## Hello!

This is an email message that was generated by the blastula package.

We can use **Markdown** formatting with the `md()` function.

Cheers,

The blastula team
"),
    footer=md("Sent via Microsoft365R"))
folder$create_email(bl_msg, subject="example blastula email")


# using emayili to create an email with attachments
ey_email <- emayili::envelope(
    text="Hello from emayili",
    to="user@example.com",
    subject="example emayili email") \%>\%
    emayili::attachment("mydocument.docx") \%>\%
    emayili::attachment("mydata.xlsx")
folder$create_email(ey_email)

}
}
\seealso{
\code{\link{ms_outlook}}, \code{\link{ms_outlook_email}}

\href{https://learn.microsoft.com/en-us/graph/overview}{Microsoft Graph overview},
\href{https://learn.microsoft.com/en-us/graph/api/resources/mail-api-overview?view=graph-rest-1.0}{Outlook API reference}
}
