% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MN_CLASSIFICATION.R
\name{PredictMN}
\alias{PredictMN}
\title{Classify matrix-valued data based on a matrix-normal linear discriminant; an object of class "MN". }
\usage{
PredictMN(object, newdata, newclass = NULL)
}
\arguments{
\item{object}{An object of class "MN"; output from \code{MatLDA} or \code{MN_MLE}. }

\item{newdata}{New data to be classified; an \eqn{r \times c \times N_{\rm test}} array.}

\item{newclass}{Class labels for new data; should be \eqn{in \left\{1, \dots, J\right\}}. Default is \code{NULL}.}
}

\value{

\item{pred.class}{An \eqn{N_{\rm test}}-vector of predicted class membership based on the inputed object.}

\item{misclass}{If \code{newclass} is non-\code{NULL}, returns the misclassification proportion on the test data set.}

\item{prob.mat}{A \eqn{N_{\rm test} \times J} matrix with the value of discriminant evaluated at each test data point.}
}
\description{
A function for prediction based on an object of class "MN"; models fit by \code{MatLDA} or \code{MN_MLE}. 
}

\references{
\itemize{
\item Molstad, A. J., and Rothman, A. J. (2018). A penalized likelihood method for classification with matrix-valued predictors. \emph{Journal of Computational and Graphical Statistics}.
}
}



\examples{
## Generate realizations of matrix-normal random variables
## set sample size, dimensionality, number of classes, 
## and marginal class probabilities

N = 75
N.test = 150

N.total = N + N.test

r = 16
p = 16
C = 3

pi.list = rep(1/C, C)

## create class means
M.array = array(0, dim=c(r, p, C))
M.array[3:4, 3:4, 1] = 1
M.array[5:6, 5:6, 2] = .5
M.array[3:4, 3:4, 3] = -2
M.array[5:6, 5:6, 3] = -.5


## create covariance matrices U and V
Uinv = matrix(0, nrow=r, ncol=r)
for (i in 1:r) {
	for (j in 1:r) {
		Uinv[i,j] = .5^abs(i-j)
	}
}


eoU = eigen(Uinv)
Uinv.sqrt = tcrossprod(tcrossprod(eoU$vec, diag(eoU$val^(1/2))),eoU$vec)

Vinv = matrix(.5, nrow=p, ncol=p)
diag(Vinv) = 1 
eoV = eigen(Vinv)
Vinv.sqrt = tcrossprod(tcrossprod(eoV$vec, diag(eoV$val^(1/2))),eoV$vec)

## generate N.total realizations of matrix-variate normal data
set.seed(1)
dat.array = array(0, dim=c(r,p,N.total))
class.total = numeric(length=N.total)
for(jj in 1:N.total){
	class.total[jj] = sample(1:C, 1, prob=pi.list)
	dat.array[,,jj] = tcrossprod(crossprod(Uinv.sqrt, matrix(rnorm(r*p), nrow=r)),
	Vinv.sqrt) + M.array[,,class.total[jj]]
}

## store generated data 
X = dat.array[,,1:N]
X.test = dat.array[,,(N+1):N.total]

class = class.total[1:N]
class.test = class.total[(N+1):N.total]

## fit matrix-normal model using maximum likelihood
out = MN_MLE(X = X, class = class)

## use output to classify test set
check = PredictMN(out, newdata = X.test, newclass = class.test)

## print misclassification proportion
check$misclass
}
