% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mvlm.R
\name{mvlm}
\alias{mvlm}
\title{Conduct multivariate multiple regression and MANOVA with analytic p-values}
\usage{
mvlm(formula, data, n.cores = 1, start.acc = 1e-20,
  contr.factor = "contr.sum", contr.ordered = "contr.poly")
}
\arguments{
\item{formula}{An object of class \code{formula} where the outcome (e.g. the
Y in the following formula: Y ~ x1 + x2) is a \code{n x q matrix}, where
\code{q} is the number of outcome variables being regressed onto the set
of predictors included in the formula.}

\item{data}{Mandatory \code{data.frame} containing all of the predictors
passed to \code{formula}.}

\item{n.cores}{Number of cores to use in parallelization through the
\code{parallel} pacakge.}

\item{start.acc}{Starting accuracy of the Davies (1980) algorithm
implemented in the \code{\link{davies}} function in the \code{CompQuadForm}
package (Duchesne &  De Micheaux, 2010) that \code{mvlm} uses to compute
multivariate linear model p-values.}

\item{contr.factor}{The type of contrasts used to test unordered categorical
variables that have type \code{factor}. Must be a string taking one of the
following values: \code{("contr.sum", "contr.treatment", "contr.helmert")}.}

\item{contr.ordered}{The type of contrasts used to test ordered categorical
variables that have type \code{ordered}. Must be a string taking one of the
following values: \code{("contr.poly", "contr.sum", "contr.treatment",
"contr.helmert")}.}
}
\value{
An object with nine elements and a summary function. Calling
\code{summary(mvlm.res)} produces a data frame comprised of:
\item{Statistic}{Value of the corresponding test statistic.}
\item{Numer DF}{Numerator degrees of freedom for each test statistic.}
\item{Pseudo R2}{Size of the corresponding (omnibus or conditional) effect
on the multivariate outcome. Note that the intercept term does not have
an estimated effect size.}
\item{p-value}{The p-value for each (omnibus or conditional) effect.}
In addition to the information in the three columns comprising
\code{summary(mvlm.res)}, the \code{mvlm.res} object also contains:

\item{p.prec}{A data.frame reporting the precision of each p-value.
These are the maximum error bound of the p-values reported by the
\code{davies} function in \code{CompQuadForm}.}
\item{y.rsq}{A matrix containing in its first row the overall variance
explained by the model for variable comprising Y (columns). The remaining
rows list the variance of each outcome that is explained by the conditional
effect of each predictor.}
\item{beta.hat}{Estimated regression coefficients.}
\item{adj.n}{Adjusted sample size used to determine whether or not the
asmptotic properties of the model are likely to hold. See McArtor et al.
(under review) for more detail.}
\item{data}{Original input data and the \code{model.matrix} used to fit the
model.}
\item{formula}{The formula passed to \code{mvlm}.}

Note that the printed output of \code{summary(res)} will truncate p-values
to the smallest trustworthy values, but the object returned by
\code{summary(mvlm.res)} will contain the p-values as computed. If the error
bound of the Davies algorithm is larger than the p-value, the only conclusion
that can be drawn with certainty is that the p-value is smaller than (or
 equal to) the error bound.
}
\description{
\code{mvlm} is used to fit linear models with a multivariate outcome. It uses
the asymptotic null distribution of the multivariate linear model test
statistic to compute p-values (McArtor et al., under review). It therefore
alleviates the need to use approximate p-values based Wilks' Lambda, Pillai's
Trace, the Hotelling-Lawley Trace, and Roy's Greatest Root.
}
\details{
Importantly, the outcome of \code{formula} must be a \code{matrix}, and the
object passed to \code{data} must be a data frame containing all of the
variables that are named as predictors in \code{formula}.

The conditional effects of variables of type \code{factor} or \code{ordered}
in \code{data} are computed based on the type of contrasts specified by
\code{contr.factor} and \code{contr.ordered}. If \code{data} contains an
(ordered or unordered) factor with \code{k} levels, a \code{k-1} degree of
freedom test will be conducted corresponding to that factor and the specified
contrast structure. If, instead, the user wants to assess \code{k-1} separate
single DF tests that comprise this omnibus effect (similar to the approach
taken by \code{lm}), then the appropriate model matrix should be formed in
advance and passed to \code{mvlm} directly in the \code{data} parameter. See
the package vigentte for an example by calling
\code{vignette('mvlm-vignette')}.
}
\examples{
data(mvlmdata)

Y <- as.matrix(Y.mvlm)

# Main effects model
mvlm.res <- mvlm(Y ~ Cont + Cat + Ord, data = X.mvlm)
summary(mvlm.res)

# Include two-way interactions
mvlm.res.int <- mvlm(Y ~ .^2, data = X.mvlm)
summary(mvlm.res.int)

}
\author{
Daniel B. McArtor (dmcartor@nd.edu) [aut, cre]
}
\references{
Davies, R. B. (1980). The Distribution of a Linear Combination of
 chi-square Random Variables. Journal of the Royal Statistical Society.
 Series C (Applied Statistics), 29(3), 323-333.

 Duchesne, P., & De Micheaux, P.L. (2010). Computing the distribution of
 quadratic forms: Further comparisons between the Liu-Tang-Zhang
 approximation and exact methods. Computational Statistics and Data
 Analysis, 54(4), 858-862.

 McArtor, D. B., Grasman, R. P. P. P., Lubke, G. H., & Bergeman, C. S.
 (under review). A new approach to conducting linear model hypothesis tests
 with a multivariate outcome.
}

