% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/getTile.R
\name{getTile}
\alias{getTile}
\alias{getTile,missing,missing,missing-method}
\alias{getTile,missing,charORnum,charORnum-method}
\alias{getTile,character,ANY,ANY-method}
\alias{getTile,character-method}
\alias{getTile,Raster,ANY,ANY-method}
\alias{getTile,Raster-method}
\alias{getTile,map,ANY,ANY-method}
\alias{getTile,map-method}
\alias{getTile,Extent,ANY,ANY-method}
\alias{getTile,Extent-method}
\alias{getTile,bbox,ANY,ANY-method}
\alias{getTile,bbox-method}
\alias{getTile,Spatial,ANY,ANY-method}
\alias{getTile,Spatial-method}
\alias{getTile,sf,ANY,ANY-method}
\alias{getTile,sf-method}
\title{Get MODIS Tile ID(s)}
\usage{
\S4method{getTile}{missing,missing,missing}(mode = c("click", "draw"), ...)

\S4method{getTile}{missing,charORnum,charORnum}(x, tileH, tileV, ...)

\S4method{getTile}{character,ANY,ANY}(x, tileH, tileV, ...)

\S4method{getTile}{Raster,ANY,ANY}(x, tileH, tileV, ...)

\S4method{getTile}{map,ANY,ANY}(x, tileH, tileV, ...)

\S4method{getTile}{Extent,ANY,ANY}(x, tileH, tileV, ...)

\S4method{getTile}{bbox,ANY,ANY}(x, tileH, tileV, ...)

\S4method{getTile}{Spatial,ANY,ANY}(x, tileH, tileV, ...)

\S4method{getTile}{sf,ANY,ANY}(x, tileH, tileV, ...)
}
\arguments{
\item{mode}{Interactive selection mode as \code{character}. Available options 
are \code{"click"} (default) and \code{"draw"} that trigger interactive MODIS 
tile selection and free feature drawing, respectively. Triggered only if 'x' 
and tile IDs are omitted.}

\item{...}{Additional arguments passed to \code{\link{MODISoptions}}, see
Details.}

\item{x}{Extent information, see Details.}

\item{tileH, tileV}{\code{numeric} or \code{character}. Horizontal and 
vertical tile number(s) of the 
\href{https://nsidc.org/data/docs/daac/mod10_modis_snow/landgrid.html}{MODIS Sinusoidal grid}
(e.g., \code{tileH = 1:5}). Cropping is disabled here and full tiles (if more 
than one then also mosaicked) are processed instead. Ignored if \code{x} is 
specified.}
}
\value{
A \code{MODISextent} object.
}
\description{
Get MODIS tile ID(s) for a specific geographic area.
}
\details{
Unless stated otherwise in the following, target 'outProj' and 'pixelSize' 
are carried over from \code{\link[MODIS]{MODISoptions}}.

If 'x' is of class (see Examples for use cases)
\tabular{ll}{
  \code{missing}:\cr
  \tab If tile IDs (see Arguments) are also missing, a viewer window 
  pops up that allows for interactive tile selection from the global MODIS 
  Sinusoidal grid or, if \code{mode = "draw"}, free feature drawing.\cr
  \cr 
  \code{character}:\cr
  \tab The country name of a \code{map} object (see \code{\link{map}}) with 
  pattern matching via regular expressions enabled. Alternatively, a valid 
  file path to a single ESRI shapefile (.shp) or an image readable by 
  \code{\link[raster]{raster}}.\cr
  \cr
  \code{Raster*}:\cr
  \tab Spatial extent, resolution, and projection of the specified 
  \code{Raster*} are determined automatically. This information is used by 
  \code{\link{runGdal}} to create perfectly matching files. If the 
  \code{Raster*} comes with no valid CRS, 
  \href{https://spatialreference.org/ref/epsg/wgs-84/}{EPSG:4326} is assumed.\cr
  \cr
  \code{Extent}, \code{bbox}:\cr
  \tab Boundary coordinates from \code{Extent} objects are generally assumed 
  to be in \href{https://spatialreference.org/ref/epsg/wgs-84/}{EPSG:4326} as 
  such objects have no projection information attached. The same applies for 
  'bbox' objects lacking CRS information.\cr
  \cr
  \code{sf}, \code{Spatial}:\cr
  \tab Except for resolution, same as for \code{Raster*}.\cr
  \cr
  Other:\cr
  \tab A \code{map} object.
}
}
\note{
\strong{MODIS} does no longer support the tile identification and automated 
download of MERIS and SRTM data. At least as far as the latter is concerned, 
easy data access is granted through \code{\link{getData}}.
}
\examples{
\dontrun{
# ex 1 ############
# interactive tile selection
getTile()
getTile(mode = "draw")
}

# ex 2: Spatial (taken from ?rgdal::readOGR) ############
dsn <- system.file("vectors/Up.tab", package = "rgdal")[1]
Up <- rgdal::readOGR(dsn, "Up")
getTile(Up)

# ex 3: sf ############
ifl <- system.file("shape/nc.shp", package = "sf")
nc <- sf::st_read(ifl, quiet = TRUE)
getTile(nc)

# ex 4: tileH,tileV ############
getTile(tileH = 18:19, tileV = 4)

# ex 5: Raster* with valid CRS ############
rst1 <- raster(xmn = 9.2, xmx = 17.47, ymn = 46.12, ymx = 49.3)
getTile(rst1)

# this also works for projected data
rst3 <- projectExtent(rst1, crs = "+init=epsg:32633")
getTile(rst3)

# ex 6: Raster* without CRS or, alternatively, Extent or bbox --> treated as EPSG:4326 ############
mat2 <- matrix(seq(180 * 360), byrow = TRUE, ncol = 360)
rst2 <- raster(mat2, xmn = -180, xmx = 180, ymn = -90, ymx = 90)
getTile(rst2)
getTile(extent(rst1))
getTile(sf::st_bbox(nc))

# ex 7: map names as returned by search4map() ############
getTile("Austria")
getTile(c("Austria", "Germany"))

# or search for specific map name patterns (use with caution):
m1 <- search4map("Per")
getTile(m1)

# or use 'map' objects directly (remember to use map(..., fill = TRUE)): 
m2 <- map("state", region = "new jersey", fill = TRUE)
getTile(m2)

}
\seealso{
\code{\link{extent}}, \code{\link[sf]{st_bbox}}, \code{\link{map}}, 
\code{\link{search4map}}.
}
\author{
Matteo Mattiuzzi, Florian Detsch
}
