# checkIntensityRules
#' Check intensity rules
#'
#' Check intensity rules to confirm chains position.
#'
#' @param intrules character vector specifying the fragments to compare. See
#' details.
#' @param rates character vector with the expected rates between fragments given
#' as a string (i.e. "3/1"). See details.
#' @param intrequired logical vector indicating if any of the rules is required.
#' If not, at least one must be verified to confirm the structure.
#' @param nchains number of chains of the targeted lipid class.
#' @param combinations output of \link{combineChains}.
#' @param sn1 list of chain fragments identified for sn1 position. Output of
#' \link{chainFrags}.
#' @param sn2 list of chain fragments identified for sn2 position. Output of
#' \link{chainFrags}. If required.
#' @param sn3 list of chain fragments identified for sn3 position. Output of
#' \link{chainFrags}. If required.
#' @param sn4 list of chain fragments identified for sn4 position. Output of
#' \link{chainFrags}. If required.
#'
#' @details This function will be employed when the targeted lipid class has
#' more than one chain.
#'
#' Taking PG subclass as an example, intensities of lysoPG fragments
#' (informative for sn1) can be employed to confirm the chains structure
#' (intrules = c("lysopg_sn1/lysopg_sn1")). In this case, the intensity of the
#' lysoPG resulting from the loss of the FA chain in sn2 is at least 3 times
#' greater (rates = c("3/1")) than the lysoPG resulting from the loss of the FA
#' chain in sn1.
#'
#' For the intrules argument, "/" will be use to separate the fragments related
#' to each chain (sn1/sn2/etc), and "_" will be use to indicate the list in
#' which they'll be searched. This will depend on the chain fragments
#' rules defined previously. Following the example, as we use lysoPG to define
#' the sn1 position, both fragments will be searched in this list (sn1).
#'
#' For classes with more than one FA chain, if some intensity rule should be
#' employed to identify their position but they are no defined yet, use "Unknown".
#' If it is not necessary because the fragmentation rules are informative enough
#' to define the position (i.e. sphingolipid species), just leave an empty vector.
#'
#' @return List of logical vectors  with the confirmation for each combination.
#'
#' @examples
#' \donttest{
#' dbs <- list(pgdb = LipidMS::pgdb, lysopgdb = LipidMS::lysopgdb,
#' fadb = LipidMS::fadb, adductsTable = LipidMS::adductsTable)
#'
#' candidates <- findCandidates(MS1 = LipidMS::mix_neg_fullMS, dbs[["pgdb"]],
#' ppm = 10, rt = c(min(MS1$RT), max(MS1$RT)), adducts = c("M-H"),
#' rttol = 3, dbs)
#'
#' MSMS <- rbind(LipidMS::mix_neg_Ce20, LipidMS::mix_neg_Ce40)
#' coelfrags <- coelutingFrags(candidates$RT, MSMS, rttol)
#'
#' sn1 <- chainFrags(coelfrags, chainfrags = c("lysopg_M-H"), ppm = 10,
#' dbs = dbs)
#' sn2 <- chainFrags(coelfrags, chainfrags = c("fa_M-H"), ppm = 10, dbs)
#'
#' chainsComb <- combineChains(candidates, nchains=2, sn1, sn2)
#'
#' intConf <- checkIntensityRules(intrules = c("lysopg_sn1/lysopg_sn1"),
#' rates = c("2/1"), intrequired = c(T), nchains=2, chainsComb, sn1, sn2)
#' }
#'
#' @author M Isabel Alcoriza-Balaguer <maialba@alumni.uv.es>
checkIntensityRules <- function(intrules, rates, intrequired, nchains,
                                combinations, sn1, sn2 = list(), sn3 = list(),
                                sn4 = list()){
  intCheck <- list()
  if (nchains == 1){
    for (c in 1:length(sn1)){
      intCheck[[c]] <- checkIntRules(intrules, rates, intrequired, nchains = 1,
                                     combinations[[c]], sn1[[c]])
    }
  } else if (nchains == 2){
    for (c in 1:length(sn1)){
      intCheck[[c]] <- checkIntRules(intrules, rates, intrequired, nchains = 2,
                                     combinations[[c]], sn1[[c]], sn2[[c]])
    }
  } else if (nchains == 3){
    for (c in 1:length(sn1)){
      intCheck[[c]] <- checkIntRules(intrules, rates, intrequired, nchains = 3,
                                     combinations[[c]], sn1[[c]], sn2[[c]],
                                     sn3[[c]])
    }
  } else if (nchains == 4){
    for (c in 1:length(sn1)){
      intCheck[[c]] <- checkIntRules(intrules, rates, intrequired, nchains = 4,
                                     combinations[[c]], sn1[[c]], sn2[[c]],
                                     sn3[[c]], sn4[[c]])
    }
  }
  return(intCheck)
}
