#' Extract results, conduct posterior inference and compute performance metrics for MCMC samples of models from the IMIFA family
#'
#' This function post-processes simulations generated by \code{\link{mcmc_IMIFA}} for any of the IMIFA family of models. It can be re-ran at little computational cost in order to extract different models explored by the sampler used for \code{sims}, without having to re-run the model itself. New results objects using different numbers of clusters and different numbers of factors (if visited by the model in question), or using different model selection criteria (if necessary) can be generated with ease. Posterior predictive checking of the appropriateness of the fitted model is also facilitated.
#' @param sims An object of class "\code{IMIFA}" generated by \code{\link{mcmc_IMIFA}}.
#' @param burnin Optional additional number of iterations to discard. Defaults to 0, corresponding to no additional burnin.
#' @param thinning Optional interval for extra thinning to be applied. Defaults to 1, corresponding to no additional thinning.
#' @param G If this argument is not specified, results will be returned with the optimal number of clusters. If different numbers of clusters were explored in \code{sims} for the "\code{MFA}" or "\code{MIFA}" methods, supplying an integer value allows pulling out a specific solution with \code{G} clusters, even if the solution is sub-optimal.
#'
#' Similarly, this allows retrieval of samples corresponding to a solution, if visited, with \code{G} clusters for the "\code{OMFA}", "\code{OMIFA}", "\code{IMFA}" and "\code{IMIFA}" methods.
#' @param Q If this argument is not specified, results will be returned with the optimal number of factors. If different numbers of factors were explored in \code{sims} for the "\code{FA}", "\code{MFA}", "\code{OMFA}" or "\code{IMFA}" methods, this allows pulling out a specific solution with \code{Q} factors, even if the solution is sub-optimal.
#'
#' Similarly, this allows retrieval of samples corresponding to a solution, if visited, with \code{Q} factors for the "\code{IFA}", "\code{MIFA}", "\code{OMIFA}" and "\code{IMIFA}" methods.
#' @param criterion The criterion to use for model selection, where model selection is only required if more than one model was run under the "\code{FA}", "\code{MFA}", "\code{MIFA}", "\code{OMFA}" or "\code{IMFA}" methods when \code{sims} was created via \code{\link{mcmc_IMIFA}}. Defaults to \code{bicm}, but note that these are \emph{all} calculated; this argument merely indicates which one will form the basis of the construction of the output.
#'
#' Note that the first three options here might exhibit bias in favour of zero-factor models for the finite factor "\code{FA}", "\code{MFA}", "\code{OMFA}" and "\code{IMFA}" methods and might exhibit bias in favour of one-cluster models for the "\code{MFA}" and "\code{MIFA}" methods. The \code{aic.mcmc} and \code{bic.mcmc} criteria will only be returned for finite factor models.
#' @param G.meth If the object in \code{sims} arises from the "\code{OMFA}", "\code{OMIFA}", "\code{IMFA}" or "\code{IMIFA}" methods, this argument determines whether the optimal number of clusters is given by the mode or median of the posterior distribution of \code{G}. Defaults to "\code{mode}". Often the mode and median will agree in any case.
#' @param Q.meth If the object in \code{sims} arises from the "\code{IFA}", "\code{MIFA}", "\code{OMIFA}" or "\code{IMIFA}" methods, this argument determines whether the optimal number of latent factors is given by the mode or median of the posterior distribution of \code{Q}. Defaults to "\code{mode}". Often the mode and median will agree in any case.
#' @param conf.level The confidence level to be used throughout for credible intervals for all parameters of inferential interest, and error metrics if \code{error.metrics} is \code{TRUE}. Defaults to 0.95.
#' @param error.metrics A logical activating or deactivating posterior predictive checking: i.e. controlling whether metrics quantifying the error between the empirical and estimated covariance matrices should be computed for every retained iteration. Defaults to \code{TRUE}, but can be time-consuming for models which achieve clustering.
#'
#' Regardless of the value of \code{error.metrics}, the metrics will still be computed at the posterior mean parameter estimates, \emph{if possible}, with or without computing a distribution of such metrics. These error metrics - and the uncertainty associated with them, if \code{error.metrics} is \code{TRUE} - can be visualised via \code{\link{plot.Results_IMIFA}}.
#'
#' For models which achieve clustering, the overall estimated covariance matrix is constructed from the cluster-specific estimated covariance matrices.
#' @param z.avgsim Logical indicating whether the clustering should also be summarised with a call to \code{\link{Zsimilarity}} by the clustering with minimum mean squared error to the similarity matrix obtained by averaging the stored adjacency matrices, in addition to the MAP estimate.
#'
#' Note that the MAP clustering is computed \emph{conditional} on the estimate of the number of clusters (whether that be the modal estimate or the estimate according to \code{criterion}) and other parameters are extracted conditional on this estimate of \code{G}: however, in constrast, the number of distinct clusters in the summarised labels obtained by specifying \code{z.avgsim=TRUE} may not necessarily coincide with the MAP estimate of \code{G}, but it may provide a useful alternative summary of the partitions explored during the chain, and the user is free to call \code{\link{get_IMIFA_results}} again with the new suggested \code{G} value.
#'
#' Please be warned that this only defaults to \code{TRUE} when the \code{mcclust} package - which \strong{must} be loaded for this feature - is loaded and the number of observations is less than 1000. However, it can still be manually set to \code{TRUE} for larger data sets. This is liable to take considerable time to compute, and may not even be possible if the number of observations &/or number of stored iterations is large and the resulting matrix isn't sufficiently sparse. When \code{TRUE}, both the summarised clustering and the similarity matrix are stored: the latter can be visualised as part of a call to \code{\link{plot.Results_IMIFA}}.
#' @param zlabels For any method that performs clustering, the true labels can be supplied if they are known in order to compute clustering performance metrics. This also has the effect of ordering the MAP labels (and thus the ordering of cluster-specific parameters) to most closely correspond to the true labels if supplied.
#' @param x,object,... Arguments required for the \code{print.Results_IMIFA} and \code{summary.Results_IMIFA} functions: \code{x} and \code{object} are objects of class \code{"Results_IMIFA"} resulting from a call to \code{\link{get_IMIFA_results}}, while \code{...} gathers additional arguments to those functions.
#'
#' @details The function also performs post-hoc corrections for label switching, as well as post-hoc Procrustes rotation of loadings matrices and scores, in order to ensure sensible posterior parameter estimates, computes error metrics, constructs credible intervals, and generally transforms the raw \code{sims} object into an object of class "\code{Results_IMIFA}" in order to prepare the results for plotting via \code{\link{plot.Results_IMIFA}}.
#'
#' @return An object of class "\code{Results_IMIFA}" to be passed to \code{\link{plot.Results_IMIFA}} for visualising results. Dedicated \code{print} and \code{summary} functions also exist for objects of this class. The object, say \code{x}, is a list of lists, the most important components of which are:
#' \item{\code{Clust}}{Everything pertaining to clustering performance can be found here for all but the "\code{FA}" and "\code{IFA}" methods, in particular \code{x$Clust$MAP}, the MAP summary of the posterior clustering, the last valid sample of cluster labels \code{x$Clust$last.z}, the matrix of posterior cluster membership probabilities \code{x$Clust$post.prob}, and the posterior confusion matrix \code{x$Clust$PCM}. More detail is given if known \code{zlabels} are supplied: performance is always evaluated against the MAP clustering, with additional evaluation against the alternative clustering computed if \code{z.avgsim=TRUE}. Posterior summaries of the mixing proportions, and the DP/PY parameters, if necessary, are also included here, as well as the last valid samples of each.}
#' \item{\code{Error}}{Average error metrics (e.g. MSE, RMSE), and credible intervals quantifying the associated uncertainty, between the empirical and estimated covariance matrix/matrices, both of which are also included. The same metrics evaluated at the posterior mean parameter estimates and evaluated the last valid samples are also given. Only relevant if \code{error.metrics} is \code{TRUE}. May not all be returned if loadings and uniquenesses are not stored.}
#' \item{\code{GQ.results}}{Everything pertaining to model choice can be found here, incl. posterior summaries for the estimated number of clusters and estimated number of factors, if applicable to the method employed. Model selection criterion values are also accessible here.}
#' \item{\code{Means}}{Posterior summaries for the means.}
#' \item{\code{Loadings}}{Posterior summaries for the factor loadings matrix/matrices. Posterior mean loadings given by \code{x$Loadings$post.load} are given the \code{\link[stats]{loadings}} class for printing purposes and thus the manner in which they are displayed can be modified.}
#' \item{\code{Scores}}{Posterior summaries for the latent factor scores.}
#' \item{\code{Uniquenesses}}{Posterior summaries for the uniquenesses.}
#'
#' The objects \code{Means}, \code{Loadings}, \code{Scores} and \code{Uniquenesses} (if stored when calling \code{\link{mcmc_IMIFA}}!) also contain, as well as the posterior summaries, the entire chain of valid samples of each, as well as, for convenience, the last valid samples of each (after conditioning on the modal \code{G} and \code{Q} values, and accounting for label switching and Procrustes rotation).
#' @note For the "\code{IMIFA}", "\code{IMFA}", "\code{OMIFA}", and "\code{OMFA}" methods, the retained mixing proportions are renormalised after conditioning on the modal \code{G}. This is especially necessary for the compution of the \code{error.metrics}, just note that the values on which posterior inference are conducted will ever so slightly differ from the actually sampled values.
#'
#' Due to the way the offline label-switching correction is performed, different runs of this function may give \emph{very slightly} different results in terms of the cluster labellings (and by extension the parameters, which are permuted in the same way), but only if the chain was run for an extremely small number of iterations, well below the number required for convergence, and samples of the cluster labels match poorly across iterations (particularly if the number of clusters suggested by those sampled labels is high).
#' @keywords IMIFA main
#' @include MainFunction.R
#' @export
#' @importFrom Rfast "colMaxs" "colTabulate" "med" "rowMaxs" "rowmeans" "rowOrder" "sort_mat" "sort_unique" "Var"
#' @importFrom mclust "classError"
#' @importFrom matrixStats "colSums2" "rowMedians" "rowQuantiles" "rowSums2"
#' @importFrom slam "as.simple_triplet_matrix"
#'
#' @seealso \code{\link{mcmc_IMIFA}}, \code{\link{plot.Results_IMIFA}}, \code{\link{Procrustes}}, \code{\link{Zsimilarity}}
#' @references Murphy, K., Gormley, I. C. and Viroli, C. (2017) Infinite Mixtures of Infinite Factor Analysers: Nonparametric Model-Based Clustering via Latent Gaussian Models, \emph{to appear}. <\href{https://arxiv.org/abs/1701.07010v4}{arXiv:1701.07010v4}>.
#'
#' @author Keefe Murphy - <\email{keefe.murphy@@ucd.ie}>
#' @usage
#' get_IMIFA_results(sims = NULL,
#'                   burnin = 0L,
#'                   thinning = 1L,
#'                   G = NULL,
#'                   Q = NULL,
#'                   criterion = c("bicm", "aicm", "dic", "bic.mcmc", "aic.mcmc"),
#'                   G.meth = c("mode", "median"),
#'                   Q.meth = c("mode", "median"),
#'                   conf.level = 0.95,
#'                   error.metrics = TRUE,
#'                   z.avgsim = TRUE,
#'                   zlabels = NULL)
#' @examples
#' # data(coffee)
#' # data(olive)
#'
#' # Run a MFA model on the coffee data over a range of clusters and factors.
#' # simMFAcoffee  <- mcmc_IMIFA(coffee, method="MFA", range.G=2:3, range.Q=0:3, n.iters=1000)
#'
#' # Accept all defaults to extract the optimal model.
#' # resMFAcoffee  <- get_IMIFA_results(simMFAcoffee)
#'
#' # Instead let's get results for a 3-cluster model, allowing Q be chosen by aic.mcmc.
#' # resMFAcoffee2 <- get_IMIFA_results(simMFAcoffee, G=3, criterion="aic.mcmc")
#'
#' # Run an IMIFA model on the olive data, accepting all defaults.
#' # simIMIFAolive <- mcmc_IMIFA(olive, method="IMIFA", n.iters=10000)
#'
#' # Extract optimum results
#' # Estimate G & Q by the median of their posterior distributions
#' # Construct 90% credible intervals and try to return the similarity matrix.
#' # resIMIFAolive <- get_IMIFA_results(simIMIFAolive, G.meth="median", Q.meth="median",
#' #                                    conf.level=0.9, z.avgsim=TRUE)
#' # summary(resIMIFAolive)
get_IMIFA_results              <- function(sims = NULL, burnin = 0L, thinning = 1L, G = NULL, Q = NULL, criterion = c("bicm", "aicm", "dic", "bic.mcmc", "aic.mcmc"),
                                           G.meth = c("mode", "median"), Q.meth = c("mode", "median"), conf.level = 0.95, error.metrics = TRUE, z.avgsim = TRUE, zlabels = NULL) {
  UseMethod("get_IMIFA_results")
}

#' @export
get_IMIFA_results.IMIFA        <- function(sims = NULL, burnin = 0L, thinning = 1L, G = NULL, Q = NULL, criterion = c("bicm", "aicm", "dic", "bic.mcmc", "aic.mcmc"),
                                           G.meth = c("mode", "median"), Q.meth = c("mode", "median"), conf.level = 0.95, error.metrics = TRUE, z.avgsim = TRUE, zlabels = NULL) {

  call           <- match.call()
  defopt         <- options()
  options(warn=1)
  on.exit(suppressWarnings(options(defopt)), add=TRUE)
  if(missing(sims))               stop("Simulations must be supplied", call.=FALSE)
  if(class(sims) != "IMIFA")      stop("Object of class 'IMIFA' must be supplied", call.=FALSE)
  if(!exists(deparse(substitute(sims)),
             envir=.GlobalEnv))   stop(paste0("Object ", match.call()$sims, " not found\n"), call.=FALSE)
  burnin         <- as.integer(burnin)
  thinning       <- as.integer(thinning)
  store          <- attr(sims, "Store")
  if(any(c(length(thinning),
           length(burnin)) > 1))  stop("'burnin' and 'thinning' must be of length 1", call.=FALSE)
  if(any(burnin   < 0, burnin >= store,
         thinning < 1))           stop("Invalid 'burnin' and/or 'thinning' supplied", call.=FALSE)
  store          <- seq(from=burnin + 1, to=store, by=thinning)
  if(length(store) < 10)          stop("Too much 'burnin' or 'thinning' applied: not enough stored samples to proceed", call.=FALSE)
  n.store        <- length(store)
  tmp.store      <- store

  dat            <- attr(sims, "Dataset")
  data.name      <- attr(sims, "Name")
  equal.pro      <- attr(sims, "Equal.Pi")
  label.switch   <- attr(sims, "Label.Switch")
  method         <- attr(sims, "Method")
  learn.alpha    <- attr(sims, "Alph.step")
  learn.d        <- attr(sims, "Disc.step")
  inf.G          <- is.element(method, c("IMIFA", "OMIFA", "IMFA", "OMFA"))
  inf.Q          <- is.element(method, c("IMIFA", "OMIFA", "MIFA",  "IFA"))
  n.fac          <- attr(sims, "Factors")
  n.grp          <- attr(sims, "Clusters")
  n.obs          <- attr(sims, "Obs")
  n.var          <- attr(sims, "Vars")
  uni            <- n.var == 1
  sw             <- tmpsw <- attr(sims, "Switch")
  cent           <- attr(sims, "Center")
  scaling        <- attr(sims, "Scaling")
  scal.meth      <- attr(scaling, "Method")
  uni.meth       <- attr(sims, "Uni.Meth")
  uni.type       <- unname(uni.meth["Uni.Type"])
  conf.level     <- as.numeric(conf.level)
  obsnames       <- attr(sims, "Obsnames")
  varnames       <- attr(sims, "Varnames")
  cov.emp        <- attr(sims, "Cov.Emp")
  cov.range      <- ifelse(uni, 1, max(cov.emp) - min(cov.emp))
  if(any(length(conf.level) != 1,
     !is.numeric(conf.level),
     (conf.level <= 0   ||
      conf.level >= 1)))          stop("'conf.level' must be a single number in the interval (0, 1)", call.=FALSE)
  conf.levels    <- c((1 - conf.level)/2, (1 + conf.level)/2)
  choice         <- length(n.grp) * length(n.fac) > 1
  crit.check     <- !all(is.character(criterion))
  if(isTRUE(crit.check))          stop("'criterion' must be a character vector of length 1", call.=FALSE)
  criterion      <- match.arg(criterion)
  if(all(inf.Q, is.element(criterion,
     c("aic.mcmc", "bic.mcmc")))) stop(paste0(ifelse(isTRUE(choice), "Model choice is", "Though model choice isn't"), " actually required -\n'criterion' cannot be 'aic.mcmc' or 'bic.mcmc' for the ", method, " method"), call.=FALSE)
  if(length(error.metrics) != 1  ||
     !is.logical(error.metrics))  stop("'error.metrics' must be a single logical indicator", call.=FALSE)
  if(isTRUE(error.metrics) &&
     anyNA(cov.emp))        {     warning("'error.metrics' forced to FALSE as there are missing values in the empirical covariance matrix", call.=FALSE)
   error.metrics <- FALSE
  }
  miss.zavg      <- missing(z.avgsim)
  if(length(z.avgsim)      != 1  ||
     !is.logical(z.avgsim))       stop("'z.avgsim' must be a single logical indicator", call.=FALSE)
  if(isTRUE(z.avgsim))  {
    if(!(has.pkg <- suppressMessages(requireNamespace("mcclust", quietly=TRUE)))) {
      z.avgwarn  <- "Forcing 'z.avgsim' to FALSE: 'mcclust' package not installed"
      z.avgsim   <- FALSE
      if(miss.zavg)     {         message(z.avgwarn)
      } else                      warning(z.avgwarn, call.=FALSE, immediate.=TRUE)
    } else if(miss.zavg    &&
      !(z.avgsim <- n.obs
                  < 1000))        message("Forcing 'z.avgsim' to FALSE as the number of observations is quite large")
  }

  G.T            <- !missing(G)
  Q.T            <- !missing(Q)
  G.ind          <- Q.ind      <- 1L
  GQs            <- length(sims[[G.ind]])
  GQ1x           <- GQs > 1    && is.element(method, c("OMFA", "IMFA"))
  if(inf.G)  {
    G.store2     <- lapply(seq_len(GQs), function(gq) sims[[G.ind]][[gq]]$G.store[store])
    G.store      <- matrix(unlist(G.store2), nrow=GQs, ncol=n.store, byrow=TRUE)
    if(is.element(method, c("IMFA", "IMIFA"))) {
      act.store  <- lapply(seq_len(GQs), function(gq) sims[[G.ind]][[gq]]$act.store[store])
    }
    G.mX         <- !all(is.character(G.meth))
    if(isTRUE(G.mX))              stop("'G.meth' must be a character vector of length 1", call.=FALSE)
    G.meth       <- match.arg(G.meth)
    G.tab        <- if(GQ1x) lapply(apply(G.store, 1, function(x) list(table(x, dnn=NULL))), "[[", 1) else table(G.store, dnn=NULL)
    G.prob       <- if(GQ1x) lapply(G.tab, prop.table) else prop.table(G.tab)
    G.mode       <- if(GQ1x) unlist(lapply(G.tab, function(gt) as.numeric(names(gt[gt == max(gt)])[1]))) else as.numeric(names(G.tab[G.tab == max(G.tab)])[1])
    G.med        <- if(GQ1x) ceiling(matrixStats::rowMedians(G.store) * 2)/2 else ceiling(med(G.store) * 2)/2
    if(!G.T) {
      G          <- switch(G.meth, mode=G.mode, floor(G.med))
    }
    G.CI         <- if(GQ1x) round(rowQuantiles(G.store, probs=conf.levels)) else round(stats::quantile(G.store, conf.levels))
  }

  if(G.T)    {
    G            <- as.integer(G)
    if(any(length(G) != 1,
           !is.integer(G)))       stop("'G' must be an integer of length 1", call.=FALSE)
    if(!inf.G) {
      if(!is.element(method, c("FA", "IFA")))  {
        if(!is.element(G, n.grp)) stop("This 'G' value was not used during simulation", call.=FALSE)
        G.ind    <- which(n.grp == G)
      } else if(G > 1)            message(paste0("Forced G=1 for the ", method, " method"))
    } else     {
      if(GQ1x) {
        if(!Q.T)                  stop(paste0("'G' cannot be supplied without 'Q' for the ", method, " method if a range of Q values were explored"), call.=FALSE)
        tmpQ     <- which(n.fac == unique(Q))
      } else {
        tmpQ     <- Q.ind
      }
      if(length(tmpQ > 0)  && !is.element(G,
         unique(G.store[tmpQ,]))) stop("This 'G' value was not visited during simulation", call.=FALSE)
    }
  }
  G              <- if(any(inf.G, all(G.T, !is.element(method, c("FA", "IFA"))))) G else 1L

  if(Q.T)    {
    Q            <- as.integer(Q)
    if(!is.integer(Q))            stop("'Q' must of integer type", call.=FALSE)
    if(G.T)  {
      if(length(Q) == 1)     Q <- rep(Q, G)
      if(length(Q) != G)          stop(paste0("'Q' must be supplied for each cluster, as a scalar or vector of length G=", G), call.=FALSE)
    } else if(length(n.grp)    != 1 && all(!is.element(length(Q),
              c(1,  n.grp))))     stop("'Q' must be a scalar if G=1, 'G' is not suppplied, or a range of G values were explored", call.=FALSE)
    if(all(is.element(method, c("FA", "MFA", "OMFA", "IMFA")))) {
      if(length(unique(Q)) != 1)  stop(paste0("'Q' cannot vary across clusters for the ", method, " method"), call.=FALSE)
      Q          <- unique(Q)
      if(!is.element(Q,   n.fac)) stop("This 'Q' value was not used during simulation", call.=FALSE)
      Q.ind      <- which(n.fac == Q)
    }
    if(inf.Q)  {
      if(any((Q  != 0) + (Q *
        (n.var - Q)   <= 0) > 1)) stop(paste0("'Q' must be less than the number of variables ", n.var), call.=FALSE)
      Qtmp       <- if(inf.G) sims[[1]][[1]]$Q.store[seq_len(G),store, drop=FALSE] else switch(method, MIFA=sims[[if(G.T) G == n.grp else G.ind]][[1]]$Q.store[,store, drop=FALSE], sims[[1]][[1]]$Q.store[,store, drop=FALSE])
      storage.mode(Qtmp)   <- switch(method, IFA="integer", "numeric")
      Qtmp       <- switch(method, IFA=max(Qtmp), Rfast::rowMaxs(Qtmp, value=TRUE))
      if(any(Q * (Qtmp - Q) < 0)) stop(paste0("'Q' can't be greater than the maximum number of factors stored in ", ifelse(method == "IFA", "", "any cluster of "), match.call()$sims), call.=FALSE)
    }
  }

  if(inf.G)    {
    tmp.store    <- if(GQ1x) lapply(seq_len(GQs), function(gq) store[G.store[gq,] == G[ifelse(G.T, 1, gq)]]) else store[G.store == G]
    GQ.temp1     <- list(G = G, G.Mode = G.mode, G.Median = G.med, G.CI = G.CI, G.Probs = G.prob, G.Counts = G.tab)
    GQ.temp1     <- c(GQ.temp1, list(Stored.G = switch(method, OMIFA=provideDimnames(G.store, base=list("Non-Empty", ""),    unique=FALSE),
                      IMIFA=provideDimnames(do.call(rbind, c(G.store2, act.store)), base=list(c("Non-Empty", "Active"), ""), unique=FALSE),
                      OMFA=lapply(seq_len(GQs), function(g) provideDimnames(t(G.store[g,]),   base=list("Non-Empty", ""),    unique=FALSE)),
                      IMFA=lapply(seq_len(GQs), function(g) provideDimnames(rbind(G.store2[[g]], act.store[[g]]), base=list(c("Non-Empty", "Active"), ""), unique=FALSE)))))
    GQ.temp1     <- c(GQ.temp1, list(G.Last = switch(method, OMIFA=, IMIFA=GQ.temp1$Stored.G[1,n.store],
                                                lapply(seq_len(GQs), function(g) GQ.temp1$Stored.G[[g]][1,n.store]))))
  }
  G.range        <- ifelse(G.T, 1, length(n.grp))
  Q.range        <- ifelse(any(Q.T, all(!is.element(method, c("OMFA", "IMFA")), inf.Q)), 1, length(n.fac))
  crit.mat       <- matrix(NA, nrow=G.range, ncol=Q.range)

  # Retrieve log-likelihoods and/or tune G &/or Q according to criterion
    if(all(G.T, Q.T)) {
      dimnames(crit.mat) <- list(paste0("G", G),     if(inf.Q) "IFA" else paste0("Q", Q))
    } else if(G.T)    {
      dimnames(crit.mat) <- list(paste0("G", G),     if(inf.Q) "IFA" else paste0("Q", n.fac))
    } else if(Q.T)    {
      dimnames(crit.mat) <- list(paste0("G", n.grp), if(inf.Q) "IFA" else paste0("Q", Q))
    } else {
      dimnames(crit.mat) <- list(paste0("G", n.grp), if(inf.Q) "IFA" else paste0("Q", n.fac))
    }
    rownames(crit.mat)   <- switch(method, IMFA=, IMIFA="IM", OMFA=, OMIFA="OM", rownames(crit.mat))
    aicm         <- bicm       <- aicm.sd   <- dic   <-
    aic.mcmc     <- bic.mcmc   <- bicm.sd   <- crit.mat
    log.N        <- log(n.obs)

    for(g in seq_len(G.range))   {
      gi                 <- ifelse(G.T, G.ind, g)
      for(q in seq_len(Q.range)) {
        qi               <- ifelse(Q.T, Q.ind, q)
        log.likes        <- if(GQ1x) sims[[gi]][[qi]]$ll.store[tmp.store[[qi]]] else sims[[gi]][[qi]]$ll.store[tmp.store]
        log.likes        <- log.likes[stats::complete.cases(log.likes)]
        S2               <- ifelse(length(log.likes) != 1, Var(log.likes), 0)
        llbar            <- mean(log.likes)
        d.hat            <- 2  * S2
        llmax2           <- 2  * max(log.likes)
        llmax2a          <- 2  * llbar   + d.hat
        aicm[g,q]        <- 4  * llbar   - llmax2a
        bicm[g,q]        <- llmax2a  - d.hat * log.N
        dic[g,q]         <- 2  * (3  * llbar - llmax2a)
        ci.tmp           <- S2 * (11 * S2    + 24)
        aicm.sd[g,q]     <- sqrt((4  * (S2   + ci.tmp))/length(log.likes))
        bicm.sd[g,q]     <- sqrt((2  * d.hat + (log.N - 1)^2 * ci.tmp)/length(log.likes))
        if(!inf.Q) {
          K              <- switch(method, OMFA=, IMFA=PGMM_dfree(Q=n.fac[qi], P=n.var, G=G[ifelse(G.T, 1, qi)],
                            method=switch(uni.type, unconstrained="UUU", isotropic="UUC", constrained="UCU", single="UCC"),
                            equal.pro=equal.pro), attr(sims[[gi]][[qi]], "K"))
          aic.mcmc[g,q]  <- llmax2   - K * 2
          bic.mcmc[g,q]  <- llmax2   - K * log.N
        }
      }
    }
    crit         <- get(criterion)
    crit.max     <- which(crit == max(crit), arr.ind=TRUE)

  # Control for supplied values of G &/or Q
    if(!any(Q.T, G.T)) {
      G.ind      <- crit.max[1]
      Q.ind      <- crit.max[2]
      if(!inf.G) {
        G        <- n.grp[G.ind]
      }
      if(!inf.Q) {
        Q        <- n.fac[Q.ind]
      }
    } else if(all(G.T, !Q.T)) {
      Q.ind      <- which.max(crit)
      if(!inf.Q) {
        Q        <- n.fac[Q.ind]
      }
    } else if(all(Q.T, !G.T)) {
      G.ind      <- which.max(crit)
      if(!inf.G) {
        G        <- n.grp[G.ind]
      }
    }
    G            <- ifelse(inf.G, ifelse(G.T, G, G[Q.ind]), ifelse(length(n.grp)  == 1, n.grp, G))
    Gseq         <- seq_len(G)
    gnames       <- paste0("Cluster", Gseq)
    G.ind        <- if(!all(inf.G, length(n.grp) > 1)) G.ind     else which(n.grp == G)
    GQ.temp2     <- list(AICMs = aicm, BICMs = bicm, DICs = dic)
    if(GQ1x)     {
      tmp.store  <- tmp.store[[Q.ind]]
    }
    storeG       <- seq_along(tmp.store)

    if(!inf.Q)   {
      Q          <- if(length(n.fac)   > 1)  Q                   else n.fac
      Q.ind      <- if(all(!Q.T, length(n.fac) > 1)) Q.ind       else which(n.fac == Q)
      Q          <- stats::setNames(if(length(Q) != G) rep(Q, G) else Q, gnames)
      if(all(inf.G, Q.T))  GQ.temp1$G <- rep(G, GQs)
      if(GQ1x)   {
        GQ.temp1$G.CI     <- lapply(seq_len(GQs), function(gq) GQ.temp1$G.CI[gq,])
        GQ.temp1 <- lapply(GQ.temp1, "[[", Q.ind)
      } else if(inf.G) {
        GQ.temp1$Stored.G <- GQ.temp1$Stored.G[[1]]
        GQ.temp1$G.Last   <- GQ.temp1$G.Last[[1]]
      }
      GQ.temp3   <- c(GQ.temp2, list(AIC.mcmcs = aic.mcmc, BIC.mcmcs = bic.mcmc))
      GQ.res     <- switch(method, OMFA=, IMFA=c(GQ.temp1, list(Q = Q), list(Criteria = GQ.temp3)), c(list(G = G, Q = Q), list(Criteria = GQ.temp3)))
    }

    clust.ind    <- !any(is.element(method,   c("FA", "IFA")),
                     all(is.element(method, c("MFA", "MIFA")), G == 1))
    sw.mx        <- ifelse(clust.ind, sw["mu.sw"],  TRUE)
    sw.px        <- ifelse(clust.ind, sw["psi.sw"], TRUE)
    if(inf.Q) {
      Q.store    <- sims[[G.ind]][[Q.ind]]$Q.store[,tmp.store, drop=FALSE]
      Q.mX       <- !all(is.character(Q.meth))
      if(isTRUE(Q.mX))            stop("'Q.meth' must be a character vector of length 1", call.=FALSE)
      Q.meth     <- match.arg(Q.meth)
    }
    TN.store     <- length(tmp.store)
    if(TN.store   < 2)            stop(paste0("Not enough samples stored to proceed", ifelse(any(G.T, Q.T), paste0(": try supplying different Q or G values"), "")), call.=FALSE)

# Manage Label Switching & retrieve cluster labels/mixing proportions
  if(clust.ind) {
    label.miss   <- missing(zlabels)
    if(!label.miss && (all(!is.factor(zlabels), !is.numeric(zlabels)) ||
       length(zlabels) != n.obs)) stop(paste0("'zlabels' must be a factor of length N=",  n.obs), call.=FALSE)
    if(sw["mu.sw"])   {
      mus        <- sims[[G.ind]][[Q.ind]]$mu[,,tmp.store, drop=FALSE]
    }
    if(sw["l.sw"])    {
      lmats      <- if(inf.Q) as.array(sims[[G.ind]][[Q.ind]]$load)[,,,tmp.store, drop=FALSE] else sims[[G.ind]][[Q.ind]]$load[,,,tmp.store, drop=FALSE]
    }
    if(sw["psi.sw"])  {
      psis       <- sims[[G.ind]][[Q.ind]]$psi[,,tmp.store, drop=FALSE]
    }
    if(sw["pi.sw"])   {
      pies       <- sims[[G.ind]][[Q.ind]]$pi.prop[,tmp.store, drop=FALSE]
    }
    zadj         <- sims[[G.ind]][[Q.ind]]$z.store
    z            <- as.matrix(zadj[tmp.store,])
    zadj         <- zadj[store,]

    if(!label.switch) {
      z.temp     <- tryCatch(factor(z[1,], labels=Gseq), error=function(e) factor(z[1,], levels=Gseq))
      for(sl in storeG)    {
        sw.lab   <- .lab_switch(z.new=z[sl,], z.old=z.temp)
        z.perm   <- sw.lab$z.perm
        left     <- as.integer(unname(z.perm))
        right    <- as.integer(names(z.perm))
        if(!identical(left, right))   {
          z[sl,] <- sw.lab$z
          if(sw["mu.sw"])  {
            mus[,left,sl]      <- mus[,right,sl]
          }
          if(sw["l.sw"])   {
            lmats[,,left,sl]   <- lmats[,,right,sl]
          }
          if(sw["psi.sw"]) {
            psis[,left,sl]     <- psis[,right,sl]
          }
          if(sw["pi.sw"])  {
            pies[left,sl]      <- pies[right,sl]
          }
          if(inf.Q)        {
            Q.store[left,sl]   <- Q.store[right,sl]
          }
        }
      }
    }
    if(sw["mu.sw"])        mus <- tryCatch(mus[,Gseq,,     drop=FALSE], error=function(e) mus)
    if(sw["l.sw"])       lmats <- tryCatch(lmats[,,Gseq,,  drop=FALSE], error=function(e) lmats)
    if(sw["psi.sw"])      psis <- tryCatch(psis[,Gseq,,    drop=FALSE], error=function(e) psis)
    MAP          <- apply(z, 2,   function(x) factor(which.max(tabulate(x, nbins=G)), levels=Gseq))
    post.prob    <- matrix(colTabulate(z, max_number=G)/TN.store, nrow=n.obs, ncol=G, byrow=TRUE)

    if(isTRUE(z.avgsim)) {
      znew       <- try(Zsimilarity(zs=zadj), silent=TRUE)
      condit     <- all(!is.element(method, c("MIFA", "MFA")), inherits(znew, "try-error"))
      if(isTRUE(condit)) {
        znew     <- try(Zsimilarity(zs=z),    silent=TRUE)
                                  warning("Constructing the similarity matrix failed:\ntrying again using iterations corresponding to the modal number of clusters", call.=FALSE)
      }
      if(!inherits(znew, "try-error")) {
        zadj     <- znew$z.avg
        zadj     <- factor(zadj, labels=seq_along(unique(zadj)))
        zadj     <- as.integer(levels(zadj))[zadj]
        zavg     <- znew$z.sim
        if(!label.miss) {
         zlabels <- factor(zlabels, labels=seq_along(unique(zlabels)))
         zadj    <- .lab_switch(z.new=zadj, z.old=zlabels)$z
         tab     <- table(zadj, zlabels, dnn=list("Predicted", "Observed"))
         tabstat <- c(.class_agreement(tab), classError(MAP, zlabels))
         if(nrow(tab) != ncol(tab))    {
         tabstat <- tabstat[-seq_len(2)]
           names(tabstat)[4]   <- "error.rate"
         } else {
           names(tabstat)[6]   <- "error.rate"
         }
         if(tabstat$error.rate == 0)   {
         tabstat$misclassified <- NULL
         }
         tabstat <- c(list(confusion.matrix = tab), tabstat)
         class(tabstat)        <- "listof"
        }
        z_simavg <- list(z.avg = zadj, z.sim = zavg)
        z_simavg <- c(z_simavg, if(!label.miss) list(avgsim.perf = tabstat))
        attr(z_simavg, "Conditional")   <- condit
      } else {                    warning("Can't compute similarity matrix or 'average' clustering: forcing 'z.avgsim' to FALSE", call.=FALSE)
        z.avgsim <- FALSE
      }
    }

    uncertain    <- 1 - Rfast::rowMaxs(post.prob, value=TRUE)
    if(sw["pi.sw"]) {
      pi.prop    <- provideDimnames(pies[Gseq,storeG, drop=FALSE], base=list(gnames, ""), unique=FALSE)
      pi.prop    <- if(inf.G) sweep(pi.prop, 2, colSums2(pi.prop), FUN="/", check.margin=FALSE) else pi.prop
      var.pi     <- stats::setNames(.row_vars(pi.prop), gnames)
      ci.pi      <- rowQuantiles(pi.prop, probs=conf.levels)
      ci.pi      <- if(G == 1) t(ci.pi) else ci.pi
      post.pi    <- stats::setNames(rowmeans(pi.prop),  gnames)
    } else {
      post.pi    <- stats::setNames(prop.table(tabulate(MAP, nbins=G)),    gnames)
    }
    if(inf.Q)       {
      Q.store    <- provideDimnames(Q.store[Gseq,, drop=FALSE],  base=list(gnames, ""), unique=FALSE)
    }

    sizes        <- stats::setNames(tabulate(MAP, nbins=G), gnames)
    if(any(sizes == 0))           warning(paste0("Empty cluster exists in modal clustering:\nexamine trace plots", ifelse(any(is.element(method, c("OMFA", "IMFA", "OMIFA", "IMIFA")), is.element(method, c("MFA", "MIFA")) && any(n.grp < G)), ", try to supply a lower G value to get_IMIFA_results(),", ""), " or re-run the model"), call.=FALSE)
    if(!label.miss) {
      zlabels    <- factor(zlabels, labels=seq_along(unique(zlabels)))
      sw.lab     <- .lab_switch(z.new=MAP, z.old=zlabels)
      MAP        <- factor(factor(sw.lab$z, labels=which(sizes > 0)), levels=Gseq)
      l.perm     <- sw.lab$z.perm
      left       <- as.integer(l.perm[Gseq])
      z.tmp      <- lapply(storeG, function(i)   factor(z[i,], labels=order(left[tabulate(z[i,], nbins=G) > 0])))
      z          <- do.call(rbind, lapply(z.tmp, function(x) as.integer(levels(as.factor(x)))[as.integer(x)]))
      index      <- order(left)
      if(sw["mu.sw"])      mus <- mus[,index,,    drop=FALSE]
      if(sw["l.sw"])     lmats <- lmats[,,index,, drop=FALSE]
      if(sw["psi.sw"])    psis <- psis[,index,,   drop=FALSE]
      post.pi    <- stats::setNames(post.pi[index], gnames)
      if(sw["pi.sw"]) {
       pi.prop   <- provideDimnames(unname(pi.prop[index,, drop=FALSE]), base=list(gnames, ""), unique=FALSE)
       var.pi    <- stats::setNames(var.pi[index],  gnames)
       ci.pi     <- provideDimnames(unname(ci.pi[index,,   drop=FALSE]), base=list(gnames,  colnames(ci.pi)))
      }
      if(inf.Q)   {
        Q.store  <- provideDimnames(unname(Q.store[index,, drop=FALSE]), base=list(gnames, ""), unique=FALSE)
      }
      tab        <- table(MAP, zlabels, dnn=list("Predicted", "Observed"))
      tab.stat   <- c(.class_agreement(tab), classError(MAP, zlabels))
      if(nrow(tab) != ncol(tab))     {
        tab.stat <- tab.stat[-seq_len(2)]
        names(tab.stat)[4]     <- "error.rate"
      } else {
        names(tab.stat)[6]     <- "error.rate"
      }
      if(tab.stat$error.rate   == 0) {
        tab.stat$misclassified <- NULL
      }
      tab.stat   <- c(list(confusion.matrix = tab), tab.stat)
      class(tab.stat)          <- "listof"
    }

    if(learn.alpha) {
      alpha      <- sims[[G.ind]][[Q.ind]]$alpha[store]
      post.alpha <- mean(alpha)
      var.alpha  <- Var(alpha)
      ci.alpha   <- stats::quantile(alpha, conf.levels)
      rate       <- sims[[G.ind]][[Q.ind]]$a.rate
      DP.alpha   <- list(alpha = alpha, post.alpha = post.alpha, var.alpha = var.alpha, ci.alpha = ci.alpha, alpha.rate = rate, last.alpha = alpha[n.store])
      DP.alpha   <- c(DP.alpha, if(isTRUE(attr(sims, "TuneZeta"))) list(avg.zeta = sims[[G.ind]][[Q.ind]]$avg.zeta))
      class(DP.alpha)          <- "listof"
    }

    if(learn.d)     {
      discount   <- as.vector(sims[[G.ind]][[Q.ind]]$discount[store])
      post.disc  <- mean(discount)
      post.kappa <- sum(discount == 0)/n.store
      var.disc   <- Var(discount)
      ci.disc    <- stats::quantile(discount, conf.levels)
      rate       <- sims[[G.ind]][[Q.ind]]$d.rate
      post.dzero <- post.disc/(1  - post.kappa)
      discount   <- if(sum(discount  == 0)/n.store > 0.5) as.simple_triplet_matrix(discount)  else discount
      PY.disc    <- list(discount = discount, post.disc = post.disc, post.kappa = post.kappa, var.disc = var.disc,
                         ci.disc  = ci.disc,  disc.rate = rate, last.disc = discount[n.store], post.d_nonzero = post.dzero)
      class(PY.disc)           <- "listof"
    }

    MAP          <- as.integer(levels(MAP))[MAP]
    uncert.obs   <- which(uncertain  >= 1/G)
    uncertain    <- if(sum(uncertain == 0)/n.obs   > 0.5)  as.simple_triplet_matrix(uncertain) else uncertain
    attr(uncertain, "Obs")     <- if(sum(uncert.obs) != 0) uncert.obs
    if(!label.miss) tab.stat$uncertain            <-       attr(uncertain, "Obs")
    cluster      <- list(MAP = MAP, z = z, uncertainty = uncertain, last.z = z[TN.store,])
    cluster      <- c(cluster, list(post.sizes  = sizes, post.ratio  = sizes/n.obs, post.pi = post.pi/sum(post.pi),
                                    post.prob   = post.prob,  PCM    = post_conf_mat(post.prob)),
                      if(sw["pi.sw"]) list(pi.prop = pi.prop, var.pi = var.pi, ci.pi = ci.pi, last.pi = pi.prop[,TN.store]),
                      if(!label.miss) list(perf = tab.stat),
                      if(learn.alpha) list(DP.alpha = DP.alpha),
                      if(learn.d)     list(PY.disc = PY.disc),
                      if(z.avgsim)    list(Z.avgsim = z_simavg),
                      if(is.element(method, c("IMFA", "IMIFA"))) list(lab.rate = sims[[G.ind]][[Q.ind]]$lab.rate))
    attr(cluster, "Z.init")    <- attr(sims[[G.ind]], "Z.init")
    attr(cluster, "Init.Meth") <- attr(sims, "Init.Z")
    attr(cluster, "Label.Sup") <- !label.miss
    z.ind        <- lapply(Gseq, function(g) MAP == g)
  } else      {
    z.ind        <- list(seq_len(n.obs))
    sizes        <- n.obs
  }

  if(inf.Q)   {
    G1           <- G > 1
    Q.tab        <- if(G1) lapply(apply(Q.store, 1, function(x) list(table(x, dnn=NULL))), "[[", 1)    else table(Q.store, dnn=NULL)
    Q.prob       <- if(G1) lapply(Q.tab, prop.table) else prop.table(Q.tab)
    Q.mode       <- if(G1) unlist(lapply(Q.tab, function(qt) as.numeric(names(qt[qt == max(qt)])[1]))) else as.numeric(names(Q.tab[Q.tab == max(Q.tab)])[1])
    Q.med        <- if(G1) stats::setNames(ceiling(matrixStats::rowMedians(Q.store) * 2)/2, gnames)    else ceiling(med(Q.store) * 2)/2
    if(!Q.T)  {
      Q          <- switch(Q.meth, mode=Q.mode, floor(Q.med))
    } else    {
      Q          <- if(G.T) Q else stats::setNames(rep(Q, G), gnames)
    }
    leder.b      <- min(n.obs - 1, Ledermann(n.var))
    if(any(unlist(Q) > leder.b))  warning(paste0("Estimate of Q", ifelse(G > 1, " in one or more clusters ", " "), "is greater than ", ifelse(any(unlist(Q) > n.var), paste0("the number of variables (", n.var, ")"), paste0("the suggested Ledermann upper bound (", leder.b, ")")), ":\nsolution may be invalid"), call.=FALSE)
    Q.CI         <- if(G1) round(rowQuantiles(Q.store, probs=conf.levels)) else round(stats::quantile(Q.store, conf.levels))
    GQ.temp4     <- list(Q = Q, Q.Mode = Q.mode, Q.Median = Q.med,
                         Q.CI = Q.CI, Q.Probs = Q.prob, Q.Counts = Q.tab,
                         Stored.Q = if(clust.ind) Q.store else as.vector(Q.store),
                         Q.Last = Q.store[,TN.store])
    GQ.res       <- if(inf.G) c(GQ.temp1, GQ.temp4) else c(list(G = G), GQ.temp4)
    GQ.res       <- c(GQ.res, list(Criteria = GQ.temp2))
    attr(GQ.res, "Q.big") <- attr(sims[[G.ind]][[Q.ind]], "Q.big")
  }
  Q0             <- Q == 0
  if(all(isTRUE(choice), is.element(criterion, c("aicm", "bicm", "dic")))) {
    if(all(!G.T, !is.element(method,
       c("FA", "IFA")), G == 1))  warning(paste0("Chosen model has only one group:\nNote that the ", criterion, " criterion may exhibit bias toward one-group models"),   call.=FALSE)
    if(all(!Q.T, method   == "MIFA")) {
      if(any(Q0))                 warning(paste0("Chosen model has ", ifelse(sum(Q0) == G, "zero factors", "a cluster with zero factors"), ":\nNote that the ", criterion, " criterion may exhibit bias toward models ", ifelse(sum(Q0) == G, "with zero factors", "where some clusters have zero factors")), call.=FALSE)
    } else if(all(Q0))            warning(paste0("Chosen model has zero factors:\nNote that the ",   criterion, " criterion may exhibit bias toward zero-factor models"), call.=FALSE)
  }

# Retrieve (unrotated) scores
  if(no.score    <- all(Q0)) {
    if(sw["s.sw"])                message("Scores & loadings not stored as model has zero factors")
    sw["s.sw"]   <- FALSE
  }
  if(inf.Q) {
    Lstore       <- lapply(Gseq, function(g) storeG[Q.store[g,] >= Q[g]])
    if(any(lengths(Lstore) < 2)) {
     sw["s.sw"]  <- tmpsw["l.sw"] <-
     sw["l.sw"]  <- FALSE;        warning("Forcing non-storage of scores and loadings due to shortage of retained samples", call.=FALSE)
    }
    eta.store    <- sort_unique(unlist(Lstore))
  } else {
    eta.store    <- tmp.store
  }
  e.store        <- length(eta.store)
  if(sw["s.sw"]) {
    eta          <- if(inf.Q) as.array(sims[[G.ind]][[Q.ind]]$eta)[,,eta.store, drop=FALSE] else sims[[G.ind]][[Q.ind]]$eta[,,eta.store, drop=FALSE]
    if(!sw["l.sw"])               message("Caution advised when examining posterior factor scores: Procrustes rotation has not taken place because loadings weren't stored")
  }

# Loop over g in G to extract other results
  result         <- list(list())
  for(g in Gseq) {
    Qg           <- Q[g]
    Q0g          <- Q0[g]
    Qgs          <- seq_len(Qg)
    sw["l.sw"]   <- tmpsw["l.sw"]
    if(Q0g)      {
      if(all(sw["l.sw"],
             !no.score))          message(paste0("Loadings ", ifelse(G > 1, paste0("for cluster ", g, " not stored as it"), " not stored as model"), " has zero factors"))
      sw["l.sw"] <- FALSE
    }

  # Retrieve (unrotated) loadings
    if(sw["l.sw"]) {
      tmpind     <- ifelse(inf.Q, which.max(Q.store[g,] == Qg), 1)
      if(clust.ind)  {
        lmat     <- .a_drop(lmats[,,g,storeG, drop=FALSE], drop=3)
        l.temp   <- .a_drop(lmat[,,tmpind,    drop=FALSE], drop=3)
      } else  {
        lmat     <- if(inf.Q) as.array(sims[[G.ind]][[Q.ind]]$load)[,,storeG, drop=FALSE] else sims[[G.ind]][[Q.ind]]$load[,,storeG, drop=FALSE]
        l.temp   <- .a_drop(lmat[,,tmpind,    drop=FALSE], drop=3)
      }
    }

  # Loadings matrix / identifiability / error metrics / etc.
    if(sw["l.sw"])          {
      for(p      in storeG) {
        if(p   %in% eta.store)   {
          proc   <- Procrustes(X=if(uni) t(lmat[,,p]) else as.matrix(lmat[,,p]), Xstar=l.temp)
          lmat[,,p]        <- proc$X.new
          if(sw["s.sw"])    {
            rot  <- proc$R
            p2   <- if(inf.Q) eta.store    == p       else p
            if(clust.ind)   {
              zp <- z[p,]  == g
              eta[zp,,p2]  <- eta[zp,,p2] %*% rot
            } else {
              eta[,,p2]    <- eta[,,p2]   %*% rot
            }
          }
        }
      }
    }

  # Retrieve means, uniquenesses & empirical covariance matrix
    if(clust.ind)  {
      if(sw["mu.sw"])       {
        mu       <- if(uni) t(mus[,g,storeG])  else as.matrix(mus[,g,storeG])
      }
      if(sw["psi.sw"])      {
        psi      <- if(uni) t(psis[,g,storeG]) else as.matrix(psis[,g,storeG])
      }
    } else {
      post.mu    <- as.matrix(sims[[G.ind]][[Q.ind]]$post.mu)
      post.psi   <- as.matrix(sims[[G.ind]][[Q.ind]]$post.psi)
      if(sw["mu.sw"])       {
        mu       <- sims[[G.ind]][[Q.ind]]$mu[,storeG,  drop=FALSE]
      }
      if(sw["psi.sw"])      {
        psi      <- sims[[G.ind]][[Q.ind]]$psi[,storeG, drop=FALSE]
      }
    }
    if(sw["mu.sw"]         &&
       is.null(rownames(mu)))   {
      rownames(mu)         <- varnames
    }
    if(sw["psi.sw"]        &&
       is.null(rownames(psi)))  {
      rownames(psi)        <- varnames
    }
    if(sw["l.sw"]          &&
       is.null(rownames(lmat))) {
      rownames(lmat)       <- varnames
    }

  # Compute posterior means and % variation explained
    if(sw["mu.sw"])  {
      post.mu    <- if(clust.ind) rowmeans(mu)  else post.mu
      var.mu     <- if(uni)       Var(mu)       else .row_vars(mu)
      ci.tmp     <- rowQuantiles(mu,  probs=conf.levels)
      ci.mu      <- if(uni)       t(ci.tmp)     else ci.tmp
    }
    if(sw["psi.sw"]) {
      post.psi   <- if(clust.ind) rowmeans(psi) else post.psi
      var.psi    <- if(uni)       Var(psi)      else .row_vars(psi)
      ci.tmp     <- rowQuantiles(psi, probs=conf.levels)
      ci.psi     <- if(uni)       t(ci.tmp)     else ci.tmp
    }
    if(sw["l.sw"])   {
      lmat       <- provideDimnames(lmat[,Qgs,if(inf.Q) Lstore[[g]] else storeG, drop=FALSE], base=list("", paste0("Factor", Qgs), ""), unique=FALSE)
      post.load  <- rowMeans(lmat, dims=2)
      var.load   <- apply(lmat, c(1, 2), Var)
      ci.load    <- apply(lmat, c(1, 2), stats::quantile, conf.levels)
      var.exp    <- sum(post.load * post.load)/n.var
      class(post.load)     <- "loadings"
    } else if(sw["psi.sw"]) {
      if(sizes[g] > 1) {
        dat.gg   <- dat[z.ind[[g]],, drop=FALSE]
        cov.gg   <- stats::cov(dat.gg)
      }
      var.exp    <- ifelse(sizes[g] == 0, 0, max(0, (sum(diag(cov.gg)) - sum(post.psi))/n.var))
    } else {
      var.exp    <- NULL
    }

    results      <- list(if(sw["mu.sw"])  list(means     = mu,
                                               var.mu    = var.mu,
                                               ci.mu     = ci.mu),
                         if(sw["l.sw"])   list(loadings  = lmat,
                                               post.load = post.load,
                                               var.load  = var.load,
                                               ci.load   = ci.load),
                         if(sw["psi.sw"]) list(psis      = psi,
                                               var.psi   = var.psi,
                                               ci.psi    = ci.psi),
                         if(sw.mx)        list(post.mu   = post.mu),
                         if(sw.px)        list(post.psi  = post.psi),
                         if(any(sw["l.sw"],
                                sw.px))   list(var.exp   = var.exp))
    result[[g]]  <- unlist(results, recursive=FALSE)
  }

  if(sw["s.sw"])   {
    Qseq         <- seq_len(max(Q))
    eta          <- provideDimnames(eta[,Qseq,, drop=FALSE], base=list("", paste0("Factor", Qseq), ""), unique=FALSE)
    scores       <- list(eta = eta, post.eta = rowMeans(eta, dims=2), var.eta = apply(eta, c(1, 2), Var),
                         ci.eta = apply(eta, c(1, 2), stats::quantile, conf.levels), last.eta = .a_drop(eta[,,e.store, drop=FALSE], drop=3))
    attr(scores, "Eta.store")  <- e.store
  }
  names(result)  <- gnames
  GQ.res$Criteria              <- c(GQ.res$Criteria, list(sd.AICMs = aicm.sd, sd.BICMs = bicm.sd,
                                                          best.models = t(vapply(GQ.res$Criteria, function(x) { inds <- arrayInd(which.max(x), dim(x));
                                                          c(clusters = rownames(x)[inds[1]], factors = colnames(x)[inds[2]]) }, character(2L)))))
  class(GQ.res)                <- "listof"
  attr(GQ.res, "Clusters")     <- n.grp
  attr(GQ.res, "Criterion")    <- criterion
  attr(GQ.res, "Factors")      <- n.fac
  attr(GQ.res, "Supplied")     <- c(Q=Q.T, G=G.T)
  if(sw["mu.sw"])  {
    mus2         <- lapply(result, "[[", "means")
    var.mu       <- provideDimnames(do.call(cbind, lapply(result, "[[", "var.mu")),   base=list(if(uni) "" else varnames, gnames))
    ci.mu        <- lapply(result, "[[", "ci.mu")
    means        <- list(mus = mus2, var.mu = var.mu, ci.mu = ci.mu)
  } else means             <- NULL
  if(sw["mu.sw"]  || sw.mx) {
    post.mu      <- provideDimnames(do.call(cbind, lapply(result, "[[", "post.mu")),  base=list(if(uni) "" else varnames, gnames))
    means        <- c(means, list(post.mu = post.mu))
    means        <- means[c(1, 4, 2, 3)]
  }
  if(sw["mu.sw"]) {
    last.mu      <- if(clust.ind) .a_drop(mus[,,TN.store, drop=FALSE], drop=3)  else  mu[,TN.store, drop=FALSE]
    colnames(last.mu)      <- gnames
    means        <- c(means, list(last.mu = last.mu))
  }
  if(sw["l.sw"]  <- tmpsw["l.sw"] && !all(Q == 0)) {
    lmats2       <- lapply(result, "[[", "loadings")
    post.load    <- lapply(result, "[[", "post.load")
    var.load     <- lapply(result, "[[", "var.load")
    ci.load      <- lapply(result, "[[", "ci.load")
    last.lmat    <- lapply(lmats2, function(x) { if(!is.null(x)) { x <- .a_drop(x[,,dim(x)[3], drop=FALSE], drop=3); class(x) <- "loadings" }; x })
    loads        <- list(lmats = lmats2, post.load = post.load, var.load = var.load, ci.load = ci.load, last.load = last.lmat)
  }
  if(sw["psi.sw"]) {
    psis2        <- lapply(result, "[[", "psis")
    var.psi      <- provideDimnames(do.call(cbind, lapply(result, "[[", "var.psi")),  base=list(if(uni) "" else varnames, gnames))
    ci.psi       <- lapply(result, "[[", "ci.psi")
    uniquenesses <- list(psis = psis2, var.psi = var.psi, ci.psi = ci.psi)
  } else uniquenesses      <- NULL
  if(sw["psi.sw"] || sw.px) {
    post.psi     <- provideDimnames(do.call(cbind, lapply(result, "[[", "post.psi")), base=list(if(uni) "" else varnames, gnames))
    uniquenesses <- c(uniquenesses, list(post.psi = post.psi))
    uniquenesses <- uniquenesses[c(1, 4, 2, 3)]
  }
  if(sw["psi.sw"]) {
    last.psi     <- if(clust.ind) .a_drop(psis[,,TN.store, drop=FALSE], drop=3) else  psi[,TN.store, drop=FALSE]
    colnames(last.psi)     <- gnames
    uniquenesses <- c(uniquenesses, list(last.psi = last.psi))
  }

# Calculate estimated covariance matrices & compute error metrics
  store.e        <- storeG   %in% eta.store
  e.store        <- sum(store.e)
  Q0E            <- if(inf.Q) Q.store[,store.e, drop=!clust.ind] == 0 else if(clust.ind) matrix(Q == 0, nrow=G, ncol=e.store) else rep(Q == 0, e.store)
  Q0X            <- all(Q0E)
  QX0            <- any(!Q0E)
  if(error.metrics) {
    Eseq         <- seq_len(e.store)
    mse          <- mae        <- medse     <-
    medae        <- rmse       <- nrmse     <- rep(NA, e.store)
  }
  if(clust.ind)     {
    if(all(sw["psi.sw"], sw["mu.sw"], any(sw["l.sw"], Q0X))) {
      a          <- Reduce("+",   lapply(Gseq, function(g) post.pi[g] * (tcrossprod(post.mu[,g]) + (if(Q0[g])   0 else tcrossprod(post.load[[g]])) + (if(uni) post.psi[,g] else diag(post.psi[,g])))))
      b          <- Reduce("+",   lapply(Gseq, function(g) post.pi[g] * post.mu[,g]))
      cov.est    <- a - tcrossprod(b)
      if(error.metrics)         {
       lmat2     <- lmats[,,,store.e, drop=FALSE]
       mu2       <- mus[,,store.e,    drop=FALSE]
       pi2       <- pi.prop[,store.e, drop=FALSE]
       psi2      <- psis[,,store.e,   drop=FALSE]
       for(r  in  Eseq)         {
        Q0Er     <- Q0E[,r]
        a        <- Reduce("+",   lapply(Gseq, function(g) pi2[g,r]   * (tcrossprod(mu2[,g,r])   + (if(Q0Er[g]) 0 else tcrossprod(lmat2[,,g,r]))   + (if(uni) psi2[,g,r]   else diag(psi2[,g,r])))))
        b        <- Reduce("+",   lapply(Gseq, function(g) pi2[g,r]   * mu2[,g,r]))
        sigma    <- a - tcrossprod(b)
        error    <- cov.emp - sigma
        sq.err   <- error   * error
        abs.err  <- abs(error)
        mse[r]   <- mean(sq.err)
        medse[r] <- med(sq.err)
        mae[r]   <- mean(abs.err)
        medae[r] <- med(abs.err)
        rmse[r]  <- sqrt(mse[r])
        nrmse[r] <- rmse[r]/cov.range
       }
      }
    } else if(!sw["mu.sw"])     { warning("Means not stored: can't compute error metrics or estimate posterior mean covariance matrix",                   call.=FALSE)
    } else if(all(QX0, !sw["l.sw"],
              !sw["psi.sw"]))   { warning("Loadings & Uniquenesses not stored: can't compute error metrics or estimate posterior mean covariance matrix", call.=FALSE)
    } else if(all(QX0,
              !sw["l.sw"]))     { warning("Loadings not stored: can't compute error metrics or estimate posterior mean covariance matrix",                call.=FALSE)
    } else                        warning("Uniquenesses not stored: can't compute error metrics or estimate posterior mean covariance matrix",            call.=FALSE)
  } else    {
    if(any(sw["l.sw"], Q0X))    {
      cov.est    <- diag(post.psi[,1, drop=!uni]) + (if(Q0X)    0 else         tcrossprod(post.load[[1]]))
      if(error.metrics && sw["psi.sw"])           {
       psi2      <- psi[,store.e,     drop=FALSE]
       for(r  in  Eseq)         {
        sigma    <- diag(psi2[,r, drop=!uni])     + (if(Q0E[r]) 0 else  if(uni)       crossprod(lmat[,,r])        else tcrossprod(lmat[,,r]))
        error    <- cov.emp - sigma
        sq.err   <- error   * error
        abs.err  <- abs(error)
        mse[r]   <- mean(sq.err)
        medse[r] <- med(sq.err)
        mae[r]   <- mean(abs.err)
        medae[r] <- med(abs.err)
        rmse[r]  <- sqrt(mse[r])
        nrmse[r] <- rmse[r]/cov.range
       }
      } else if(error.metrics)    warning("Uniquenesses not stored: can't compute error metrics", call.=FALSE)
    } else if(all(QX0, !sw["l.sw"],
              !sw["psi.sw"]))   { warning("Loadings & Uniquenesses not stored: can't compute error metrics or estimate posterior mean covariance matrix", call.=FALSE)
    } else if(all(QX0,
              !sw["l.sw"]))       warning("Loadings not stored: can't compute error metrics or estimate posterior mean covariance matrix",                call.=FALSE)
  }
  if(errs        <- all(sw["psi.sw"] || !clust.ind, any(sw["l.sw"], Q0X))) {
   var.exps      <- vapply(lapply(result, "[[", "var.exp"), function(x) ifelse(is.null(x), NA, x), numeric(1L))
   var.exps      <- if(sum(is.na(var.exps)) == G) NULL else var.exps
   Err           <- list(Var.Exps = var.exps, Exp.Var = ifelse(clust.ind, sum(var.exps * post.pi), unname(var.exps)))
   if(sw["mu.sw"] || !clust.ind)       {
     cov.est     <- as.matrix(cov.est)
     dimnames(cov.est)        <- list(varnames, varnames)
     err         <- cov.emp - cov.est
     sq.err      <- err     * err
     abs.err     <- abs(err)
     mse2        <- mean(sq.err)
     rmse2       <- sqrt(mse2)
     post.met    <- c(MSE = mse2, MEDSE = med(sq.err), MAE = mean(abs.err), MEDAE = med(abs.err), RMSE = rmse2, NRMSE = rmse2/cov.range)
     Err         <- c(list(Empirical.Cov = cov.emp, Estimated.Cov = cov.est, Post = post.met), Err)
     if(error.metrics && sw["psi.sw"]) {
       metrics   <- rbind(MSE = mse, MEDSE = medse, MAE = mae, MEDAE = medae, RMSE = rmse, NRMSE = nrmse)
       metricCIs <- rowQuantiles(metrics, probs=conf.levels)
       mean.met  <- stats::setNames(rowmeans(metrics), rownames(metrics))
       last.met  <- c(MSE = mse[e.store], MEDSE = medse[e.store], MAE = mae[e.store], MEDAE = medae[e.store], RMSE = rmse[e.store], NRMSE = nrmse[e.store])
       Err       <- c(list(Avg = mean.met, CIs = metricCIs), Err[1:3], c(list(Last.Cov = sigma, Final = last.met)), Err[4:5])
       errs      <- "All"
     } else errs <- "Post"
   } else   errs <- "Vars"
   class(Err)    <- "listof"
  } else    errs <- "None"
  error.metrics  <- errs       != "None"
  if(error.metrics && anyNA(cov.emp)) {
    Err          <- Err[names(Err) %in% if(errs == "Vars") c("Var.Exps", "Exp.Var") else c("Var.Exps", "Exp.Var", "Estimated.Cov")]
  }

  result         <- c(if(exists("cluster", envir=environment()))          list(Clust      = cluster),
                      if(error.metrics)         list(Error        = Err), list(GQ.results = GQ.res),
                      if(sw["mu.sw"]  || sw.mx) list(Means        =       means),
                      if(sw["l.sw"])            list(Loadings     =       loads),
                      if(sw["s.sw"])            list(Scores       =       scores),
                      if(sw["psi.sw"] || sw.px) list(Uniquenesses = uniquenesses))

  attr(result, "Adapt")        <- attr(sims, "Adapt")
  attr(result, "Alph.step")    <- if(is.element(method, c("IMFA", "IMIFA"))) learn.alpha
  attr(result, "Alpha")        <- if(!learn.alpha) attr(sims, "Alpha")
  attr(result, "Call")         <- call
  attr(result, "Conf.Level")   <- conf.level
  attr(result, "Disc.step")    <- if(is.element(method, c("IMFA", "IMIFA"))) learn.alpha
  attr(result, "Discount")     <- if(is.element(method, c("IMFA", "IMIFA")) && !learn.d) attr(sims, "Discount")
  attr(result, "Errors")       <- ifelse(anyNA(cov.emp), "None", switch(errs, Vars="None", errs))
  attr(result, "Equal.Pi")     <- equal.pro
  attr(result, "G.init")       <- if(inf.G) attr(sims, "G.init")
  attr(result, "Ind.Slice")    <- if(is.element(method, c("IMFA", "IMIFA"))) attr(sims, "Ind.Slice")
  attr(result, "Method")       <- method
  attr(result, "N.Loadstore")  <- if(inf.Q) vapply(Lstore, length, numeric(1L)) else rep(TN.store, G)
  attr(result, "Name")         <- data.name
  attr(result, "Obs")          <- n.obs
  attr(result, "Obsnames")     <- obsnames
  attr(result, "Pitman")       <- attr(sims, "Pitman")
  attr(result, "range.G")      <- attr(sims, "Clusters")
  attr(result, "range.Q")      <- attr(sims, "Factors")
  attr(result, "Sd0.drop")     <- attr(sims, "Sd0.drop")
  attr(result, "Store")        <- tmp.store
  attr(result, "Switch")       <- sw
  attr(result, "TuneZeta")     <- attr(sims, "TuneZeta")
  attr(result, "Uni.Meth")     <- uni.meth
  attr(result, "Varnames")     <- varnames
  attr(result, "Vars")         <- n.var
  attr(result, "Z.sim")        <- z.avgsim
  class(result)                <- "Results_IMIFA"
  cat(print(result))
    return(result)
}
