% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/FullConditionals.R
\name{gumbel_max}
\alias{gumbel_max}
\title{Simulate Cluster Labels from Unnormalised Log-Probabilities using the Gumbel-Max Trick}
\usage{
gumbel_max(probs, log.like = FALSE, slice = FALSE)
}
\arguments{
\item{probs}{An N x G matrix of unnormalised probabilities on the log scale, where N is he number of observations that require labels to be sampled and G is the number of active clusters s.t. sampled labels can take values in \code{1:G}.}

\item{log.like}{A logical indicating whether the normalising constant is to be computed. Defaults to \code{FALSE} but is \code{TRUE} for all methods under \code{\link{mcmc_IMIFA}} where it's necessary for computation of the log-likelihoods required for model choice.}

\item{slice}{A logical indicating whether or not the indicator correction for slice sampling has been applied to \code{probs}. Defaults to \code{FALSE} but is \code{TRUE} for the "\code{IMIFA}" and "\code{IMFA}" methods under \code{\link{mcmc_IMIFA}}. Details of this correction are given in Murphy et. al. (2017).}
}
\value{
Either a N-vector of sampled cluster labels, or if \code{isTRUE(log.like)}, a list with two elements:
\describe{
\item{z}{The numeric vector of \code{N} sampled cluster labels, with the largest label no greater than \code{G}.}
\item{log.like}{The log-likelihood(s), given by the normalising constant(s), computed with the aid of \code{\link[matrixStats]{rowLogSumExps}}.}
}
}
\description{
Samples cluster labels for N observations from G groups efficiently using log-probabilities and the so-called Gumbel-Max trick, without requiring that the log-probabilities need to be normalised; thus redunant computation can be avoided. Computation takes place on the log scale for stability/underflow reasons (to ensure negligible probabilities won't cause computational difficulties); in any case, many functions for calculating multivariate normal densities already output on the log scale. Please note that while the function is available for standalone use that no checks take place, in order to speed up repeated calls to the function inside \code{\link{mcmc_IMIFA}}.
}
\examples{
# Set the dimensions & simulate a matrix of weights
  N         <- 1
  G         <- 3
  weights   <- matrix(c(1, 2, 3), nrow=N, ncol=G)

# Call gumbel_max() repeatedly to obtain samples of the labels, zs
  iters     <- 10000
  zs        <- vapply(seq_len(iters), function(i)
               gumbel_max(probs=log(weights)), numeric(1L))

# Compare answer to the normalised weights
  tabulate(zs, nbins=G)/iters
  normalised <- as.numeric(weights/sum(weights))

# Simulate a matrix of dirichlet weights & the associated vector of N labels
  N       <- 400
  G       <- 8
  sizes   <- seq(from=85, to=15, by=-10)
  weights <- matrix(rDirichlet(N * G, alpha=1, nn=sizes), byrow=TRUE, nrow=N, ncol=G)
  zs      <- gumbel_max(probs=log(weights))
}
\references{
Murphy, K., Gormley, I. C. and Viroli, C. (2017) Infinite Mixtures of Infinite Factor Analysers: Nonparametric Model-Based Clustering via Latent Gaussian Models, \href{https://arxiv.org/abs/1701.07010}{arXiv:1701.07010}.

Yellot, J. I. Jr. (1977) The relationship between Luce's choice axiom, Thurstone's theory of comparative judgment, and the double exponential distribution, \emph{Journal of Mathematical Psychology}, 15: 109-144.
}
\seealso{
\code{\link{mcmc_IMIFA}}, \code{\link[matrixStats]{rowLogSumExps}}
}
\author{
Keefe Murphy
}
