% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/LD50.R
\name{LD50}
\alias{LD50}
\title{Estimate the parameters that best describe LD50}
\usage{
LD50(df = NULL, alive = NULL, dead = NULL, N = NULL,
  doses = NULL, l = 0.05, parameters.initial = NULL,
  fixed.parameters = NULL, SE = NULL, equation = "logistic",
  replicates = 1000, range.CI = 0.95, limit.low.TRD.minimum = 5,
  limit.high.TRD.maximum = 1000, print = TRUE, doses.plot = seq(from
  = 0, to = 1000, by = 0.1))
}
\arguments{
\item{df}{A dataframe with at least two columns named alive, dead or N and doses columns}

\item{alive}{A vector with alive individuals at the end of experiment}

\item{dead}{A vector with dead individuals at the end of experiment}

\item{N}{A vector with total numbers of tested individuals}

\item{doses}{The doses}

\item{l}{The limit to define TRD (see Girondot, 1999)}

\item{parameters.initial}{Initial values for P, S or K search as a vector, ex. c(P=29, S=-0.3)}

\item{fixed.parameters}{Parameters that will not be changed during fit}

\item{SE}{Standard errors for parameters}

\item{equation}{Could be "logistic", "logit", "probit", Hill", "Richards", "Hulin", "flexit" or "Double-Richards"}

\item{replicates}{Number of replicates to estimate confidence intervals}

\item{range.CI}{The range of confidence interval for estimation, default=0.95}

\item{limit.low.TRD.minimum}{Minimum lower limit for TRD}

\item{limit.high.TRD.maximum}{Maximum higher limit for TRD}

\item{print}{Do the results must be printed at screen? TRUE (default) or FALSE}

\item{doses.plot}{Sequences of doses that will be used for plotting. If NULL, does not estimate them}
}
\value{
A list with the LD50, Transitional Range of Doses and their SE
}
\description{
Estimate the parameters that best describe LD50\cr
Logistic and logit models are the same but with different parametrization:\cr
logistic = 1/(1+exp((1/S)*(P-d)))\cr
logit = 1/(1+exp(P+d*S))\cr
See these publications for the description of equations:\cr
Girondot, M. 1999. Statistical description of temperature-dependent sex determination using maximum likelihood. Evolutionary Ecology Research, 1, 479-486.\cr
Godfrey, M.H., Delmas, V., Girondot, M., 2003. Assessment of patterns of temperature-dependent sex determination using maximum likelihood model selection. Ecoscience 10, 265-272.\cr
Hulin, V., Delmas, V., Girondot, M., Godfrey, M.H., Guillon, J.-M., 2009. Temperature-dependent sex determination and global change: are some species at greater risk? Oecologia 160, 493-506.\cr
The flexit equation is not still published :
\deqn{if dose < P then (1 + (2^K1 - 1) *  exp(4 * S1 * (P - x)))^(-1/K1)}{if dose < P then (1 + (2^K1 - 1) *  exp(4 * S1 * (P - x)))^(-1/K1)}
\deqn{if dose > P then 1-((1 + (2^K2 - 1) * exp(4 * S2 * (x - P)))^(-1/K2)}{if dose > P then 1-((1 + (2^K2 - 1) * exp(4 * S2 * (x - P)))^(-1/K2)}
with:\cr
     \deqn{S1 = S/((4/K1)*(2^(-K1))^(1/K1+1)*(2^K1-1))}{S1 = S/((4/K1)*(2^(-K1))^(1/K1+1)*(2^K1-1))}
     \deqn{S2 = S/((4/K2)*(2^(-K2))^(1/K2+1)*(2^K2-1))}{S2 = S/((4/K2)*(2^(-K2))^(1/K2+1)*(2^K2-1))}
}
\details{
LD50 estimates the parameters that best describe LD50
}
\examples{
\dontrun{
library("HelpersMG")
data <- data.frame(Doses=c(80, 120, 150, 150, 180, 200),
Alive=c(10, 12, 8, 6, 2, 1),
Dead=c(0, 1, 5, 6, 9, 15))
LD50_logistic <- LD50(data, equation="logistic")
predict(LD50_logistic, doses=c(140, 170))
plot(LD50_logistic, xlim=c(0, 300), at=seq(from=0, to=300, by=50))
LD50_probit <- LD50(data, equation="probit")
predict(LD50_probit, doses=c(140, 170))
plot(LD50_probit)
LD50_logit <- LD50(data, equation="logit")
predict(LD50_logit, doses=c(140, 170))
plot(LD50_logit)
LD50_hill <- LD50(data, equation="hill")
predict(LD50_hill, doses=c(140, 170))
plot(LD50_hill)
LD50_Richards <- LD50(data, equation="Richards")
predict(LD50_Richards, doses=c(140, 170))
plot(LD50_Richards)
LD50_Hulin <- LD50(data, equation="Hulin")
predict(LD50_Hulin, doses=c(140, 170))
plot(LD50_Hulin)
LD50_DoubleRichards <- LD50(data, equation="Double-Richards")
predict(LD50_DoubleRichards, doses=c(140, 170))
plot(LD50_DoubleRichards)
LD50_flexit <- LD50(data, equation="flexit")
predict(LD50_flexit, doses=c(140, 170))
plot(LD50_flexit)
}
}
\seealso{
Other LD50 functions: \code{\link{LD50_MHmcmc_p}},
  \code{\link{LD50_MHmcmc}}, \code{\link{logLik.LD50}},
  \code{\link{plot.LD50}}, \code{\link{predict.LD50}}
}
\author{
Marc Girondot \email{marc.girondot@u-psud.fr}
}
\concept{LD50 functions}
