\name{hac}
\alias{hac}
\alias{hac.full}
\alias{columns.hac}
\alias{print.hac}
\alias{check.var}
\alias{check.par}

\title{Construction of \env{hac} objects
}
\description{
The function \code{hac} provides one way to construct objects of the class \env{hac}. Such objects are required as input argument for several functions, e.g. \code{\link{plot.hac}} and \code{\link{rHAC}}. The function \code{hac.full} returns also objects of the class \env{hac}. Its advantage compared to \code{hac} is its simple tractibility. The drawback is, that the function cannot be used for building partial nested Archimedean copulae. The supplementary function \code{colums.hac} returns the maximal number of columns, which should be contained in the matrix \code{X}. The function \code{check.var} tests, whether each path of the model ends with a variable or a vector of variables. The function \code{check.par} tests, whether the dependency parameter of an initial node has a lower value than the dependency parameters of successive nodes. The last two functions are called, when \code{hac} or \code{hac.full} are used and they do not return a value, if the input argument (a \env{hac} object) is correctly specified.
}
\usage{
hac(type = HAC_GUMBEL, X, dim = NULL)
hac.full(type = HAC_GUMBEL, y, theta)
columns.hac(n)
check.var(x)
check.par(x)
\method{print}{hac}(x, digits = 2, \dots)
}
\arguments{
  \item{n}{denotes the amount of rows of the matrix \code{X}, which has to contain at least two rows, i.e. \eqn{n \geq 2}. This implies, that a HAC (generated by this matrix) contains nodes on exact \eqn{(n - 1)} nesting levels (hierachies). 
}
  \item{theta}{
a vector containing the dependency parameters of a HAC. The parameters should be ordered from top to down. The length of \code{theta} must be equal to \code{length(y) - 1}.
}
  \item{y}{
a vector, which contains the variables. They are denoted by a \code{\link{formula}}, e.g. \code{~X1}. If a specific node should host a group of variables one can extend the \code{formula} by adding variables with a ``plus'' sign, e.g. \code{~X1 + X2 + X3}. The (groups of) variables should be ordered from top to down and separated by comma.
}
  \item{dim}{
  a positive integer specifying the dimension of Archimedean copulae. It is not called, when a HAC is constructed.
}
  \item{type}{defines the model. It can be chosen between \code{HAC_CLAYTON}, \code{HAC_GUMBEL}, \code{HAC_ROTATED_GUMBEL}, \code{AC_GUMBEL} and \code{AC_CLAYTON}.
}
  \item{x}{
an object of the class \env{hac}.
}
  \item{X}{
a matrix, which should follow a certain structure. It is useful to imagine a binary tree, which ``lies'' on the matrix and has its starting point at the first element of the first row. The first two paths go to the first and the second element of the second row. (If the tree ends at these points, an AC model would be returned.) Starting at these two points the paths go to the first and second element of the third row (originating from the first element of the second row) as well as to the third and fourth element of the third row (originating from the second element of the second row) and so on. A path ends, if the matrix contains a variable. The following elements in the same column are not taken into account and should be set to \eqn{0}. Variables are denoted by a \code{\link{formula}}, e.g. \code{~X1}. If one node should host a group of variables one can extend the \code{formula} by adding variables with a ``plus'' sign, e.g. \code{~X1 + X2 + X3}. However, assure that all terminal points contain a variable. Warnings are returned otherwise. Nodes, (the dependency parameter(s)), are denoted by a numeric value bigger than \eqn{0}. Each element of the matrix unequal to \eqn{0} corresponds to one element of the HAC-tree. All elements, which do not lie on the path should be equal to \eqn{0}.

Generally, consider the construction of a \env{hac} object with \eqn{(n - 1)} nesting levels. Then the matrix needs to contain \eqn{\mathbf{n}} \bold{rows}. The \eqn{i}-th element  of the \eqn{j}-th row needs to be \eqn{0 \quad \forall \, i \, > 2(j - 1)} with exception of the first and the last row. Furthermore the matrix can contain less than \code{columns.hac(n)} columns, if the omitted columns are \eqn{0} vectors.

When Archimedean copulae are constructed the matrix collapses to a scalar (the dependence parameter of the copula).

Be sure that the matrix fulfills the required format, otherwise the \env{hac} object could be misspecified. To get a better idea about the structure of the matrix have a look at the details and the several examples.
}
  \item{digits}{
specifying the digits, (see \code{\link{tree2str}}).
}
  \item{\dots}{
arguments to be passed to methods, (see \code{\link{print}}).
}
}
\details{
The following matrix demonstrates ``theoretically'' how the structure of the matrix should look. The elements of the matrix refer to the element, on which they depend. So the elements in the second row depend on the 1. Node (or element [1, 1]). The further elements in the second row need to be \eqn{0}.
\tabular{cccccccccccc}{
			  \tab \tab \tab \tab \tab \tab \eqn{i}-th column \tab \tab \tab \tab \tab \cr
  			  \tab \tab \tab [ , 1] \tab [ , 2] \tab [ , 3] \tab [ , 4] \tab [ , 5] \tab [ , 6] \tab [ , 7] \tab [ , 8] \tab \eqn{\cdots} \cr 
 			  \tab \tab \tab \tab \tab \tab \tab \tab \tab \tab \tab  \cr
			  \tab [1, ] \tab \tab 1. Node \tab 0 \tab 0 \tab 0 \tab 0 \tab 0 \tab 0 \tab 0 \tab \eqn{\cdots}\cr 
     \eqn{j}-th row \tab [2, ] \tab \tab [1, 1]  \tab [1, 1] \tab 0 \tab 0 \tab 0 \tab 0 \tab 0 \tab 0 \tab \eqn{\cdots}\cr
    			  \tab [3, ] \tab \tab [2, 1]  \tab [2, 1] \tab [2, 2] \tab [2, 2] \tab 0 \tab 0 \tab 0 \tab 0 \tab \eqn{\cdots}\cr
			  \tab [4, ] \tab \tab [3, 1]  \tab [3, 1] \tab [3, 2] \tab [3, 2] \tab [3, 3] \tab [3, 3] \tab [3, 4] \tab [3, 4] \tab \eqn{\cdots}\cr
			  \tab \eqn{\vdots} \tab \tab \eqn{\vdots} \tab \eqn{\vdots} \tab \eqn{\vdots} \tab \eqn{\vdots} \tab \eqn{\vdots} \tab \eqn{\vdots} \tab \eqn{\vdots} \tab \eqn{\vdots}  \tab \eqn{\ddots}
}
However, one should think about the order of the elements, since there are sometimes more than one representation, which result into the same or in a mirrored \env{hac} object (see example 2). 
%In a concrete case the matrices regarding a mirrored \env{hac} object look in the following way:
%\tabular{cccc}{
%2 \tab 0 \tab 0 \tab 0 \cr 
%\eqn{\sim}X1 \tab 4  \tab 0 \tab 0 \cr
%0 \tab 0 \tab \eqn{\sim}X2 \tab \eqn{\sim}X3
%}
%\tabular{cccc}{
%2 \tab 0 \cr 
%4 \tab \eqn{\sim}X1\cr
%\eqn{\sim}X2 \tab \eqn{\sim}X3
%}
}
\value{
An object of the class \env{hac} is returned.
\item{type}{the specified copula type, e.g. \code{HAC_GUMBEL}.}
\item{model}{contains the binary tree.}
}
\author{
Alexander Ristig}
\examples{
# maybe it is helpful to use the function plot.hac to verify the
# results (see example 2)

# Example 1: 4-dim AC
Matrix = 2.5
Object = hac(AC_CLAYTON, X = Matrix, dim = 4)
# class(Object) # [1] "hac"
tree2str(Object) # [1] "(V1.V2.V3.V4)_2.5"
plot(Object, , ylim = c(-0.4, 1.25), l = 1, h = 0.35)


# Example 2: 4-dim HAC
Matrix1 = matrix(c(2, 0, 4, ~X1 + X4, ~X2, ~X3), nrow = 3, byrow = TRUE)
y = c(~X1 + X4, ~X3, ~X2)
theta = c(2, 4)
Matrix3 = matrix(c(2, 0, 0, 0, ~X1 + X4, 4, 0, 0, 0, 0, ~X2, ~X3),
nrow = 3, byrow = TRUE)
# Matrix2
#      [,1]       [,2] [,3]       [,4]      
# [1,] 2          0    0          0         
# [2,] Expression 4    0          0         
# [3,] 0          0    Expression Expression

Object1 = hac(HAC_CLAYTON, Matrix1)
Object2 = hac.full(HAC_CLAYTON, y, theta)
tree2str(Object1) == tree2str(Object2) # [1] TRUE
tree2str(Object1) # [1] "((X2.X3)_{4}.X1.X4)_{2}"
# Object1 and Object 2 contain identical information

Object3 = hac(HAC_CLAYTON, Matrix3)
# the plots show, that the Objects are mirrored
plot.hac(Object1)
plot.hac(Object3)


# Example 3: 6-dim HAC
Matrix = matrix(c(2, 0, 0, 0, 3, 4, 0, 0, ~X1 + X6, ~X2, ~X3, ~X4 + X5),
nrow = 3, byrow = TRUE)
# Matrix
#      [,1]       [,2]       [,3]       [,4]      
# [1,] 2          0          0          0         
# [2,] 3          4          0          0         
# [3,] Expression Expression Expression Expression

Object = hac(HAC_CLAYTON, Matrix)
Object
plot.hac(Object)


# Example 4: 5-dim HAC
Matrix1 = matrix(c(1, 0, 0, 0, 2, 3, 0, 0, ~X1, 
4, ~X4, ~X5, 0, 0, ~X2, ~X3), nrow = 4, byrow = TRUE)
Matrix2 = matrix(c(1, 0, 0, 0, 0, 0, 0, 0, 2, 3, 0, 0, 0, 0, 0, 0, ~X1, 
4, ~X4, ~X5, 0, 0, 0, 0, 0, 0, ~X2, ~X3,  0, 0, 0, 0), nrow = 4,
byrow = TRUE)
# Matrix2
#      [,1]       [,2] [,3]       [,4]       [,5] [,6] [,7] [,8]
# [1,] 1          0    0          0          0    0    0    0   
# [2,] 2          3    0          0          0    0    0    0   
# [3,] Expression 4    Expression Expression 0    0    0    0   
# [4,] 0          0    Expression Expression 0    0    0    0   

Object1 = hac(HAC_CLAYTON, Matrix1)
Object2 = hac(HAC_CLAYTON, Matrix2)
tree2str(Object1) == tree2str(Object2) # [1] TRUE
Object1
plot(Object1)
# the previous result implies, that it is not necessary to include vectors 
# of 0 in the matrix
}