% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot.R
\name{setup_plot}
\alias{setup_plot}
\title{Set up information for a GenericML() plot}
\usage{
setup_plot(
  x,
  type = "GATES",
  learner = "best",
  CLAN_variable = NULL,
  groups = "all"
)
}
\arguments{
\item{x}{An instance of \code{\link{GenericML}}.}

\item{type}{The analysis whose parameters shall be plotted. Either \code{"GATES"}, \code{"BLP"}, or \code{"CLAN"}. Default is \code{"GATES"}.}

\item{learner}{The learner whose results are to be returned. Default is \code{"best"} for the best learner as measured by the \eqn{\Lambda} parameters.}

\item{CLAN_variable}{Name of the CLAN variable to be plotted. Only applicable if \code{type = "CLAN"}.}

\item{groups}{Character vector indicating the per-group parameter estimates that shall be plotted in GATES and CLAN analyses. Default is \code{"all"} for all parameters. If there are \eqn{K} groups, this variable is a subset of \code{c("G1", "G2",...,"GK", "G1-G2", "G1-G2",..., "G1-GK", "GK-G1", "GK-G2",...)}, where Gk denotes the k-th group. Note that this set depends on the choices of the arguments \code{"diff_GATES"} and \code{"diff_CLAN"} of the \code{\link{GenericML}} object.}
}
\value{
An object of class \code{setup_plot}, which is a list with the following elements.
\describe{
\item{\code{data_plot}}{A data frame containing point and interval estimates of the generic target specified in the argument \code{type}.}
\item{\code{data_BLP}}{A data frame containing point and interval estimates of the BLP analysis.}
\item{\code{confidence_level}}{The confidence level reflected by the confidence bounds of the interval estimates. The confidence level is equal to  \code{1 - 2 * significance_level}, which is the adjustment proposed in the paper.}}
}
\description{
Extract the relevant information for visualizing the point and interval estimates of the generic targets of interest. The generic targets of interest can be (subsets of) the parameters of the BLP, GATES, or CLAN analysis.
}
\details{
This function is used internally by \code{\link{plot.GenericML}}. It may also be useful for users who want to produce a similar plot, but who want more control over what information to display or how to display that information.
}
\examples{
if(require("ranger") && require("ggplot2")) {

## generate data
set.seed(1)
n  <- 150                                  # number of observations
p  <- 5                                    # number of covariates
D  <- rbinom(n, 1, 0.5)                    # random treatment assignment
Z  <- matrix(runif(n*p), n, p)             # design matrix
Y0 <- as.numeric(Z \%*\% rexp(p) + rnorm(n)) # potential outcome without treatment
Y1 <- 2 + Y0                               # potential outcome under treatment
Y  <- ifelse(D == 1, Y1, Y0)               # observed outcome

## name the columns of Z
colnames(Z) <- paste0("V", 1:p)

## specify learners
learners <- c("random_forest")

## perform generic ML inference
# small number of splits to keep computation time low
x <- GenericML(Z, D, Y, learners,
               num_splits = 2,
               parallel = FALSE)

## the plot we wish to replicate
plot(x = x, type = "GATES")

## get the data to plot the GATES estimates
data <- setup_plot(x = x, type = "GATES")

## define variables to appease the R CMD check
group <- estimate <- cb_lower <- cb_upper <- NULL

## replicate the plot(x, type = "GATES")
# for simplicity, we skip aligning the colors
ggplot(mapping = aes(x = group,
                     y = estimate), data = data$data_plot) +
  geom_hline(aes(yintercept = 0),
             color = "black", linetype = "dotted") +
  geom_hline(aes(yintercept = data$data_BLP["beta.1", "estimate"],
                 color = "ATE"),
             linetype = "dashed") +
  geom_hline(aes(yintercept = data$data_BLP["beta.1", "cb_lower"],
                 color = paste0(100*data$confidence_level, "\% CB (ATE)")),
             linetype = "dashed")  +
  geom_hline(yintercept = data$data_BLP["beta.1", "cb_upper"],
             linetype = "dashed", color = "red") +
  geom_point(aes(color = paste0("GATES with ",  100*data$confidence_level, "\% CB")), size = 3) +
  geom_errorbar(mapping = aes(ymin = cb_lower,
                              ymax = cb_upper))
}

}
\seealso{
\code{\link[=plot.GenericML]{plot}}
}
