#' @title
#' Estimate the conditional exposure response function using Gaussian process
#'
#' @description
#' Estimates the conditional exposure response function (cerf) using Gaussian
#' Process (gp). The function tune the best match (the lowest covariate balance)
#' for the provided set of hyperparameters.
#'
#' @param data A data.frame of observation data.
#'   - Column 1: Outcome (Y)
#'   - Column 2: Exposure or treatment (w)
#'   - Column 3~m: Confounders (C)
#' @param w A vector of exposure level to compute CERF.
#' @param GPS_m A data.frame of GPS vectors.
#'   - Column 1: GPS
#'   - Column 2: Prediction of exposure for covariate of each data
#'   sample (e_gps_pred).
#'   - Column 3: Standard deviation of
#'   e_gps (e_gps_std)
#' @param params A list of parameters that is required to run the process.
#' These parameters include:
#'   - alpha: A scaling factor for the GPS value.
#'   - beta: A scaling factor for the exposure value.
#'   - g_sigma: A scaling factor for kernel function (gamma/sigma).
#'   - tune_app: A tuning approach. Available approaches:
#'     - all: try all combinations of hyperparameters.
#' alpha, beta, and g_sigma can be a vector of parameters.
#' @param nthread An integer value that represents the number of threads to be
#' used by internal packages.
#' @param kernel_fn A kernel function. A default value is a Gaussian Kernel.
#'
#' @return
#' A cerf_gp object that includes the following values:
#'  - w, the vector of exposure levels.
#'  - pst_mean, Computed mean for the w vector.
#'  - pst_sd, Computed credible interval for the w vector.
#'
#' @export
#'
#' @examples
#' \donttest{
#' set.seed(129)
#' data <- generate_synthetic_data(sample_size = 100, gps_spec = 3)
#'
#'
#' # Estimate GPS function
#' GPS_m <- train_gps(cov_mt = data[,-(1:2)],
#'                    w_all = data$treat,
#'                    sl_lib = c("SL.xgboost"),
#'                    dnorm_log = FALSE)
#'
#' # exposure values
#' w_all <- seq(0,10,1)
#'
#'
#' cerf_gp_obj <- estimate_cerf_gp(data,
#'                                 w_all,
#'                                 GPS_m,
#'                                 params = list(alpha = c(0.1),
#'                                               beta=0.2,
#'                                               g_sigma = 1,
#'                                               tune_app = "all"),
#'                                 nthread = 1)
#' }
#'
estimate_cerf_gp <- function(data, w, GPS_m, params, nthread = 1,
                             kernel_fn = function(x) exp(-x ^ 2)){


  # Log system info
  log_system_info()

  # function call
  fcall <- match.call()

  # Double-check input parameters ----------------------------------------------
  if (!is.data.frame(data)) {
    stop(paste0("Data should be a data.frame. ",
                "Current format: ", class(data)[1]))
  }

  if (!is.data.frame(GPS_m)) {
    stop(paste0("The GPS_m should be a data.frame. ",
                "Current format: ", class(GPS_m)[1]))
  }

  check_params <- function(my_param, params) {
    for (item in my_param) {
      if (!is.element(c(item), names(params))) {
        stop(paste0("The required parameter, ", item,", is not provided. ",
                    "Current parameters: ", paste(unlist(names(params)),
                                                  collapse = ", ")))
      }
    }
  }

  check_params(c("alpha", "beta", "g_sigma", "tune_app"), params)

  # TODO: Check values of parameters, too.

  # Expand the grid of parameters (alpha, beta, g_sigma) -----------------------
  tune_params <-  expand.grid(getElement(params, "alpha"),
                              getElement(params, "beta"),
                              getElement(params, "g_sigma"))

  if (nrow(tune_params) == 0) {
    stop(paste("Something went wrong with tuning parameters. ",
               "The expanded grid has not been generated."))
  }

  # Choose subset of tuning parameters based on tuning approach ----------------
  if (getElement(params, "tune_app") == "all") {
    tune_params_subset <- tune_params
  } else if (getElement(params, "tune_app") == "at_random") {
    stop("This approach is not implemented.")
  } else {
    stop(paste("The provided tune_app approach, ",
               getElement(params, "tune_app"),
               " is not supported."))
  }

  # Compute m, "confidence interval", and covariate balance for provided
  # hyperparameters. -----------------------------------------------------------
  if(nthread > 1 && nrow(tune_params_subset) > 1) {
    lfp <- get_options("logger_file_path")

    # make a cluster
    cl <- parallel::makeCluster(nthread, type="PSOCK",
                                outfile = lfp)

    # export variables and functions to cluster cores
    parallel::clusterExport(cl = cl,
                            varlist = c("w", "data", "GPS_m",
                                        "tune_params_subset",
                                        "kernel_fn",
                                        "compute_m_sigma"),
                            envir=environment())

    tune_res <- parallel::parApply(cl, tune_params_subset, 1,
                                   function(x) {
                                     compute_m_sigma(hyperparam = x,
                                                     data = data,
                                                     w = w,
                                                     GPS_m = GPS_m,
                                                     tuning = TRUE,
                                                     kernel_fn = kernel_fn)
                                   })

    parallel::stopCluster(cl)
  } else if (nrow(tune_params_subset) > 1) {
    tune_res <- apply(tune_params_subset, 1, function(x) {
      compute_m_sigma(hyperparam = x,
                      data = data,
                      w = w,
                      GPS_m = GPS_m,
                      tuning = TRUE,
                      kernel_fn = kernel_fn)
      })
  }

  # Select the combination of hyperparameters that provides the lowest
  # covariate balance ----------------------------------------------------------
  if (nrow(tune_params_subset) > 1) {
    opt_idx <- order(sapply(tune_res, function(x) {mean(x$cb)}))[1]
  } else {
    opt_idx <- 1
  }
  opt_param <- tune_params_subset[opt_idx,]
  gp_cerf_final <- compute_m_sigma(hyperparam = opt_param,
                                   data = data,
                                   w = w,
                                   GPS_m = GPS_m,
                                   tuning = FALSE,
                                   kernel_fn = kernel_fn)

  # Build gp_cerf S3 object
  result <- list()
  class(result) <- "cerf_gp"

  result$pst_mean <- gp_cerf_final$est
  result$pst_sd <- gp_cerf_final$pst
  result$w <- w
  result$cb <- gp_cerf_final$cb
  result$params <- opt_param
  result$fcall <- fcall

  # Add best match to the gp_cerf object

  # Add other useful info form the processing to the gp_cerf object

  # return gp_cerf S3 object
  invisible(result)
}
