% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/graphfanova.r
\name{graph.fanova}
\alias{graph.fanova}
\title{One-way graphical functional ANOVA}
\usage{
graph.fanova(
  nsim,
  curve_set,
  groups,
  variances = "equal",
  contrasts = FALSE,
  n.aver = 1L,
  mirror = FALSE,
  savefuns = FALSE,
  test.equality = c("mean", "var", "cov"),
  cov.lag = 1,
  ...
)
}
\arguments{
\item{nsim}{The number of random permutations.}

\item{curve_set}{The original data (an array of functions) provided as a
\code{\link{curve_set}} object or a \code{fdata} object of \pkg{fda.usc}.
The curve set should include the argument values for the functions in the component \code{r}, and
the observed functions in the component \code{obs}.}

\item{groups}{The original groups (a factor vector representing the assignment to groups).}

\item{variances}{Either "equal" or "unequal". If "unequal", then correction for unequal variances
as explained in details will be done. Only relevant for the case \code{test.equality = "means"} (default).}

\item{contrasts}{Logical. FALSE and TRUE specify the two test functions as described in
description part of this help file.}

\item{n.aver}{If variances = "unequal", there is a possibility to use variances smoothed
by appying moving average to the estimated sample variances. n.aver determines
how many values on each side do contribute (incl. value itself).}

\item{mirror}{The complement of the argument circular of \code{\link[stats]{filter}}. Another parameter
for the moving average to estimate sample variances (see \code{n.aver}).}

\item{savefuns}{Logical. If TRUE, then the functions from permutations are saved to the attribute
simfuns.}

\item{test.equality}{A character with possible values \code{mean} (default), \code{var} and
\code{cov}. If \code{mean}, the functional ANOVA is performed to compare the means in the groups.
If \code{var}, then the equality of variances of the curves in the groups is tested by performing
the graphical functional ANOVA test on the functions
\deqn{Z_{ij}(r) = T_{ij}(r) - \bar{T}_j(r).}{Z_{ij}(r) = T_{ij}(r) - \bar{T}_j(r).}
If \code{cov}, then the equality of lag \code{cov.lag} covariance is tested by performing the fANOVA with
\deqn{W_{ij}(r) = \sqrt{|V_{ij}(r)|\cdot sign(V_{ij}(r))},}{|V_{ij}(r)| sign(V_{ij}(r)),}
where
\deqn{V_{ij}(r) = (T_{ij}(r) - \bar{T}_j(r))((T_{ij}(r+s) - \bar{T}_j(r+s))).}{V_{ij}(r) = (T_{ij}(r) - \bar{T}_j(r))((T_{ij}(r+s) - \bar{T}_j(r+s))).}
See Mrkvicka et al. (2020) for more details.}

\item{cov.lag}{The lag of the covariance for testing the equality of covariances,
see \code{test.equality}.}

\item{...}{Additional parameters to be passed to \code{\link{global_envelope_test}}.
For example, the type of multiple testing control, FWER or FDR must be set by \code{typeone}.
And, if \code{typeone = "fwer"}, the type of the global envelope can be chosen by
specifying the argument \code{type}. See \code{\link{global_envelope_test}}
for the defaults and available options.
(The test here uses \code{alternative="two.sided"} and \code{nstep=1} (when relevant),
but all the other specifications are to be given in \code{...}.)}
}
\description{
One-way ANOVA tests for functional data with graphical interpretation
}
\details{
This function can be used to perform one-way graphical functional ANOVA tests described
in Mrkvička et al. (2020). Both 1d and 2d functions are allowed in curve sets.

The tests assume that there are \eqn{J}{J} groups which contain
\eqn{n_1,\dots,n_J}{n1, ..., nJ} functions
\eqn{T_{ij}, i=\dots,J, j=1,\dots,n_j}{T_{ij}, i=1,...,J, j=1,...,nj}.
The functions should be given in the argument \code{curve_set},
and the groups in the argument \code{groups}.
The tests assume that \eqn{T_{ij}, i=1,...,n_j}{T_{ij}, i=1,...,n_j} is an iid sample from
a stochastic process with mean function \eqn{\mu_j}{\mu_j} and
covariance function \eqn{\gamma_j(s,t)}{\gamma_j(s,t)} for s,t in R and j = 1,..., J.

To test the hypothesis
\deqn{H_0 : \mu_1(r) \equiv \mu_2(r)\equiv \dots \equiv \mu_J(r),}{H0: \mu_1(r) = \mu_2(r) = ... = \mu_J(r),}
you can use the test function
\deqn{\mathbf{T} = (\overline{T}_1({\bf r}), \overline{T}_2({\bf r}), \dots , \overline{T}_J({\bf r}))}{T = (\bar{T}_1(r), \bar{T}_2(r), ..., \bar{T}_J(r))}
where \eqn{\overline{T}_i({\bf r})}{\bar{T}_i(r)} is a vector of mean values of functions in the group j.
This test function is used when \code{contrasts = FALSE} (default).

The hypothesis can equivalently be written as
\deqn{H_0 : \mu_i(r) - \mu_j(r) = 0, i=1,\dots,J-1, j=1,\dots,J.}{H0: \mu_i(r) - \mu_j(r) = 0, i=1,...,J-1, j=i,...,J.}
and, alternatively, one can use the test function (vector)
taken to consist of the differences of the group averages,
\deqn{\mathbf{T'} = (\overline{T}_1({\bf r})-\overline{T}_2({\bf r}),
\overline{T}_1({\bf r})-\overline{T}_3({\bf r}), \dots , \overline{T}_{J-1}({\bf r})-\overline{T}_J({\bf r})).}{T' = (\bar{T}_1(r)-\bar{T}_2(r), \bar{T}_1(r)-\bar{T}_3(r), ..., \bar{T}_{J-1}(r)-\bar{T}_J(r)).}
The choice is available with the option \code{contrasts = TRUE}.
This test corresponds to the post-hoc test done usually after an ANOVA test is significant, but
it can be directed tested by means of the combined rank test (Mrkvička et al., 2017) with this test vector.

The test as such assumes that the variances are equal across the groups of functions. To deal with
unequal variances, the differences are rescaled as the first step as follows
\deqn{S_{ij}(r) = \frac{T_{ij}(r) - \overline{T}(r))}{\sqrt{Var(T_j(r))}} \sqrt{Var(T(r))} + \overline{T}(r))}{S_{ij}(r) = ( T_{ij}(r) - \bar{T}(r) ) / Sd(T_j(r)) * Sd(T(r)) + \bar{T}(r))}
where \eqn{\overline{T}({\bf r})}{\bar{T}(r)} is the overall sample mean and
\eqn{\sqrt{Var(T(r))}}{Sd(T(r))} is the overall sample standard deviation.
This scaling of the test functions can be obtained by giving the argument \code{variances = "unequal"}.
}
\examples{
#-- NOx levels example (see for details Myllymaki and Mrkvicka, 2020)
if(require("fda.usc", quietly=TRUE)) {
  # Prepare data
  data("poblenou")
  fest <- poblenou$df$day.festive; week <- as.integer(poblenou$df$day.week)
  Type <- vector(length=length(fest))
  Type[fest == 1 | week >= 6] <- "Free"
  Type[fest == 0 & week \%in\% 1:4] <- "MonThu"
  Type[fest == 0 & week == 5] <- "Fri"
  Type <- factor(Type, levels = c("MonThu", "Fri", "Free"))

  # (log) Data as a curve_set
  cset <- curve_set(r = 0:23,
     obs = t(log(poblenou[['nox']][['data']])))
  # Graphical functional ANOVA
  \donttest{nsim <- 2999}
  \dontshow{
  nsim <- 19
  # Decrease the data to reduce time
  cset$funcs <- cset$funcs[, 20:40]
  Type <- Type[20:40]
  }
  res.c <- graph.fanova(nsim = nsim, curve_set = cset,
                        groups = Type, variances = "unequal",
                        contrasts = TRUE)
  plot(res.c) + ggplot2::labs(x = "Hour", y = "Diff.")
}

#-- Centred government expenditure centralization ratios example
# This is an example analysis of the centred GEC in Mrkvicka et al.
data("cgec")

# Number of simulations
\donttest{nsim <- 2499 # increase to reduce Monte Carlo error}
\dontshow{nsim <- 19}

# Test for unequal lag 1 covariances
res.cov1 <- graph.fanova(nsim = nsim, curve_set = cgec$cgec,
                         groups = cgec$group,
                         test.equality = "cov", cov.lag = 1)
plot(res.cov1)
# Add labels
plot(res.cov1, labels = paste("Group ", 1:3, sep="")) +
  ggplot2::xlab(substitute(paste(italic(i), " (", j, ")", sep=""), list(i="r", j="Year")))
# Test for equality of variances among groups
res.var <- graph.fanova(nsim = nsim, curve_set = cgec$cgec,
                        groups = cgec$group,
                        test.equality = "var")
plot(res.var)

# Test for equality of means assuming equality of variances
# a) using 'means'
res <- graph.fanova(nsim = nsim, curve_set = cgec$cgec,
                    groups = cgec$group,
                    variances = "equal", contrasts = FALSE)
plot(res)
# b) using 'contrasts'
res2 <- graph.fanova(nsim = nsim, curve_set = cgec$cgec,
                     groups = cgec$group,
                     variances = "equal", contrasts = TRUE)
plot(res2)

# Image set examples
data("imageset3")
\dontshow{
# Cut the data to reduce time
imageset3$image_set$r <- imageset3$image_set$r[c(1:3, 52:55, 103),]
imageset3$image_set$funcs <- imageset3$image_set$funcs[c(1:3, 52:55, 103), ]
}
res <- graph.fanova(nsim = 19, # Increase nsim for serious analysis!
                    curve_set = imageset3$image_set,
                    groups = imageset3$Group)
plot(res, what = c("obs", "lo.sign", "hi.sign"), sign.type = "col")

}
\references{
Mrkvička, T., Myllymäki, M., Jilek, M. and Hahn, U. (2020) A one-way ANOVA test for functional data with graphical interpretation. Kybernetika 56 (3), 432-458. doi: 10.14736/kyb-2020-3-0432

Mrkvička, T., Myllymäki, M., and Hahn, U. (2017). Multiple Monte Carlo testing, with applications in spatial point processes. Statistics and Computing 27 (5): 1239-1255. doi:10.1007/s11222-016-9683-9

Myllymäki, M and Mrkvička, T. (2020). GET: Global envelopes in R. arXiv:1911.06583 [stat.ME]. https://doi.org/10.48550/arXiv.1911.06583
}
\seealso{
\code{\link{frank.fanova}}
}
