#' @export
#' @title The Identical Steady-state Equilibrium: Four Models Illustrating Dividend
#' @aliases gemIntertemporal_Dividend
#' @description  Four models are presented to illustrate dividend, which have the same steady-state equilibrium.
#'
#' These models are as follows:
#' (1) a real timeline model with head-tail adjustment;
#' (2) a financial timeline model with dividend and head-tail adjustment;
#' (3) a financial sequential model with dividend;
#' (4) a financial time-circle model with dividend.
#' @param ... arguments to be passed to the function sdm2.
#' @examples
#' \donttest{
#' #### (1) a real timeline model with head-tail adjustment.
#' eis <- 0.8 # the elasticity of intertemporal substitution
#' rho.beta <- 0.8 # the subjective discount factor
#' gr <- 0.03 # the growth rate
#' np <- 5 # the number of internal periods
#' S0Exg <- matrix(NA, 2 * np - 1, np)
#' S0Exg[(np + 1):(2 * np - 1), np] <- 100 * (1 + gr)^(0:(np - 2))
#' S0Exg[1, np] <- 140 # the product supply in the first period, which will be adjusted.
#'
#' B <- matrix(0, 2 * np - 1, np)
#' B[2:np, 1:(np - 1)] <- diag(np - 1)
#'
#' dstl.firm <- list()
#' for (k in 1:(np - 1)) {
#'   dstl.firm[[k]] <- node_new(
#'     "prod",
#'     type = "CD",
#'     alpha = 2, beta = c(0.5, 0.5),
#'     paste0("prod", k), paste0("lab", k)
#'   )
#' }
#'
#' dst.consumer <- node_new(
#'   "util",
#'   type = "CES", es = eis,
#'   alpha = 1, beta = prop.table(rho.beta^(1:np)),
#'   paste0("prod", 1:np)
#' )
#'
#' ge <- sdm2(
#'   A = c(dstl.firm, dst.consumer),
#'   B = B,
#'   S0Exg = S0Exg,
#'   names.commodity = c(paste0("prod", 1:np), paste0("lab", 1:(np - 1))),
#'   names.agent = c(paste0("firm", 1:(np - 1)), "consumer"),
#'   numeraire = "prod1",
#'   policy = makePolicyHeadTailAdjustment(gr = gr, np = np)
#' )
#'
#' ((1 + gr))^(1 / eis) / rho.beta - 1 # the real interest rate.
#' ge$p[1:(np - 1)] / ge$p[2:np] - 1
#' ge$z
#'
#' ## (2) a financial timeline model with dividend and head-tail adjustment.
#' yield <- 0.2593 # (1 + gr)^(1 / eis - 1) / rho.beta - 1
#'
#' S0Exg <- matrix(NA, 2 * np, np)
#' S0Exg[(np + 1):(2 * np - 1), np] <- 100 * (1 + gr)^(0:(np - 2))
#' S0Exg[2 * np, np] <- 100
#' S0Exg[1, np] <- 140 # the product supply in the first period, which will be adjusted.
#'
#' B <- matrix(0, 2 * np, np)
#' B[2:np, 1:(np - 1)] <- diag(np - 1)
#'
#' dstl.firm <- list()
#' for (k in 1:(np - 1)) {
#'   dstl.firm[[k]] <- node_new(
#'     "prod",
#'     type = "FIN", rate = c(1, yield),
#'     "cc1", "claim"
#'   )
#'   node_set(dstl.firm[[k]], "cc1",
#'            type = "CD", alpha = 2, beta = c(0.5, 0.5),
#'            paste0("prod", k), paste0("lab", k)
#'   )
#' }
#'
#' dst.consumer <- node_new(
#'   "util",
#'   type = "CES", es = 1,
#'   alpha = 1, beta = prop.table(rep(1, np)), # prop.table(rho.beta^(1:np)),
#'   paste0("prod", 1:np)
#' )
#'
#' ge <- sdm2(
#'   A = c(dstl.firm, dst.consumer),
#'   B = B,
#'   S0Exg = S0Exg,
#'   names.commodity = c(paste0("prod", 1:np), paste0("lab", 1:(np - 1)), "claim"),
#'   names.agent = c(paste0("firm", 1:(np - 1)), "consumer"),
#'   numeraire = "prod1",
#'   policy = makePolicyHeadTailAdjustment(gr = gr, np = np)
#' )
#'
#' #### (3) a financial sequential model with dividend.
#' gr <- 0.03 # the growth rate.
#' yield <- 0.2593 # (1 + gr)^(1 / eis - 1) / rho.beta - 1
#'
#' dst.firm <- node_new("output",
#'   type = "FIN",
#'   rate = c(1, dividend.rate = yield),
#'   "cc1", "equity.share"
#' )
#' node_set(dst.firm, "cc1",
#'   type = "CD",
#'   alpha = 2, beta = c(0.5, 0.5),
#'   "prod", "lab"
#' )
#'
#' dst.laborer <- node_new("util",
#'   type = "Leontief", a = 1,
#'   "prod"
#' )
#'
#' dst.shareholder <- Clone(dst.laborer)
#'
#' ge <- sdm2(
#'   A = list(dst.firm, dst.laborer, dst.shareholder),
#'   B = diag(c(1, 0, 0)),
#'   S0Exg = {
#'     S0Exg <- matrix(NA, 3, 3)
#'     S0Exg[2, 2] <- S0Exg[3, 3] <- 100
#'     S0Exg
#'   },
#'   names.commodity = c("prod", "lab", "equity.share"),
#'   names.agent = c("firm", "laborer", "shareholder"),
#'   numeraire = "prod",
#'   GRExg = gr
#' )
#'
#' ge$z
#'
#' ge <- sdm2(
#'   A = list(dst.firm, dst.laborer, dst.shareholder),
#'   B = diag(c(1, 0, 0)),
#'   S0Exg = {
#'     S0Exg <- matrix(NA, 3, 3)
#'     S0Exg[2, 2] <- S0Exg[3, 3] <- 100
#'     S0Exg
#'   },
#'   names.commodity = c("prod", "lab", "equity.share"),
#'   names.agent = c("firm", "laborer", "shareholder"),
#'   numeraire = "prod",
#'   GRExg = gr,
#'   maxIteration = 1,
#'   numberOfPeriods = 20,
#'   z0 = c(154.20, 0, 0),
#'   policy = policyMarketClearingPrice,
#'   ts = TRUE
#' )
#'
#' ge$ts.z[, 1]
#' growth_rate(ge$ts.z[, 1])
#'
#' #### (4) a financial time-circle model with dividend.
#' np <- 5
#' gr <- 0.03
#' yield <- 0.2593
#'
#' zeta <- (1 + gr)^np # the ratio of repayments to loans
#' S <- matrix(NA, 2 * np + 1, np + 1)
#' S[(np + 1):(2 * np + 1), np + 1] <- c(100 * (1 + gr)^(0:(np - 1)), 100)
#'
#' B <- matrix(0, 2 * np + 1, np + 1)
#' B[1:np, 1:np] <- diag(np)[, c(2:np, 1)]
#' B[1, np] <- 1 / zeta
#'
#' dstl.firm <- list()
#' for (k in 1:np) {
#'   dstl.firm[[k]] <- node_new("output",
#'     type = "FIN", rate = c(1, yield),
#'     "cc1", "claim"
#'   )
#'   node_set(dstl.firm[[k]], "cc1",
#'     type = "CD", alpha = 2,
#'     beta = c(0.5, 0.5),
#'     paste0("lab", k), paste0("prod", k)
#'   )
#' }
#'
#' dst.consumer <- node_new(
#'   "util",
#'   type = "CES", es = 1,
#'   alpha = 1, beta = prop.table(rep(1, np)),
#'   paste0("prod", 1:np)
#' )
#'
#' ge <- sdm2(
#'   A = c(dstl.firm, dst.consumer),
#'   B = B,
#'   S0Exg = S,
#'   names.commodity = c(paste0("prod", 1:np), paste0("lab", 1:np), "claim"),
#'   names.agent = c(paste0("firm", 1:np), "consumer"),
#'   numeraire = "prod1",
#'   ts = TRUE
#' )
#'
#' ge$z
#' }

gemIntertemporal_Dividend <- function(...) sdm2(...)
