% Generated by roxygen2 (4.1.0): do not edit by hand
% Please edit documentation in R/discrete_entropy.R
\name{discrete_entropy}
\alias{discrete_entropy}
\title{Shannon entropy for discrete pmf}
\usage{
discrete_entropy(probs, base = 2, method = c("MLE"), threshold = 0,
  prior.probs = rep(1/length(probs), length = length(probs)),
  prior.weight = 0)
}
\arguments{
\item{probs}{numeric; probabilities (empirical frequencies). Must be non-negative and add up to \eqn{1}.}

\item{base}{logarithm base; entropy is measured in ``nats'' for
\code{base = exp(1)}; in ``bits'' if \code{base = 2} (default).}

\item{method}{string; method to estimate entropy; see Details below.}

\item{threshold}{numeric; frequencies below \code{threshold} are set to \eqn{0};
default \code{threshold = 0}, i.e., no thresholding.
If \code{prior.weight > 0} then thresholding will be done \emph{before} smoothing.}

\item{prior.probs}{optional; only used if \code{prior.weight > 0}.
Add a prior probability distribution to \code{probs}. By default it uses a
uniform distribution putting equal probability on each outcome.}

\item{prior.weight}{numeric; how much weight does the prior distribution get in a mixture
model between data and prior distribution? Must be between \code{0} and \code{1}.
Default: \code{0} (no prior).}
}
\value{
numeric; non-negative real value.
}
\description{
Computes the Shannon entropy \eqn{\mathcal{H}(p) = -\sum_{i=1}^{n} p_i \log p_i}
of a discrete RV \eqn{X} taking
values in \eqn{\lbrace x_1, \ldots, x_n \rbrace} with probability
mass function (pmf) \eqn{P(X = x_i) = p_i} with
\eqn{p_i \geq 0} for all \eqn{i} and \eqn{\sum_{i=1}^{n} p_i = 1}.
}
\details{
\code{discrete_entropy} uses a plug-in estimator (\code{method = "MLE"}):
\deqn{
\widehat{\mathcal{H}}(p) = - \sum_{i=1}^{n} \widehat{p}_i \log \widehat{p}_i.
}

If \code{prior.weight > 0}, then it mixes the observed proportions \eqn{\widehat{p}_i}
 with a prior distribution
\deqn{
\widehat{p}_i \leftarrow (1-\lambda) \cdot \widehat{p_i} + \lambda \cdot prior_i, \quad i=1, \ldots, n,
}
where \eqn{\lambda \in [0, 1]} is the \code{prior.weight} parameter.  By default
the prior is a uniform distribution, i.e., \eqn{prior_i = \frac{1}{n}} for all i.

Note that this plugin estimator is biased. See References for an overview of alternative
methods.
}
\examples{
probs.tmp <- rexp(5)
probs.tmp <- sort(probs.tmp / sum(probs.tmp))

unif.distr <- rep(1/length(probs.tmp), length(probs.tmp))

matplot(cbind(probs.tmp, unif.distr), pch = 19,
        ylab = "P(X = k)", xlab = "k")
matlines(cbind(probs.tmp, unif.distr))
legend("topleft", c("non-uniform", "uniform"), pch = 19,
       lty = 1:2, col = 1:2, box.lty = 0)

discrete_entropy(probs.tmp)
# uniform has largest entropy among all bounded discrete pmfs
# (here = log(5))
discrete_entropy(unif.distr)
# no uncertainty if one element occurs with probability 1
discrete_entropy(c(1, 0, 0))
}
\references{
Archer E., Park I. M., Pillow J.W. (2014). \dQuote{Bayesian Entropy Estimation for
Countable Discrete Distributions}. Journal of Machine Learning Research (JMLR) 15,
2833-2868. Available at \url{jmlr.org/papers/v15/archer14a.html}.
}
\seealso{
\code{\link{continuous_entropy}}
}
\keyword{math}
\keyword{univar}

