% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CVST_linkfunction.R
\name{fastkrr}
\alias{fastkrr}
\title{Fit kernel ridge regression using exact or approximate methods}
\usage{
fastkrr(
  x,
  y,
  kernel = "gaussian",
  opt = "exact",
  m = NULL,
  eps = 1e-06,
  rho = 1,
  lambda = NULL,
  fastcv = FALSE,
  n_threads = 4,
  verbose = TRUE
)
}
\arguments{
\item{x}{Design matrix \eqn{X \in \mathbb{R}^{n\times d}}.}

\item{y}{Response variable \eqn{y  \in \mathbb{R}^{n}}.}

\item{kernel}{Kernel type either "gaussian"or "laplace".}

\item{opt}{Method for constructing or approximating :
 \describe{
  \item{\code{"exact"}}{Construct the full kernel matrix
  \eqn{K \in \mathbb{R}^{n\times n}} using design martix \eqn{X}.}
  \item{\code{"nystrom"}}{Construct a low-rank approximation of
      the kernel matrix \eqn{K \in \mathbb{R}^{n \times n}}
      using the Nyström approximation.}
  \item{\code{"pivoted"}}{Construct a low-rank approximation of
      the kernel matrix \eqn{K \in \mathbb{R}^{n \times n}}
      using Pivoted Cholesky decomposition.}
\item{\code{"rff"}}{Use Random Fourier Features to construct a feature map
  \eqn{Z \in \mathbb{R}^{n \times m}} (with \eqn{m} random features) so that
  \eqn{K \approx Z Z^\top}. Here, \eqn{m} is the number of features.}
 }}

\item{m}{Approximation rank(number of random features) used for the low-rank kernel approximation.
If not provided by the user, it defaults to
\deqn{\lceil n \cdot \frac{\log(d + 5)}{10} \rceil,}
where \eqn{n = nrow(X)} and \eqn{d = ncol(X)}.}

\item{eps}{Tolerance parameter used only in \code{"pivoted"}
for stopping criterion of the Pivoted Cholesky decomposition.}

\item{rho}{Scaling parameter of the kernel(\eqn{\rho}),  specified by the user.
  Defaults to \code{1}.
\deqn{\text{Gaussian kernel : } \mathcal{K}(x, x') = \exp(-\rho \| x - x'\|^2_2)}
\deqn{\text{Laplace kernel : } \mathcal{K}(x, x') = \exp(-\rho \| x - x'\|_1)}}

\item{lambda}{Regularization parameter. If \code{NULL}, the penalty parameter
is chosen automatically via \pkg{CVST} package. If not provided, the argument is set to a
kernel-specific grid of 100 values: \eqn{[10^{-10}, 10^{-3}]} for Gaussian, \eqn{[10^{-5}, 10^{-2}]} for Laplace.}

\item{fastcv}{If \code{TRUE}, accelerated cross-validation is
performed via sequential testing (early stopping) as implemented in the \pkg{CVST} package.
The default is \code{FALSE}.}

\item{n_threads}{Number of parallel threads.
The default is 4. If the system does not support 4 threads,
it automatically falls back to 1 thread.
Parallelization (implemented in C++) is one of the main advantages
of this package and is applied only for \code{opt = "nystrom"} or \code{opt = "rff"}, and for the
Laplace kernel (\code{kernel = "laplace"}).}

\item{verbose}{If TRUE, detailed progress and cross-validation
results are printed to the console. If FALSE, suppresses
intermediate output and only returns the final result.}
}
\value{
An object of class \code{fastkrr}. The returned object is a list with components:

\describe{
  \item{\code{coefficients}}{Estimated coefficient vector \eqn{\mathbb{R}^{n}}. Accessible via \code{model$coefficients}.}
  \item{\code{fitted.values}}{Fitted values \eqn{\mathbb{R}^{n}}. Accessible via \code{model$fitted.values}.}
  \item{\code{opt}}{Kernel approximation option. One of \code{"exact"}, \code{"pivoted"}, \code{"nystrom"}, \code{"rff"}.}
  \item{\code{kernel}}{Kernel used (\code{"gaussian"} or \code{"laplace"}).}
  \item{\code{x}}{Input design matrix.}
  \item{\code{y}}{Response vector.}
  \item{\code{lambda}}{Regularization parameter. If \code{NULL}, tuned by cross-validation via \pkg{CVST}.}
  \item{\code{rho}}{Additional user-specified hyperparameter.}
  \item{\code{n_threads}}{Number of threads used for parallelization.}
}

Additional components depend on the value of \code{opt}:

\subsection{opt = \dQuote{exact}}{
\describe{
  \item{\code{K}}{The full kernel matrix.}
}}

\subsection{opt = \dQuote{pivoted}}{
\describe{
  \item{\code{m}}{Effective rank actually used.
  This value is at most the requested \code{m} and
  may be smaller if early stopping is triggered by \code{eps}.}
  \item{\code{K}}{Exact kernel matrix \eqn{K \in \mathbb{R}^{n \times n}}
    computed by \code{make_kernel(..., opt = "exact")}.}
  \item{\code{PR}}{The method provides a low-rank approximation to the kernel matrix
    \eqn{PR \in \mathbb{R}^{ n \times m}} obtained via
    Pivoted Cholesky decomposition; satisfies \eqn{K \approx PR\,(PR)^\top}.}
}}

\subsection{opt = \dQuote{nystrom}}{
\describe{
  \item{\code{m}}{Approximation rank used for the low-rank kernel approximation.}
  \item{\code{K}}{Exact kernel matrix \eqn{K \in \mathbb{R}^{n \times n}}
    computed by \code{make_kernel(..., opt = "exact")}.}
  \item{\code{R}}{The method provides a low-rank approximation to the kernel matrix
    \eqn{R \in \mathbb{R}^{n \times m}} obtained via
    Nyström approximation; satisfies \eqn{K \approx R R^\top}.}
}}

\subsection{opt = \dQuote{rff}}{
\describe{
  \item{\code{m}}{Number of random features.}
  \item{\code{z}}{Random Fourier Feature matrix \eqn{Z \in \mathbb{R}^{n \times m}} with
    \eqn{Z_{ij} = z_j(x_i) = \sqrt{2/m}\cos(\omega_j^\top x_i + b_j), \quad j = 1, \cdots, m,}
    so that \eqn{K \approx Z Z^\top}.}
  \item{\code{w}}{Random frequency matrix \eqn{\omega \in \mathbb{R}^{m \times d}}
      (row \eqn{j} is \eqn{\omega_j^\top \in \mathbb{R}^d}), drawn i.i.d. from the spectral density of the chosen kernel:
      \itemize{
        \item Gaussian: \eqn{\omega_{jk} \sim \mathcal{N}(0, 2\gamma)} (e.g., \eqn{\gamma=1/\ell^2}).
        \item Laplace: \eqn{\omega_{jk} \sim \mathrm{Cauchy}(0, 1/\sigma)} i.i.d.
      }}
  \item{\code{b}}{Random phase vector \eqn{b \in \mathbb{R}^m}, i.i.d. \eqn{\mathrm{Unif}[0,\,2\pi]}.}
}}
}
\description{
This function performs kernel ridge regression (KRR) in high-dimensional
settings. The regularization parameter \eqn{\lambda} can be selected via the
CVST (Cross-Validation via Sequential Testing) procedure. For scalability,
three different kernel approximation strategies are supported (Nyström approximation,
Pivoted Cholesky decomposition, Random Fourier Features(RFF)), and kernel matrix
can be computed using two methods(Gaussian kernel, Laplace kerenl).
}
\details{
The function performs several input checks and automatic adjustments:

\itemize{
  \item If \code{x} is a vector, it is converted to a one column matrix.
    Otherwise, \code{x} must be a matrix; otherwise an error is thrown.
  \item \code{y} must be a vector, and its length must match \code{nrow(x)}.
  \item \code{kernel} must be either \code{gaussian} or \code{laplace}.
  \item \code{opt} must be one of \code{"exact"}, \code{"pivoted"},
    \code{"nystrom"}, or \code{"rff"}.
  \item If \code{m} is \code{NULL}, it defaults to
    \deqn{\lceil n \cdot \log(d + 5) / 10 \rceil}
    where \eqn{n = nrow(X)} and \eqn{d = ncol(X)}.
    Otherwise, \code{m} must be a positive integer.
  \item \code{rho} must be a positive real number (default is 1).
  \item \code{lambda} can be specified in three ways:
    \enumerate{
      \item A positive numeric scalar, in which case the model is fitted with
         this single value.
      \item A numeric vector (length >= 3) of positive values used as a tuning grid;
         selection is performed by \pkg{CVST} cross-validation (sequential testing if
         \code{fastcv = TRUE}).
      \item \code{NULL}: use a default grid (internal setting) and tune \code{lambda}
        via \pkg{CVST} cross-validation (sequential testing if \code{fastcv = TRUE}).}
  \item \code{n_threads}: Number of threads for parallel computation.
    Default is \code{4}. If the system has <= 3 available processors,
    it uses \code{1}.
}
}
\examples{
set.seed(1)
lambda = 1e-4
d = 1
rho = 1
n = 50
X = matrix(runif(n*d, 0, 1), nrow = n, ncol = d)
y = as.vector(sin(2*pi*rowMeans(X)^3) + rnorm(n, 0, 0.1))

# Exapmle: pivoted cholesky
model = fastkrr(X, y, kernel = "gaussian", opt = "pivoted", rho = rho, lambda = 1e-4)

# Example: nystrom
model = fastkrr(X, y, kernel = "gaussian", opt = "nystrom", rho = rho, lambda = 1e-4)

# Example: random fourier features
model = fastkrr(X, y, kernel = "gaussian", opt = "rff", rho = rho, lambda = 1e-4)

# Example: Laplace kernel
model = fastkrr(X, y, kernel = "laplace", opt = "nystrom", n_threads = 1, rho = rho)

}
