% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ships_methods.R
\name{make.stationary.ships}
\alias{make.stationary.ships}
\title{Identify and parameterize stationary active ships in a DeponsShips object}
\usage{
make.stationary.ships(
  x,
  action = "check",
  candidates = NULL,
  distcrit = "shore",
  landscape = NULL,
  structure_locations = NULL,
  start_day = NA,
  start_times = NULL,
  verbose = F
)
}
\arguments{
\item{x}{DeponsShips object}

\item{action}{Character. If "check" (default), returns a data frame of pause positions that are candidates for stationary activity based on the selected criteria. If "replace" and a candidates data frame is provided, returns a DeponsShip object where the pauses identified in the data frame have been converted to stationary active status (i.e., a non-zero speed has been assigned)}

\item{candidates}{A data frame of pause positions that are candidates for stationary activity. Required if 'action = "replace"'. Generated by using 'action = "check"'}

\item{distcrit}{Character. Main criterion for finding candidates for stationary activity. If "shore" (default), all ship positions in open water are eligible, subject to a number of secondary criteria (see Details). In this case, a DeponsRaster must be provided that allows determination of distance from land (see below). If any other value or NA, only ship positions close to specified structure locations (such as turbine piles) are eligible, and these locations must be provided via 'structure_locations' (see below). In this case, a start day for the ship records and individual start times for the structure locations may also be provided to allow simulation of an ongoing construction process (see below)}

\item{landscape}{A DeponsRaster where land areas are indicated as NA (e.g., the prey map for the simulation). Required if 'distcrit = "shore"' to determine distance of candidate positions from land}

\item{structure_locations}{A data frame with columns "id", "x" (numerical) and "y" (numerical), and one row for each structure that is to be used as a proximity criterion for finding candidates. Required if distcrit != 'shore'}

\item{start_day}{A character string or POSIX object of the form 'YYYY-MM-DD HH:MM:SS'. Defines the start time of x. Optional; can be provided together with start_times if distcrit != 'shore', to allow checking whether structures under construction are present at a given time point}

\item{start_times}{A data frame with columns "time" (character string or POSIX of format 'YYYY-MM-DD HH:MM:SS') and "id", and one row for each structure that is to be used as a proximity criterion for finding candidates. Defines time from which onward the structure is present. Optional; can be provided together with start_day if distcrit != 'shore', to allow checking whether structures under construction are present at a given time point}

\item{verbose}{Logical (default False). If True, writes a summary of each candidate to the console during "check" runs}
}
\value{
If 'action = "check"' (default), returns a data frame with columns "route_number", "ship_name", "ship_type", "route_pos" (position number along route), and "pauses" (number of pauses at this position), with one row for each position that is a candidate for stationary activity based on the selected criteria. If "replace" and a candidates data frame is provided, returns a DeponsShip object where the pauses identified in the data frame have been converted to stationary active status (i.e., a non-zero speed has been assigned).
}
\description{
Identifies ship positions in a DeponsShips object where the ship is stationary
(pausing) but potentially still actively using its engine (bollard pushing or
using dynamic positioning system), and if desired assigns a suitable non-zero
speed to ensure noise generation at that time point. Candidates may be found
either among all ships that are at a minimum distance from shore, or among
those that are close to specific structures of interest, such as wind turbines.
}
\details{
When a DeponsShips object is created using [ais.to.DeponsShips()], positions
are interpolated at 30-minute intervals (ticks). If a ship's position does not
change during sequential ticks, these ticks are combined into a pause of the
appropriate duration, with a movement speed of 0. However, in some cases, an
unmoving ship is actually using its engine to hold position, such as a crew
transfer vessel performing a bollard push against a turbine pile, or an
offshore supply vessel using a dynamic position system (DPS). Under these
circumstances, the ship should emit noise to affect porpoise agents.
This function attempts to identify and rewrite such pausing instances in an
existing DeponsShips object. A pause is converted into an active stationary
position by assigning a non-zero speed and thus noise emission. Note that
assigning a speed does not translate into movement, as movement in the model
is only derived from position changes, and speed is only used to drive noise
calculation.

The intended functionality is to first run the function using 'action = "check"'
to return a table of candidate instances. After this has been inspected and
thinned as desired by the user, the function is run again using 'action = "replace"'
while providing the table as 'candidates', which returns a DeponsShips object
where the identified candidate pauses have been replaced with speed values.
No testing criteria (distcrit, landscape, stucture_locations, start_day, start_times)
are required for a "replace" run, as the instances provided as 'candidates'
are then modified without further checks.

Only ships with type "Other" or "Government/Research" (following the key
in Table 1 in MacGillivray & de Jong 2021) are tested, as these categories
contain the survey, construction and crew transfer ships that are the primary
candidate types. Passenger, recreational, fishing and cargo vessels are assumed
to not or rarely use DPS and are omitted. However, the "Other" category also
contains vessels that hold position for extended periods without using DPS,
such as jack-up rigs and platforms; also, ship type codes provided in AIS data
are frequently unreliable. We therefore strongly suggest that the user should
carefully scrutinize the candidates table produced in a "check" run, look up
vessels by their MMSI code, and remove any false positives from the table before
processing it in a "replace" run.

The inserted speed values are 7.4 knots for "Other" and 8 knots for "Government/Research", based on the class reference speeds in MacGillivray & de Jong (2021).

When 'distcrit = "shore"', pause instances are additionally tested against the following criteria: 1) not in a cell (400x400 m) directly adjacent to land, to exclude berthed ships; 2) not in a cell at the map boundary, as [ais.to.DeponsShips()] will create inactive (pausing) placeholder positions at the point of entry if a ship enters the map with a delay after the object's start, or at the point of exit if it leaves before the end of the object's duration; 3) not in the first or last position of the ship's track (same reason).

When candidates are identified based on proximity to a list of structures, a maximum distance of 97.72 m is allowed, based on an estimate of mean AIS positioning error (Jankowski et al. 2021).
}
\section{References}{

MacGillivray, A., & de Jong, C (2021). A reference spectrum model for estimating source levels of marine shipping based on Automated Identification System data. Journal of Marince Science and Engineering, 9(4), 369. doi:10.3390/jmse9040369"

Jankowski, D, Lamm A, & Hahn, A (2021). Determination of AIS position accuracy and evaluation of reconstruction methods for maritime observation data. IFAC-PapersOnLine, 54(16), 97-104. doi:10.1016/j.ifacol.2021.10.079
}

\examples{
\dontrun{
data(shipdata)
data(bathymetry)
candidates <- make.stationary.ships(shipdata,
                                   landscape = bathymetry,
                                   verbose = T)
shipdata.updated <- make.stationary.ships(shipdata,
                                         action = "replace",
                                         candidates = candidates,
                                         landscape = bathymetry)}

}
\seealso{
\code{\link{ais.to.DeponsShips}} for creation of DeponsShips objects (including calculated speeds) from AIS data
}
