% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/balance.indices.R
\name{balance.indices}
\alias{balance.indices}
\title{Computes Colles-like, Sackin and cophenetic balance indices of a phylogenetic tree.}
\usage{
balance.indices(tree, norm = FALSE)
}
\arguments{
\item{tree}{a single phylogenetic tree. It can be entered as a string in Newick format, as a 'phylo' object (\code{ape} package) or as an 'igraph' object (\code{igraph} package).}

\item{norm}{a logical variable that indicates whether the indices should be normalized or not.}
}
\value{
A numeric vector with the three computed balance indices of the tree: 
\code{Colless-like}, \code{Sackin} and \code{Cophenetic} values.
}
\description{
Given a phylogenetic tree, computes Colles-like, Sackin and cophenetic balance indices of that tree.
}
\details{
The Colless-like index is the generalization of the Colless' index for non-binary trees (see Mir et al. , 2017).

The Sackin's index is computed as the sum of the number of ancestors for each leave of the tree (see Mir et al. , 2013).

The cophenetic index is computed as the sum of the depths of the least common ancestor (LCA) of every pair of leaves of the tree(see Sackin et al, 1972).
}
\examples{
# Computation of the Colless-Like, Sackin and Cophenetic 
# balance indices of trees entered in newick format:
balance.indices("(1,2,3,4,5);")
balance.indices("(1,(2,(3,(4,5))));")

# Computation of the Colless-Like, Sackin and Cophenetic
# balance indices of a tree entered as a phylo object:
require(ape)
random.tree = rtree(5,rooted=TRUE)
balance.indices(random.tree)

# Computation of the Colless-Like, Sackin and Cophenetic 
# balance indices of a tree entered as a igraph object.
# The tree is randomly generated from all trees with 5
# leaves following the alpha-gamma model with alpha=0.5
# and gamma=0.3.
a.g.tree = a.g.model(5,0.5,0.3)
balance.indices(a.g.tree)

# All of them can be normalized (values between 0 and 1)
balance.indices("(1,2,3,4,5);",norm=TRUE)
balance.indices("(1,(2,(3,(4,5))));",norm=TRUE)
balance.indices(random.tree,norm=TRUE)
balance.indices(a.g.tree,norm=TRUE)

}
\references{
A. Mir, F. Rossello, L.Rotger, A Colless-like balance index for multifurcating phylogenetic trees.\emph{}

A. Mir, F. Rossello, L.Rotger, A new balance index for phylogenetic trees. \emph{Mathematical Biosciences} \bold{241} (2013), 125-136.

M. J. Sackin, "Good" and  "bad" phenograms. \emph{Sys. Zool}, \bold{21} (1972), 225-226.
}
\author{
Lucia Rotger
}
