% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/main_func.R
\name{power.glm.random.a0}
\alias{power.glm.random.a0}
\title{Power/type I error calculation for generalized linear models with random a0}
\usage{
power.glm.random.a0(
  data.type,
  data.link,
  data.size,
  n = 1,
  historical,
  samp.prior.beta,
  samp.prior.var,
  prior.a0.shape1 = 1,
  prior.a0.shape2 = 1,
  a0.coefficients,
  lower.limits = rep(-100, 50),
  upper.limits = rep(100, 50),
  slice.widths = rep(0.1, 50),
  delta = 0,
  gamma = 0.95,
  nMC = 10000,
  nBI = 250,
  N = 10000
)
}
\arguments{
\item{data.type}{Character string specifying the type of response. The options are "Normal", "Bernoulli", "Binomial", "Poisson" and "Exponential".}

\item{data.link}{Character string specifying the link function. The options are "Logistic", "Probit", "Log", "Identity-Positive", "Identity-Probability" and "Complementary Log-Log". Does not apply if \code{data.type} is "Normal".}

\item{data.size}{Sample size of the simulated datasets.}

\item{n}{(For binomial data only) vector of integers specifying the number of subjects who have a particular value of the covariate vector. If the data is binary and all covariates are discrete, collapsing Bernoulli data into a binomial structure can make the slice sampler much faster.}

\item{historical}{List of historical dataset(s). East historical dataset is stored in a list which contains two \emph{named} elements: \code{y0} and \code{x0}.
\itemize{
\item \code{y0} is a vector of responses.
\item \code{x0} is a matrix of covariates. \code{x0} should NOT have the treatment indicator. Apart from missing the treatent/control indicator, \code{x0} should have the same set of covariates in the same order as \code{x}.
}
For binomial data, an additional element \code{n0} is required.
\itemize{
\item \code{n0} is vector of integers specifying the number of subjects who have a particular value of the covariate vector.
}}

\item{samp.prior.beta}{Matrix of possible values of \eqn{\beta} to sample (with replacement) from. Each row is a possible \eqn{\beta} vector (a realization from the sampling prior for \eqn{\beta}), where the first element is the coefficient for the intercept and the second element is the coefficient for the treatment indicator.
The length of the vector should be equal to the total number of parameters, i.e. P+2 where P is the number of columns of \code{x0} in \code{historical}.}

\item{samp.prior.var}{Vector of possible values of \eqn{\sigma^2} to sample (with replacement) from. Only applies if \code{data.type} is "Normal". The vector contains realizations from the sampling prior (e.g. inverse-gamma distribution) for \eqn{\sigma^2}.}

\item{prior.a0.shape1}{First shape parameter of the beta prior for \eqn{a_0}. The default is 1.}

\item{prior.a0.shape2}{Second shape parameter of the beta prior for \eqn{a_0}. The default is 1.}

\item{a0.coefficients}{Vector of coefficients for \eqn{a_0} returned by the function \code{\link{normalizing.constant}}. This is necessary for estimating the normalizing constant for the normalized power prior. Does not apply if \code{data.type} is "Normal".}

\item{lower.limits}{Vector of lower limits for parameters to be used by the slice sampler. If \code{data.type} is "Normal", slice sampling is used for \eqn{a_0}, and the length of the vector should be equal to the number of historical datasets.
For all other data types, slice sampling is used for \eqn{\beta} and \eqn{a_0}. The first P+1 elements apply to the sampling of \eqn{\beta} and the rest apply to the sampling of \eqn{a_0}.
The length of the vector should be equal to the sum of the total number of parameters (i.e. P+1 where P is the number of covariates) and the number of historical datasets.
The default is -100 for all parameters (may not be appropriate for all situations).}

\item{upper.limits}{Vector of upper limits for parameters to be used by the slice sampler. If \code{data.type} is "Normal", slice sampling is used for \eqn{a_0}, and the length of the vector should be equal to the number of historical datasets.
For all other data types, slice sampling is used for \eqn{\beta} and \eqn{a_0}. The first P+1 elements apply to the sampling of \eqn{\beta} and the rest apply to the sampling of \eqn{a_0}.
The length of the vector should be equal to the sum of the total number of parameters (i.e. P+1 where P is the number of covariates) and the number of historical datasets.
The default is 100 for all parameters (may not be appropriate for all situations).}

\item{slice.widths}{Vector of initial slice widths used by the slice sampler. If \code{data.type} is "Normal", slice sampling is used for \eqn{a_0}, and the length of the vector should be equal to the number of historical datasets.
For all other data types, slice sampling is used for \eqn{\beta} and \eqn{a_0}. The first P+1 elements apply to the sampling of \eqn{\beta} and the rest apply to the sampling of \eqn{a_0}.
The length of the vector should be equal to the sum of the total number of parameters (i.e. P+1 where P is the number of covariates) and the number of historical datasets.
The default is 0.1 for all parameter (may not be appropriate for all situations).}

\item{delta}{Prespecified constant that defines the boundary of the null hypothesis. The default is zero.}

\item{gamma}{Posterior probability threshold for rejecting the null. The null hypothesis is rejected if posterior probability is greater \code{gamma}. The default is 0.95.}

\item{nMC}{Number of iterations (excluding burn-in samples) for the slice sampler or Gibbs sampler. The default is 10,000.}

\item{nBI}{Number of burn-in samples for the slice sampler or Gibbs sampler. The default is 250.}

\item{N}{Number of simulated datasets to generate. The default is 10,000.}
}
\value{
Power or type I error is returned, depending on the sampling prior used. If \code{data.type} is "Normal", average posterior means of \eqn{\beta}, \eqn{\tau} and \eqn{a_0} are also returned.
For all other data types, average posterior means of \eqn{\beta} and \eqn{a_0} are also returned.
}
\description{
Power/type I error calculation using normalized power priors for generalized linear models with random \eqn{a_0}
}
\details{
The user should use the function \code{\link{normalizing.constant}} to obtain \code{a0.coefficients} (does not apply if \code{data.type} is "Normal").

\code{samp.prior.beta} can be generated using the sampling priors (see example).
\code{samp.prior.var} is necessary for generating normally distributed data.

If \code{data.type} is "Normal", the response \eqn{y_i} is assumed to follow \eqn{N(x_i'\beta, \tau^{-1})} where \eqn{x_i} is the vector of covariates for subject \eqn{i}.
Historical datasets are assumed to have the same precision parameter as the current dataset for computational simplicity.
The initial prior for \eqn{\tau} is the Jeffery's prior, \eqn{\tau^{-1}}.
The initial prior for \eqn{\beta} is the uniform improper prior. Posterior samples for \eqn{\beta} and \eqn{\tau} are obtained through Gibbs sampling.
Posterior samples for \eqn{a_0} are obtained through slice sampling.

For all other data types, posterior samples are obtained through slice sampling. The initial prior for \eqn{\beta} is the uniform improper prior.
The default lower limits for the parameters are -100. The default upper limits
for the parameters are 100. The default slice widths for the parameters are 0.1.
The defaults may not be appropriate for all situations, and the user can specify the appropriate limits
and slice width for each parameter.

If a sampling prior with support in the null space is used, the value returned is a Bayesian type I error rate.
If a sampling prior with support in the alternative space is used, the value returned is a Bayesian power.
}
\examples{

data.type <- "Bernoulli"
data.link <- "Logistic"
data.size <- 100

# Simulate two historical datasets
p <- 3
historical <- list(list(y0=rbinom(data.size,size=1,prob=0.2),
                        x0=matrix(rnorm(p*data.size),ncol=p,nrow=data.size)),
                   list(y0=rbinom(data.size, size=1, prob=0.5),
                        x0=matrix(rnorm(p*data.size),ncol=p,nrow=data.size)))

# Generate sampling priors

# The null hypothesis here is H0: beta_1 >= 0. To calculate power,
# we can provide samples of beta_1 such that the mass of beta_1 < 0.
# To calculate type I error, we can provide samples of beta_1 such that
# the mass of beta_1 >= 0.
samp.prior.beta1 <- rnorm(100, mean=-3, sd=1)
# Here, mass is put on the alternative region, so power is calculated.
samp.prior.beta <- cbind(rnorm(100), samp.prior.beta1, matrix(rnorm(100*p), 100, p))

# Please see function "normalizing.constant" for how to obtain a0.coefficients
# Here, suppose one-degree polynomial regression is chosen by the "normalizing.constant" 
# function. The coefficients are obtained for the intercept, a0_1 and a0_2. 
a0.coefficients <- c(1, 0.5, -1)

nMC <- 100 # nMC should be larger in practice
nBI <- 50
N <- 3 # N should be larger in practice
result <- power.glm.random.a0(data.type=data.type, data.link=data.link,
                              data.size=data.size, historical=historical,
                              samp.prior.beta=samp.prior.beta, a0.coefficients=a0.coefficients,
                              delta=0, nMC=nMC, nBI=nBI, N=N)

}
\references{
Chen, Ming-Hui, et al. "Bayesian design of noninferiority trials for medical devices using historical data." Biometrics 67.3 (2011): 1163-1170.

Neal, Radford M. Slice sampling. Ann. Statist. 31 (2003), no. 3, 705--767.
}
\seealso{
\code{\link{normalizing.constant}} and \code{\link{glm.random.a0}}
}
